﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <climits>
#include <algorithm>
#include "nim_ShopServiceAccessPrivate.h"

#include <nn/account/account_ApiForSystemServices.h>
#include <nn/dauth/dauth_Api.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/nsd/nsd_ApiForNasService.h>

#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/time/time_StandardNetworkSystemClock.h>
#include <nn/time/time_TimeZoneApi.h>

namespace {

// デバッグコード
#if !defined(NN_SDK_BUILD_RELEASE) && defined(ENABLE_DEBUG_TRACE)
#define DEBUG_TRACE(...) NN_DETAIL_NIM_TRACE( "[ShopServiceAccess::Private] " __VA_ARGS__ )
#else
#define DEBUG_TRACE(...)    static_cast<void>(0)
#endif

}

namespace nn { namespace nim { namespace srv {

//-----------------------------------------------------------------------------
namespace {

const char g_PresetDigitalValues[] = "0123456789abcdef";

NN_STATIC_ASSERT(sizeof(g_PresetDigitalValues) > 16);

}   // ~unnamed

//-----------------------------------------------------------------------------
char* ShopServiceAccess::ToHexStringFromApplicationId(char* pOut, const char* pOutEnd, const ::nn::ncm::ApplicationId& id) NN_NOEXCEPT
{
    Bit64 digit = id.value;
    const int maxBitLength = sizeof(::nn::ncm::ApplicationId) * 8;
    for (int shift = maxBitLength - 4; pOut < pOutEnd && shift >= 0; shift -= 4)
    {
        *pOut++ = g_PresetDigitalValues[(digit >> shift) & 0x0f];
    }
    return pOut;
}

//-----------------------------------------------------------------------------
char* ShopServiceAccess::ToDecimalStringFrom(char* pOut, const char* pOutEnd, size_t value) NN_NOEXCEPT
{
    auto pReverse = pOut;
    do
    {
        *pReverse++ = g_PresetDigitalValues[value % 10];
    } while ((value /= 10) > 0 && pReverse < pOutEnd);

    auto pRear = pReverse;
    *pRear = '\0';
    while (pOut < (--pReverse))
    {
        const auto c = *pOut;
        *pOut++ = *pReverse;
        *pReverse = c;
    }
    return pRear;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccess::GetCachedUserProperty(::nn::account::CachedNintendoAccountInfoForSystemService* pOut, const ::nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);

    ::nn::account::NetworkServiceAccountManager nsaManager;
    NN_RESULT_DO(::nn::account::GetNetworkServiceAccountManager(&nsaManager, uid));

    NN_ALIGNAS(sizeof(uintptr_t)) Bit8 buffer[::nn::account::RequiredBufferSizeForCachedNintendoAccountInfo];
    NN_RESULT_DO(nsaManager.LoadCachedNintendoAccountInfo(pOut, buffer, sizeof(buffer)));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccess::GetUserCountryCode(size_t* pOutBytes, char* pOut, const char* pOutEnd, const ::nn::account::CachedNintendoAccountInfoForSystemService& prop) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutBytes);
    NN_SDK_ASSERT_NOT_NULL(pOutEnd);
    NN_SDK_ASSERT_NOT_NULL(pOut);

    // NOTE: pOutBytes には null 終端を含むバイト数が返される。
    const auto pValue = prop.GetCountry(pOutBytes);
    const auto valueLength = *pOutBytes;
    *pOutBytes = 0;

    const auto length = static_cast<size_t>((valueLength > 0) ? (valueLength - 1) : valueLength);
    NN_RESULT_THROW_UNLESS(static_cast<size_t>(pOutEnd - pOut) >= length, ResultBufferNotEnough());

    std::memcpy(pOut, pValue, length);
    *pOutBytes = length;
    NN_RESULT_SUCCESS;
}

#if defined(ENABLE_ISO639_LANGUAGE_CODE_EXTRACT_FROM_NA_REGISTORY)
//-----------------------------------------------------------------------------
Result ShopServiceAccess::GetUserLanguageCode(size_t* pOutBytes, char* pOut, const char* pOutEnd, const ::nn::account::CachedNintendoAccountInfoForSystemService& prop) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutBytes);
    NN_SDK_ASSERT_NOT_NULL(pOutEnd);
    NN_SDK_ASSERT_NOT_NULL(pOut);

    // NOTE: pOutBytes には null 終端を含むバイト数が返される。
    const auto pValue = prop.GetLanguage(pOutBytes);
    const auto valueLength = *pOutBytes;
    *pOutBytes = 0;

    // ITEF 言語コードの language 下位タグフィールドを抽出 ( ISO 639-X ) 準拠。
    //  ISO 639-1 以外の言語コード( 3文字以上表記 )でも、そのまま採用します。
    const auto notNullLength = (valueLength > 0) ? (valueLength - 1) : valueLength;
    const auto pEnd = &pValue[notNullLength];
    const auto pFind = std::find(pValue, pEnd, '-');
    const auto length = (pEnd != pFind) ? static_cast<size_t>(pFind - pValue) : notNullLength;
    NN_RESULT_THROW_UNLESS(static_cast<size_t>(pOutEnd - pOut) >= length, ResultBufferNotEnough());

    std::memcpy(pOut, pValue, length);
    *pOutBytes = length;
    NN_RESULT_SUCCESS;
}
#endif  // #if defined(ENABLE_ISO639_LANGUAGE_CODE_EXTRACT_FROM_NA_REGISTORY)

//-----------------------------------------------------------------------------
Result ShopServiceAccess::VerifyQuebecAgeRestriction(const ::nn::account::Uid& uid) NN_NOEXCEPT
{
    // ケベック13歳未満制限チェック。

    // NA キャッシュプロパティ取得。
    ::nn::account::CachedNintendoAccountInfoForSystemService property;
    NN_RESULT_DO(GetCachedUserProperty(&property, uid));

    // 地域の取得
    size_t regionLength;
    const auto region = property.GetRegion(&regionLength);
    // ケベック州以外の場合正常終了
    NN_RESULT_THROW_UNLESS(0 == std::strcmp(region, "QC"), ResultSuccess());

    // ネットワーク時計の取得。
    ::nn::time::PosixTime currentNetworkTime;
    NN_RESULT_TRY(::nn::time::StandardNetworkSystemClock::GetCurrentTime(&currentNetworkTime))
    NN_RESULT_CATCH_CONVERT(::nn::time::ResultClockInvalid, ResultNetworkTimeUnavailable());
    NN_RESULT_END_TRY;

    // UTC+12 で計算する。
    const auto todayUtc12 = currentNetworkTime + TimeSpan::FromSeconds(12 * 60 * 60);
    // ネットワーク時計をカレンダー形式に変換
    auto calendarTime = nn::time::ToCalendarTimeInUtc(todayUtc12);

    const int todayValue = calendarTime.year * 10000 + calendarTime.month * 100 + calendarTime.day;

    // 誕生日の取得。例[1970-01-01]
    size_t birthdayLength;
    int numericBirthday = 0;
    const char* birthday = property.GetBirthday(&birthdayLength);
    for (size_t i = 0; i < birthdayLength; i++)
    {
        // 文字を数値に変換 [1970-01-01]⇒[19700101]
        const int oneLetter = birthday[i] - '0';
        if (oneLetter >= 0 && oneLetter < 10)
        {
            numericBirthday = numericBirthday * 10 + oneLetter;
        }
    }

    // 年齢の計算。
    int age = (todayValue - numericBirthday) / 10000;
    NN_RESULT_THROW_UNLESS(age >= 13, ResultAgeRestriction());

    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
ShopServiceAccess::HeapAllocator::HeapAllocator() NN_NOEXCEPT
    : m_Handle()
{
}

//-----------------------------------------------------------------------------
Result ShopServiceAccess::HeapAllocator::Initialize(void* pTop, size_t size) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(nullptr != pTop && size >= MinimumRequiredCapacity, ResultBufferNotEnough());

    Finalize();
    m_Handle = ::nn::lmem::CreateExpHeap(pTop, size, ::nn::lmem::CreationOption_ThreadSafe);
    NN_RESULT_THROW_UNLESS(m_Handle, ResultBufferNotEnough());
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
bool ShopServiceAccess::HeapAllocator::Finalize() NN_NOEXCEPT
{
    const auto handle = m_Handle;
    if (handle)
    {
        m_Handle = ::nn::lmem::HeapHandle();
        ::nn::lmem::DestroyExpHeap(handle);
        return true;
    }
    return false;
}

//-----------------------------------------------------------------------------
void* ShopServiceAccess::HeapAllocator::Allocate(size_t size, int alignment) NN_NOEXCEPT
{
    const auto handle = m_Handle;
    return (!handle) ? nullptr : ::nn::lmem::AllocateFromExpHeap(handle, size, alignment);
}

//-----------------------------------------------------------------------------
void ShopServiceAccess::HeapAllocator::Deallocate(void* pBlock) NN_NOEXCEPT
{
    const auto handle = m_Handle;
    if (handle)
    {
        ::nn::lmem::FreeToExpHeap(handle, pBlock);
    }
}

//-----------------------------------------------------------------------------
Result ShopServiceAccess::TransferStorage::Initialize(const ::nn::sf::NativeHandle& handle, uint64_t size) NN_NOEXCEPT
{
    Finalize();

    void* pOutAddress;
    const size_t actualSize = static_cast<size_t>(size);
    ::nn::os::AttachTransferMemory(&m_TransferHandle, actualSize, handle.GetOsHandle(), handle.IsManaged());
    NN_RESULT_TRY(::nn::os::MapTransferMemory(&pOutAddress, &m_TransferHandle, ::nn::os::MemoryPermission_None))
    NN_RESULT_CATCH_ALL
    {
        ::nn::os::DestroyTransferMemory(&m_TransferHandle);
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY;

    NN_RESULT_TRY(m_BaseAllocator.Initialize(pOutAddress, actualSize))
    NN_RESULT_CATCH_ALL
    {
        ::nn::os::UnmapTransferMemory(&m_TransferHandle);
        ::nn::os::DestroyTransferMemory(&m_TransferHandle);
        NN_RESULT_THROW(ResultShopServiceAccessInsufficientWorkMemory());
    }
    NN_RESULT_END_TRY;

    m_pMappedAddress = pOutAddress;
    m_MappedSize = actualSize;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
void ShopServiceAccess::TransferStorage::CleanMappedMemory() NN_NOEXCEPT
{
    const auto size = m_MappedSize;
    const auto pOutAddress = m_pMappedAddress;
    NN_SDK_ASSERT(nullptr != pOutAddress && size > 0);

    // ビルトインならアライン考慮した最適化コードになってる想定。
    // でなければ、ちゃんとアライメントで分岐して最適化 Write にすべき。
    std::memset(pOutAddress, 0, size);
    m_pMappedAddress = nullptr;
    m_MappedSize = 0;
}

//-----------------------------------------------------------------------------
void ShopServiceAccess::TransferStorage::Finalize() NN_NOEXCEPT
{
    if (m_BaseAllocator.Finalize())
    {
        CleanMappedMemory();
        ::nn::os::UnmapTransferMemory(&m_TransferHandle);
        ::nn::os::DestroyTransferMemory(&m_TransferHandle);
    }
}

//-----------------------------------------------------------------------------
ShopServiceAccess::RequestCountRestriction::RequestCountRestriction(int restrictCount, const ::nn::TimeSpan& restrictPeriod) NN_NOEXCEPT
    : m_Timer(::nn::os::EventClearMode_ManualClear)
    , m_Cancel(::nn::os::EventClearMode_ManualClear)
    , m_Period(restrictPeriod)
    , m_Limit(restrictCount - 1)
    , m_Counter(restrictCount)
{
    NN_SDK_REQUIRES(restrictCount > 0);
}

//-----------------------------------------------------------------------------
ShopServiceAccess::RequestCountRestriction::~RequestCountRestriction() NN_NOEXCEPT
{
    m_Cancel.Signal();
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        m_Timer.Clear();
        m_Timer.Stop();
        m_Cancel.Clear();
    }
}

//-----------------------------------------------------------------------------
bool ShopServiceAccess::RequestCountRestriction::AcquirePermission(Canceler* pCanceler) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCanceler);

    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    auto count = m_Counter - 1;
    if (count < 0)
    {
        // 資源なし状態で要求されたので、シグナルくるまでストール
        m_Cancel.Clear();
        pCanceler->store(&m_Cancel, std::memory_order_release);
        const auto index = ::nn::os::WaitAny(m_Timer.GetBase(), m_Cancel.GetBase());
        pCanceler->Invalidate();
        m_Cancel.Clear();
        switch (index)
        {
        case 0:
            m_Timer.Clear();
            m_Timer.Stop();
            count = m_Limit;
            break;
        case 1:
            return false;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    m_Counter = count;
    if (count == m_Limit)
    {
        m_Timer.StartOneShot(m_Period);
    }
    return true;
}

}}}
