﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/sf/sf_Types.h>
#include <nn/nim/nim_ShopServiceAccessTypes.h>

namespace nn { namespace nim { namespace srv {

namespace ShopServiceAccessDebug
{
    /**
     * @brief       デバッグ支援機能の利用可否状態の更新要求を実施します。
     */
    void RefreshAvailability() NN_NOEXCEPT;

    /**
     * @brief       デバッグ支援機能の利用可否を取得します。
     *
     * @return      処理結果が返されます。
     * @retval      nn::ResultSuccess           利用可能です。
     * @retval      nn::nim::ResultNotSupported 機能利用を拒否されました。
     */
    Result IsAvailable() NN_NOEXCEPT;

    /**
     * @brief       デバッグレスポンス支援機能インタフェース提供クラス。
     */
    namespace Response
    {
        //! @brief  柔軟な権利管理モジュール用。
        constexpr ::nn::nim::ShopServiceAccessTypes::Server DragonsServer = 0xffu;

        /**
         * @brief       デバッグレスポンス照合結果受信コンテナ。
         */
        class EmulationResult
        {
        private:
            Result  m_Result;       //!< 検出されたデバッグ結果値。( m_IsApplied == true 時のみ有効 )
            bool    m_IsApplied;    //!< デバッグレスポンスが適用されたかどうか。

        public:
            /**
             * @brief       デフォルトコンストラクタ。
             */
            EmulationResult() NN_NOEXCEPT : m_Result(), m_IsApplied(false) {}

            /**
             * @brief       評価値更新。
             */
            NN_FORCEINLINE void Update(const Result& result = ::nn::ResultSuccess(), bool applied = false) NN_NOEXCEPT
            {
                m_Result = result;
                m_IsApplied = applied;
            }

            /**
             * @brief       デバッグレスポンス適用状態の取得。
             *
             * @retval  true    デバッグレスポンスが適用されました。
             * @retval  false   デバッグレスポンスが適用されませんでした。
             */
            NN_FORCEINLINE bool IsApplied() const NN_NOEXCEPT
            {
                return m_IsApplied;
            }

            /**
             * @brief       適用されたデバッグレスポンスリザルトの取得。
             *
             * @details     @ref IsApplied() が true を返却した時のみ有効です。
             */
            NN_FORCEINLINE const Result& GetResult() const NN_NOEXCEPT
            {
                return m_Result;
            }

            /**
             * @brief       模擬対象として採用されたデバッグレスポンスの転送先メモリ領域生成コールバック。
             *
             * @param[in]   responseSize    模擬対象として採用されたデバッグレスポンスデータの総サイズ( バイト単位 )。
             *
             * @return      アロケートされたメモリ領域先頭アドレス。@n
             *              nullptr が返された場合、本コールバックを利用する API が @ref nn::nim::ResultShopServiceAccessInsufficientWorkMemory を返します。
             *
             * @details     デバッグシステム側で模擬対象デバッグレスポンスが検出された場合、本コールバックを呼び出してレスポンス転送を行います。@n
             *              nullptr 以外のアドレスが返された場合、対象アドレスに対して @a responseSize 分のデータ転送を行います。
             */
            virtual void* AllocateResponseStore(size_t responseSize) NN_NOEXCEPT = 0;

            /**
             * @brief       URL文字列に対するリプレース処理の実施。
             *
             * @param[out]  pOut        リプレース後URL文字列の出力先バッファ。
             * @param[in]   outCapacity リプレース後URL文字列の出力先バッファ容量。( バイト単位, null終端を含めたバッファ長を設定します )
             * @param[in]   pSource     リプレース前URL文字列( null 終端必要 )。
             */
            virtual Result ExecuteAliasReplaceForUrl(char* const pOut, const size_t outCapacity, const char* pSource) NN_NOEXCEPT = 0;
        };

        /**
         * @brief       デバッグレスポンス全消去。
         *
         * @return      処理結果が返されます。
         * @retval      nn::ResultSuccess                       正常に処理が終了しました。
         * @retval      nn::nim::ResultNotSupported             機能利用を拒否されました。
         *
         * @details     登録中の全デバッグレスポンスを消去します。@n
         *              本関数は、@ref ShopServiceAccessDebug::Response クラス内メソッドに対してスレッドセーフです。
         */
        Result Clear() NN_NOEXCEPT;

        /**
         * @brief       デバッグレスポンス登録。
         *
         * @param[in]   keyTarget       照合対象キー[接続先ショップサービスサーバー種別]。
         * @param[in]   keyPath         照合対象キー[URLパスキー文字列]。@n
         *                              文字列長には null 終端文字は含みません。
         * @param[in]   valueResponse   照合適用時採用レスポンスボディデータ。
         * @param[in]   expectResult    照合適用時採用Result値( InnerValue 値を指定してください )。
         * @param[in]   happenedRate    デバッグレスポンス適用事象発生率( 10000分率, 10000 で 100% 発生します。 )。
         *
         * @return      処理結果が返されます。
         * @retval      nn::ResultSuccess                       正常に処理が終了しました。
         * @retval      nn::nim::ResultNotSupported             機能利用を拒否されました。
         * @retval      nn::nim::ResultAllocationMemoryFailed   プロセスヒープメモリが不足したため処理に失敗しました。
         *
         * @details     指定のキー内容に対するレスポンスボディを値としたペアデータとしてデバッグレスポンスに登録します。@n
         *
         *              尚、本関数は内部でデバッグレスポンス登録のために一時的にプロセスヒープを利用します。@n
         *              本関数は、@ref ShopServiceAccessDebug::Response クラス内メソッドに対してスレッドセーフです。
         */
        Result Register(const ::nn::nim::ShopServiceAccessTypes::Server& keyTarget, const ::nn::sf::InArray<char>& keyPath, const ::nn::sf::InArray<char>& valueResponse, const uint32_t expectResult, const uint32_t happenedRate) NN_NOEXCEPT;

        /**
         * @brief       デバッグレスポンス照合。
         *
         * @param[out]  pOutResult      デバッグレスポンス照合結果受信用コンテナ。
         * @param[in]   pVerifyPath     照合対象[URLパスキー文字列]( null 終端必要 )。
         * @param[in]   verifyServer    照合対象[接続先ショップサービスサーバー種別]。
         *
         * @return      処理結果が返されます。@n
         *              照合結果ではなく要求自体の成否です。@n
         *              照合結果は @a pOutResult の isApplied を確認してください。
         *
         * @retval      nn::ResultSuccess                                       正常に処理が終了しました。
         * @retval      nn::nim::ResultNotSupported                             機能利用を拒否されました。
         * @retval      nn::nim::ResultAllocationMemoryFailed                   プロセスヒープメモリが不足したため処理に失敗しました。
         * @retval      nn::nim::ResultShopServiceAccessInsufficientWorkMemory  ワークメモリの不足により、レスポンス出力領域の確保に失敗しました。
         *
         * @pre
         *              - pOutResult != nullptr
         *              - pVerifyUrl != nullptr
         *
         * @details     指定されたキー内容がデバッグレスポンス登録のキー内容と一致するか照合します。@n
         *              一致した場合、@a pOutResult に結果を出力します。@n
         *              レスポンス出力先領域は @ref ApplyResult::AllocateResponseStore() で呼び出し元へ確保を要求します。@n
         *
         *              尚、本関数は内部で登録中デバッグレスポンス参照のために一時的にプロセスヒープを利用します。@n
         *              本関数は、@ref ShopServiceAccessDebug::Response クラス内メソッドに対してスレッドセーフです。
         */
        Result DoUseDebugResponse(EmulationResult* pOutResult, const char* pVerifyPath, const ::nn::nim::ShopServiceAccessTypes::Server& verifyServer) NN_NOEXCEPT;
    };
};

}}}
