﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <algorithm>

#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Context.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Result.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaExtendedData.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include <nn/ncm/ncm_StorageUtil.h>

#include <nn/ovln/format/ovln_DownloadMessage.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_NetworkInstallTask.h>
#include <nn/nim/detail/nim_Log.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include "nim_NetworkInstallUtil.h"
#include "nim_ResultMappingUtil.h"

namespace nn { namespace nim { namespace srv {

// 更新経路を決定するためのクラス
class RouteFinder
{
    NN_DISALLOW_COPY(RouteFinder);
    NN_DISALLOW_MOVE(RouteFinder);

private:
    struct Route
    {
        // パッチが v0 ということはないので、0 のときは未定義と扱う
        uint32_t version;
        uint32_t nextVersion;
        int      deltaIndex;

        bool operator < (const Route& rhs) const NN_NOEXCEPT
        {
            return version < rhs.version;
        }
    };

public:
    RouteFinder() NN_NOEXCEPT {}

    Result Initialize(const ncm::PatchMetaExtendedDataReader& reader, uint32_t sourceVersion, uint32_t destinationVersion) NN_NOEXCEPT
    {
        m_SourceVersion = sourceVersion;
        m_DestinationVersion = destinationVersion;

        std::memset(m_Route, 0, sizeof(m_Route));
        auto header = reader.GetHeader();

        // 更新に関係するバージョンの初期設定。更新後のバージョンは history には存在しないのであらかじめ登録する
        // また、更新後のバージョンはあらかじめ経路が見つかった扱いとしておく
        m_RouteCount = 1;
        m_Route[0] = {destinationVersion, destinationVersion};
        for (int i = 0; i < static_cast<int>(header->historyCount); ++i)
        {
            auto history = reader.GetPatchHistoryHeader(i);
            if (sourceVersion <= history->key.version && history->key.version <= destinationVersion)
            {
                m_Route[m_RouteCount].version = history->key.version;
                m_RouteCount++;
                NN_RESULT_THROW_UNLESS(m_RouteCount < MaxUpdateStep, ResultTooManyStepsToUpdate());
            }
        }
        std::sort(m_Route, m_Route + m_RouteCount);

        // 経路情報の書き込み。更新先情報が書き込まれている (0 ではない) 場合に経路を設定する
        // sourceVersion の順でソートされているのを前提とする
        // goal からの動的計画法と考える
        for (int i = static_cast<int>(header->deltaHistoryCount) - 1; i >= 0; --i)
        {
            auto delta = reader.GetPatchDeltaHistory(i);
            if (delta->sourceVersion < sourceVersion || destinationVersion < delta->destinationVersion)
            {
                continue;
            }
            auto destination = FindRoute(delta->destinationVersion);
            auto source = FindRoute(delta->sourceVersion);

            if (destination != nullptr && destination->nextVersion != 0)
            {
                if (source != nullptr && (source->nextVersion == 0 || source->nextVersion < delta->destinationVersion))
                {
                    source->nextVersion = delta->destinationVersion;
                    source->deltaIndex  = i;
                }
            }
        }
        ConfirmAndShrinkRoute();
        NN_RESULT_SUCCESS;
    }
    int64_t GetIndirectDownloadSize(const ncm::PatchMetaExtendedDataReader& reader) NN_NOEXCEPT
    {
        int64_t size = 0;
        for (int i = 0; i < m_RouteCount; ++i)
        {
            auto delta = reader.GetPatchDeltaHistory(m_Route[i].deltaIndex);
            size += delta->downloadSize;
        }
        return size;
    }

    bool HasRoute() const NN_NOEXCEPT { return m_Found; }
    int  GetRouteCount() const NN_NOEXCEPT { return m_RouteCount; }
    const Route& GetRoute(int index) const NN_NOEXCEPT { return m_Route[index]; }

private:
    Route* FindRoute(uint32_t version) NN_NOEXCEPT
    {
        Route r = {version, 0};
        auto found = std::lower_bound(m_Route, m_Route + m_RouteCount, r);
        return (found != (m_Route + m_RouteCount)) ? found : nullptr;
    }
    void ConfirmAndShrinkRoute() NN_NOEXCEPT
    {
        auto version = m_SourceVersion;
        m_Found = false;
        for(int i = 0; i < MaxUpdateStep; ++i)
        {
            auto r = FindRoute(version);
            if (r == nullptr)
            {
                break;
            }

            // 同じアドレスの可能性があるのでコピーしてから
            auto route = *r;
            m_Route[i] = route;
            if (route.version == m_DestinationVersion)
            {
                m_Found = true;
                m_RouteCount = i; // c->c という経路情報は以降必要ないものなので、i + 1 ではなく i とする
                break;
            }
            if (route.nextVersion == 0)
            {
                break;
            }
            version = route.nextVersion;
        }
    }

    Route    m_Route[MaxUpdateStep];
    int      m_RouteCount;
    uint32_t m_SourceVersion;
    uint32_t m_DestinationVersion;
    bool     m_Found;

};

namespace
{
bool WillPreferDeltaForcibly() NN_NOEXCEPT
{
    bool flag;
    auto size = settings::fwdbg::GetSettingsItemValue(&flag, sizeof(flag), "nim.install", "prefer_delta_evenif_inefficient");
    if (size != sizeof(flag))
    {
        NN_DETAIL_NIM_TRACE("[DeltaUpdate] Failed to read fwdbg of prefer_delta_evenif_inefficient\n");
        return false;
    }
    return flag;
}

int64_t GetDirectDownloadSize(const ncm::PackagedContentMetaReader& reader) NN_NOEXCEPT
{
    // 直接更新をした場合のサイズ取得
    int64_t size = 0;
    for (int i = 0; i < reader.CountContent(); ++i)
    {
        auto contentInfo = reader.GetContentInfo(i);
        if (contentInfo->info.type != ncm::ContentType::DeltaFragment)
        {
            size += contentInfo->info.GetSize();
        }
    }
    return size;
}

// サーバが実装されるまでのとりあえず
util::optional<ncm::ContentMetaKey> GetIfIncludesNewer(const ncm::ContentMetaKey& key, const ncm::ContentMetaKey list[], int count) NN_NOEXCEPT
{
    for (int i = 0; i < count; i++)
    {
        if (key.id == list[i].id && key.version < list[i].version)
        {
            return list[i];
        }
    }

    return util::nullopt;
}

bool IncludesId(const ncm::ContentMetaKey& key, const ncm::ContentMetaKey list[], int count) NN_NOEXCEPT
{
    for (int i = 0; i < count; i++)
    {
        if (key.id == list[i].id)
        {
            return true;
        }
    }

    return false;
}

bool CompareContentMetaKey(const nn::ncm::ContentMetaKey& va, const nn::ncm::ContentMetaKey& vb)
{
    // std::sort用 第一候補 id、第二候補 version の昇順にする
    const auto vaId = va.id;
    const auto vbId = vb.id;
    if (vaId == vbId)
    {
        return (va.version < vb.version);
    }
    return (vaId < vbId);
}

Result UpdateOrAddToList(ncm::ContentMetaKey list[], int* outCount, int listCount, int maxListCount, const ncm::ContentMetaKey& key) NN_NOEXCEPT
{
    const int lastNo = listCount - 1;
    if (lastNo >= 0)
    {
        if (key.id == list[lastNo].id && key.version > list[lastNo].version)
        {
            list[lastNo] = key;
            *outCount = listCount;
            NN_RESULT_SUCCESS;
        }
    }

    if (listCount < maxListCount)
    {
        list[listCount] = key;
        *outCount = listCount + 1;
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW(ResultBufferNotEnough());
}

} // namespace

    Result NetworkInstallTask::Initialize(DeviceContext* deviceContext, DeviceAccountStore* deviceAccountStore, const char* metaFilePath, const char* dataFilePath, ovln::SenderForOverlayType* sender) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Meta.Initialize(metaFilePath));
        NN_RESULT_DO(m_Data.Initialize(dataFilePath));
        NN_RESULT_DO(NetworkInstallTaskBase::Initialize(deviceContext, m_Meta.GetInstallStorage(), &m_Data, false, m_Meta.GetConfig()));

        m_Sender = sender;
        m_DeviceAccountStore = deviceAccountStore;

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallTask::Add(const ncm::ContentMetaKey keyList[], int count) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Meta.Add(keyList, count));
        NetworkInstallTaskBase::PrepareAgain();

        NN_RESULT_SUCCESS;
    }

    ncm::InstallProgress NetworkInstallTask::GetProgress() NN_NOEXCEPT
    {
        auto progress = InstallTaskBase::GetProgress();
        auto mappedResult = MapTaskResult(util::Get(progress.lastResult));
        std::memcpy(&(progress.lastResult), &mappedResult, sizeof(mappedResult));

        return progress;
    }


    int NetworkInstallTask::CountInstallContentMetaKey() NN_NOEXCEPT
    {
        return m_Meta.CountContentMeta();
    }

    Result NetworkInstallTask::GetInstallContentMetaKey(ncm::ContentMetaKey* outValue, int index) NN_NOEXCEPT
    {
        return m_Meta.GetContentMeta(outValue, index);
    }

    Result NetworkInstallTask::OnExecuteComplete() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallTask::UpdateAocInstallContentMetaData(ncm::ContentMetaKey outPrepared[], int *outCount, const ncm::ContentMetaKey latestKeyList[], int latestKeyCount, int maxAddOnContent) NN_NOEXCEPT
    {
        int count;
        NN_RESULT_DO(CountInstallContentMetaData(&count));

        // List に対して Delete/Add を繰り返す可能性があるため、
        // 先頭から走査すると index がずれる。
        // それを防ぐために末尾から先頭に向かって走査する。
        int preparedCount = 0;
        for (int i = count - 1; i >= 0 && preparedCount < maxAddOnContent; i--)
        {
            ncm::InstallContentMeta installContentMeta;
            NN_RESULT_DO(GetInstallContentMetaData(&installContentMeta, i));
            ncm::InstallContentMetaReader reader(installContentMeta.data.get(), installContentMeta.size);
            if (reader.GetHeader()->type == ncm::ContentMetaType::AddOnContent)
            {
                auto latestKey = GetIfIncludesNewer(reader.GetKey(), latestKeyList, latestKeyCount);
                if (latestKey)
                {
                    auto key = reader.GetKey();
                    NN_RESULT_DO(DeleteInstallContentMetaData(&key, 1));

                    NN_RESULT_DO(AddInstallContentMeta(*latestKey));
                }
                else
                {
                    latestKey = reader.GetKey();
                }

                outPrepared[preparedCount] = *latestKey;
                preparedCount++;
            }
        }
        *outCount = preparedCount;
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallTask::ListLatestInstalledContentMetaKey(ncm::ContentMetaKey out[], int* outCount, int maxOutCount, const ncm::ContentMetaKey excludeList[], int excludeListCount, ncm::ApplicationId& ownerId) NN_NOEXCEPT
    {
        static ncm::ContentMetaKey s_ExistKeyList[MaxAddOnContentPerApplication];
        const int MaxExistKeyListCount = sizeof(s_ExistKeyList) / sizeof(s_ExistKeyList[0]);

        static os::Mutex s_Mutex(false);
        std::lock_guard<os::Mutex> guard(s_Mutex);

        ncm::StorageId targetStorages[] = { ncm::StorageId::BuildInUser, ncm::StorageId::SdCard };

        int count = 0;
        for (auto targetStorage : targetStorages)
        {
            ncm::ContentMetaDatabase db;
            NN_RESULT_TRY(ncm::OpenContentMetaDatabase(&db, targetStorage))
                // TODO: ncm が正しい Result を返すようになったらそれをハンドルする
                NN_RESULT_CATCH_ALL { continue; }
            NN_RESULT_END_TRY;
            auto listCount = db.ListContentMeta(s_ExistKeyList, MaxExistKeyListCount, ncm::ContentMetaType::AddOnContent, ownerId);
            std::sort(s_ExistKeyList, s_ExistKeyList + listCount.listed, CompareContentMetaKey);

            for (int i = 0; i < listCount.listed; i++)
            {
                auto& key = s_ExistKeyList[i];
                if (IncludesId(key, excludeList, excludeListCount))
                {
                    continue;
                }

                NN_RESULT_DO(UpdateOrAddToList(out, &count, count, maxOutCount, key));
            }
        }
        *outCount = count;
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallTask::AddAocInstallContentMetaDataNewerThanInstalled(const ncm::ContentMetaKey preparedKeyList[], int preparedKeyCount, const ncm::ContentMetaKey latestKeyList[], int latestKeyCount, ncm::ApplicationId ownerId) NN_NOEXCEPT
    {
        static os::Mutex s_Mutex(false);
        std::lock_guard<os::Mutex> guard(s_Mutex);

        static ncm::ContentMetaKey s_LatestInstalledExistKeyList[MaxAddOnContentPerApplication];
        const int MaxLatestInstalledExistKeyCount = sizeof(s_LatestInstalledExistKeyList) / sizeof(s_LatestInstalledExistKeyList[0]);

        auto integratedKeyListCount = 0;
        NN_RESULT_DO(ListLatestInstalledContentMetaKey(s_LatestInstalledExistKeyList, &integratedKeyListCount, MaxLatestInstalledExistKeyCount, preparedKeyList, preparedKeyCount, ownerId));

        // インストール済みのメタと最新のメタを比較してインストール済みのものが古いなら新しいものをタスクに積む
        for (int i = 0; i < integratedKeyListCount; ++i)
        {
            auto& key = s_LatestInstalledExistKeyList[i];
            auto latestKey = GetIfIncludesNewer(key, latestKeyList, latestKeyCount);
            if (latestKey)
            {
                NN_RESULT_DO(AddInstallContentMeta(*latestKey));
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallTask::AddInstallContentMeta(const ncm::ContentMetaKey& key) NN_NOEXCEPT
    {
        ncm::InstallContentMetaInfo info;
        NN_RESULT_DO(GetInstallContentMetaInfo(&info, key));
        NN_RESULT_DO(PrepareContentMeta(info, key));
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallTask::HasAocInstallContentMeta(bool* out) NN_NOEXCEPT
    {
        int count;
        NN_RESULT_DO(CountInstallContentMetaData(&count));

        for (int i = 0; i < count; i++)
        {
            ncm::InstallContentMeta installContentMeta;
            NN_RESULT_DO(GetInstallContentMetaData(&installContentMeta, i));
            ncm::InstallContentMetaReader reader(installContentMeta.data.get(), installContentMeta.size);
            if (reader.GetHeader()->type == ncm::ContentMetaType::AddOnContent)
            {
                *out = true;
                NN_RESULT_SUCCESS;
            }
        }
        *out = false;
        NN_RESULT_SUCCESS;
    }

    bool NetworkInstallTask::IsAnyAocInstalled() NN_NOEXCEPT
    {
        auto storageList = ncm::GetStorageList(ncm::StorageId::Any);
        for (int i = 0; i < storageList.Count(); ++i)
        {
            auto targetStorage = storageList[i];
            if (ncm::IsInstallableStorage(targetStorage))
            {
                ncm::ContentMetaDatabase db;
                NN_RESULT_TRY(OpenContentMetaDatabase(&db, targetStorage))
                    NN_RESULT_CATCH_ALL
                {
                    continue;
                }
                NN_RESULT_END_TRY;

                auto listCount = db.ListContentMeta(nullptr, 0, ncm::ContentMetaType::AddOnContent, GetApplicationId());
                if (listCount.total > 0)
                {
                    return true;
                }
            }
        }
        return false;
    }

    Result NetworkInstallTask::PrepareAocDependency() NN_NOEXCEPT
    {
        bool hasAocInstallContentMeta;
        NN_RESULT_DO(HasAocInstallContentMeta(&hasAocInstallContentMeta));
        if (!hasAocInstallContentMeta && !IsAnyAocInstalled())
        {
            // インストールタスクにもストレージにも Aoc が存在しなければ Aoc の依存は存在しないことが確定するので
            // superfly へのアクセスを行わずに直ちに終了する。
            NN_RESULT_SUCCESS;
        }

        auto ownerId = GetApplicationId();

        static os::Mutex s_Mutex(false);
        std::lock_guard<os::Mutex> guard(s_Mutex);

        static ncm::ContentMetaKey s_LatestKeyList[MaxOperationalContentCountPerApplication];
        const int MaxLatestKeyCount = sizeof(s_LatestKeyList) / sizeof(s_LatestKeyList[0]);

        static ncm::ContentMetaKey s_PreparedKeyList[MaxAddOnContentPerApplication];
        const int MaxPreparedKeyCount = sizeof(s_PreparedKeyList) / sizeof(s_PreparedKeyList[0]);

        // 最新バージョンをサーバに問い合わせる。
        int latestKeyCount;
        NN_RESULT_TRY(GetLatestContentMetaKey(s_LatestKeyList, &latestKeyCount, MaxLatestKeyCount, GetHttpConnection(), ownerId.value))
            NN_RESULT_CATCH(ResultLatestVersionNotFound)
            {
                // 最新バージョンのエントリが見つからない場合は、Aoc の更新なしとして依存解決を終了する
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        // インストールタスクを確認して古いものをインストールしているならそれを消して、新しいものを積む
        int preparedCount = 0;
        NN_RESULT_DO(UpdateAocInstallContentMetaData(s_PreparedKeyList, &preparedCount, s_LatestKeyList, latestKeyCount, MaxPreparedKeyCount));

        // インストール済みの追加コンテンツを確認し
        // 古いものがインストールされているなら新しいものをインストールするタスクを積む
        NN_RESULT_DO(AddAocInstallContentMetaDataNewerThanInstalled(s_PreparedKeyList, preparedCount, s_LatestKeyList, latestKeyCount, ownerId));

        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallTask::RegisterRoute(const ncm::PatchMetaExtendedDataReader& reader, const ncm::ContentMetaKey& key, const RouteFinder* routeFinder) NN_NOEXCEPT
    {
        // 途中キャンセルされた場合に中途半端な状態にならないよう、ロールバックする
        bool success = false;
        int processedIndex = 0;
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                NN_DETAIL_NIM_TRACE("[DeltaUpdate] Do rollback\n");
                auto result = RollbackRoute(reader, routeFinder, processedIndex);
                if (result.IsFailure())
                {
                    NN_DETAIL_NIM_TRACE("[DeltaUpdate] Rollback failed %08x\n", result.GetInnerValueForDebug());
                }
            }
        };
        for (; processedIndex < routeFinder->GetRouteCount(); ++processedIndex)
        {
            auto route = routeFinder->GetRoute(processedIndex);
            auto delta = reader.GetPatchDeltaHistory(route.deltaIndex);
            auto patchKey = ncm::ContentMetaKey::Make(delta->destinationId, delta->destinationVersion);

            {
                NN_DETAIL_NIM_TRACE("[DeltaUpdate] register dependency: %016llx, %u\n", patchKey.id, patchKey.version);
                ncm::InstallContentMetaInfo info;
                NN_RESULT_TRY(GetInstallContentMetaInfo(&info, patchKey))
                    NN_RESULT_CATCH(ResultHttpStatus404NotFound)
                    {
                        NN_DETAIL_NIM_TRACE("[DeltaUpdate] dependency 404\n");
                        NN_RESULT_SUCCESS;
                    }
                NN_RESULT_END_TRY
                NN_RESULT_TRY(PrepareContentMeta(info, patchKey, delta->sourceVersion))
                    NN_RESULT_CATCH(ResultHttpStatus404NotFound)
                    {
                        NN_DETAIL_NIM_TRACE("[DeltaUpdate] dependency 404\n");
                        NN_RESULT_SUCCESS;
                    }
                NN_RESULT_END_TRY
            }
        }
        success = true;

        NN_DETAIL_NIM_TRACE("[DeltaUpdate] Delete registered task\n");
        NN_RESULT_DO(DeleteInstallContentMetaData(&key, 1));
        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallTask::RollbackRoute(const ncm::PatchMetaExtendedDataReader& reader, const RouteFinder* routeFinder, int index) NN_NOEXCEPT
    {
        for (int i = 0; i < index; ++i)
        {
            auto route = routeFinder->GetRoute(i);
            auto delta = reader.GetPatchDeltaHistory(route.deltaIndex);
            auto patchKey = ncm::ContentMetaKey::Make(delta->destinationId, delta->destinationVersion, ncm::ContentInstallType::FragmentOnly);

            NN_DETAIL_NIM_TRACE("[DeltaUpdate] unregister: %016llx, %u\n", patchKey.id, patchKey.version);
            NN_RESULT_DO(DeleteInstallContentMetaData(&patchKey, 1));
        }
        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallTask::ResolvePatchUpdateRoute(const ncm::ContentMetaKey& key, const ncm::PackagedContentMetaReader& reader) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(key.type == ncm::ContentMetaType::Patch);

        ncm::StorageId storageId;
        ncm::PatchId patchId = {key.id};
        NN_RESULT_DO(ncm::SelectPatchStorage(&storageId, GetInstallStorage(), patchId));
        if (!ncm::IsUniqueStorage(storageId))
        {
            NN_DETAIL_NIM_TRACE("[DeltaUpdate] No patches are installed, download directly\n");
            NN_RESULT_SUCCESS;
        }

        // ストレージを指定したタスクの場合は、上記判定を抜けてくることがあるので、改めてチェック
        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, storageId));
        ncm::ContentMetaKey installedKey;
        NN_RESULT_TRY(db.GetLatest(&installedKey, key.id))
            NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
            {
                // direct
                NN_DETAIL_NIM_TRACE("[DeltaUpdate] No patches are installed, download directly\n");
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        // パッチのヒストリに登録されている情報と、コンテントメタデータベース上の比較
        bool existInHistory;
        NN_RESULT_DO(VerifyPatchHistory(&existInHistory, &db, installedKey, reader));
        if (existInHistory == false)
        {
            // direct
            NN_DETAIL_NIM_TRACE("[DeltaUpdate] Installed patch is NOT registered in the history, download directly\n");
            NN_RESULT_SUCCESS;
        }

        // 差分での更新経路が見つからなければ direct
        ncm::PatchMetaExtendedDataReader exReader(reader.GetExtendedData(), reader.GetExtendedDataSize());
        static RouteFinder routeFinder;
        NN_RESULT_TRY(routeFinder.Initialize(exReader, installedKey.version, key.version))
            NN_RESULT_CATCH(ResultTooManyStepsToUpdate)
            {
                // direct
                NN_DETAIL_NIM_TRACE("[DeltaUpdate] Too many steps, download directly\n");
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY
        if (!routeFinder.HasRoute())
        {
            // direct
            NN_DETAIL_NIM_TRACE("[DeltaUpdate] No route found\n");
            NN_RESULT_SUCCESS;
        }
        auto directSize = GetDirectDownloadSize(reader);
        auto indirectSize = routeFinder.GetIndirectDownloadSize(exReader);

        // 直接更新と差分での更新で、直接のほうがダウンロードサイズが小さければ direct
        if (directSize < indirectSize)
        {
            NN_DETAIL_NIM_TRACE("[DeltaUpdate] direct is better route (direct: %lld / indirect: %lld)\n", directSize, indirectSize);
            if (!WillPreferDeltaForcibly())
            {
                NN_RESULT_SUCCESS;
            }
            NN_DETAIL_NIM_TRACE("[DeltaUpdate] but currently not use direct based on fwdbg\n");
        }

        NN_RESULT_DO(RegisterRoute(exReader, key, &routeFinder));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallTask::PreparePatchDependency(const ncm::ContentMetaKey& key) NN_NOEXCEPT
    {
        // まず、key をダウンロード
        // TODO: テンポラリにシステムのストレージ
        ncm::InstallContentMetaInfo info;
        NN_RESULT_DO(GetInstallContentMetaInfo(&info, key));

        ncm::ContentStorage storage;
        NN_RESULT_DO(OpenContentStorage(&storage, ncm::StorageId::BuildInSystem));

        ncm::InstallContentInfo installInfo;
        NN_RESULT_DO(WriteContentMetaToPlaceHolder(&installInfo, &storage, info));
        NN_UTIL_SCOPE_EXIT
        {
            auto result = storage.DeletePlaceHolder(installInfo.placeHolderId);
            if (result.IsFailure() && !ncm::ResultPlaceHolderNotFound().Includes(result))
            {
                NN_DETAIL_NIM_TRACE("[InstallTaskBase] Failed to delete place holder as 0x%08x\n", result.GetInnerValueForDebug());
            }
        };

        ncm::Path path;
        storage.GetPlaceHolderPath(&path, installInfo.placeHolderId);

        // パッチのコンテントメタなので大きいことに注意
        ncm::AutoBuffer buffer;
        {
            fs::ScopedAutoAbortDisabler disableAbort;
            NN_RESULT_DO(ncm::ReadContentMetaPath(&buffer, path.string));
        }
        ncm::PackagedContentMetaReader reader(buffer.Get(), buffer.GetSize());

        // パッチの更新ルートを決めて、必要ならタスクの再登録を行う
        NN_RESULT_DO(ResolvePatchUpdateRoute(reader.GetKey(), reader));

        NN_DETAIL_NIM_TRACE("[DeltaUpdate] Resolved\n");

        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallTask::PrepareDependency() NN_NOEXCEPT
    {
        if (GetConfig() & ncm::InstallConfig_LatestAddOnContentsAuto)
        {
            NN_RESULT_DO(PrepareAocDependency());
        }
        if (GetConfig() & ncm::InstallConfig_SkipIndirectUpdateCheck)
        {
            NN_DETAIL_NIM_TRACE("[NetworkInstallTask] Indirect update check is skipped\n");
            NN_RESULT_SUCCESS;
        }
        int count;
        NN_RESULT_DO(CountInstallContentMetaData(&count));

        for (int i = 0; i < count; i++)
        {
            ncm::InstallContentMeta installContentMeta;
            NN_RESULT_DO(GetInstallContentMetaData(&installContentMeta, i));
            ncm::InstallContentMetaReader reader(installContentMeta.data.get(), installContentMeta.size);
            auto key = reader.GetKey();
            if (key.type == ncm::ContentMetaType::Patch && key.installType == ncm::ContentInstallType::Full)
            {
                NN_RESULT_DO(PreparePatchDependency(key));
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallTask::InstallTicket(const nn::fs::RightsId& rightsId, nn::ncm::ContentMetaType contentMetaType) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コモンチケットの取得のみを試行する ContentMetaType の場合
        if (contentMetaType == ncm::ContentMetaType::Patch)
        {
            NN_RESULT_DO(GetAndInstallCommonTicket(GetHttpConnection(), nn::es::RightsIdIncludingKeyId::Construct(rightsId)));
            NN_RESULT_SUCCESS;
        }

        DeviceAccountInfo deviceAccountInfo;
        bool hasDeviceAccount = true;

        NN_RESULT_TRY(m_DeviceAccountStore->Get(&deviceAccountInfo))
            NN_RESULT_CATCH(ResultDeviceAccountNotRegistered)
            {
                hasDeviceAccount = false;
            }
        NN_RESULT_END_TRY

        // チケットのダウンロード
        NN_RESULT_TRY(GetAndInstallCommonTicket(GetHttpConnection(), nn::es::RightsIdIncludingKeyId::Construct(rightsId)))
            NN_RESULT_CATCH(ResultCommonTicketNotFound)
            {
                if(hasDeviceAccount)
                {
                    EciAccessor accessor(GetDeviceContext(), GetHttpConnection());

                    NN_RESULT_TRY(accessor.DownloadTicket(deviceAccountInfo.id, deviceAccountInfo.token, nn::es::RightsIdIncludingKeyId::Construct(rightsId)))
                        NN_RESULT_CATCH(ResultPersonalizedTicketNotFound)
                        {
                            NN_RESULT_THROW(nn::ncm::ResultIgnorableInstallTicketFailure());
                        }
                    NN_RESULT_END_TRY
                }
                else
                {
                    NN_RESULT_THROW(ResultDeviceAccountNotRegisteredForDownloadTicket());
                }
            }
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;

#else
        NN_UNUSED(rightsId);
        NN_UNUSED(contentMetaType);

#endif
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallTask::ListContentMetaKeyFromInstallMeta(int* outCount, ncm::ContentMetaKey* outList, int numList, int offset) NN_NOEXCEPT
    {
        const int CountMeta = m_Meta.CountContentMeta();

        int index = 0;
        for (int  i = offset; i < CountMeta && index < numList; ++i)
        {
            NN_RESULT_DO(m_Meta.GetContentMeta(&outList[index], i));
            ++index;
        }

        *outCount = index;
        NN_RESULT_SUCCESS;
    }
}}}
