﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <cstring>
#include <limits>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/nim/nim_Result.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/http/json/http_JsonPath.h>
#include <nn/http/json/http_JsonErrorMap.h>
#include <nn/http/json/http_RapidJsonApi.h>
#include <nn/http/json/http_RapidJsonInputStream.h>


namespace nn { namespace nim { namespace srv {

namespace HttpJson {

//------------------------------------------------------------------------------------------------------
/**
 * @brief   JSONエントリルックアップ
 */
template<int SizeOfJsonPathMaxDepth, int SizeOfJsonPathMaxLength>
struct EntryLookup
{
    typedef ::nn::http::json::JsonPath<SizeOfJsonPathMaxDepth, SizeOfJsonPathMaxLength> JsonPathType;
    char    pPath[SizeOfJsonPathMaxLength];
    bool    found;

    explicit EntryLookup() NN_NOEXCEPT : pPath(""), found(false) {}

    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return found;
    }

    bool CanAccept(const JsonPathType& jsonPath) const NN_NOEXCEPT
    {
        return !found && jsonPath.Match(pPath);
    }

    void MarkAccepted() NN_NOEXCEPT
    {
        found = true;
    }

    void Reset() NN_NOEXCEPT
    {
        found = false;
        pPath[0] = '\0';
    }
};

//------------------------------------------------------------------------------------------------------
/**
 * @brief   JSONエントリストアベース
 */
template<typename Lookup>
class EntryStoreBase
{
private:
    Lookup  m_Lookup;

public:
    explicit EntryStoreBase() NN_NOEXCEPT {}

    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return static_cast<bool>(m_Lookup);
    }

    void Reset() NN_NOEXCEPT
    {
        m_Lookup.Reset();
    }

    char* GetLookupPathTop() NN_NOEXCEPT
    {
        return m_Lookup.pPath;
    }

    size_t GetLookupPathCapacity() const NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(sizeof(m_Lookup.pPath) < std::numeric_limits<int>::max());
        return sizeof(m_Lookup.pPath);
    }

    bool CreateLookupPath(const char* const pSourcePath) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(nullptr != pSourcePath);
        const auto size = GetLookupPathCapacity();
        const size_t sourceLength = ::nn::util::Strlcpy(GetLookupPathTop(), pSourcePath, static_cast<int>(size));
        return (sourceLength < size);
    }

    template<typename TArgs>
    bool CreateLookupPath(const char* const pPathFormat, TArgs index) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(nullptr != pPathFormat);
        const auto size = GetLookupPathCapacity();
        const size_t length = ::nn::util::SNPrintf(GetLookupPathTop(), size, pPathFormat, index);
        return (length < size);
    }

    bool CanAccept(const typename Lookup::JsonPathType& jsonPath) const NN_NOEXCEPT
    {
        return m_Lookup.CanAccept(jsonPath);
    }

    void MarkAccepted() NN_NOEXCEPT
    {
        m_Lookup.MarkAccepted();
    }
};

//------------------------------------------------------------------------------------------------------
/**
 * @brief   JSONエントリストア( 文字列 )
 */
template<size_t StoreCapacity, typename Lookup>
class StringStore : public EntryStoreBase<Lookup>
{
public:
    static const size_t Capacity = StoreCapacity;
    typedef EntryStoreBase<Lookup> BaseType;
    typedef char ValueType;

    /**
     * @brief       デフォルトコンストラクタ。
     */
    explicit StringStore() NN_NOEXCEPT : m_Value(""), m_Length(0) {}

    /**
     * @brief       文字列データをストアします。
     *
     * @param[in]   jsonPath    @ref ::nn::http::json::JsonPath で表現された JSONパス文字列コンテナ。@n
     *                          @a pValue が存在するJSONパスを保持している想定です。
     * @param[in]   pValue      ストア対象文字列データ先頭アドレス。( null 終端想定 )
     * @param[in]   valueLength ストア対象文字列データ長。( null 終端含まない )
     *
     * @return      ストア結果を返します。
     * @retval      nn::ResultSuccess               正常にストアされました。
     * @retval      nn::nim::ResultContentNotFound  指定の JsonPath は、このエントリストアが管理すべきパスではありません。
     * @retval      nn::nim::ResultBufferNotEnough  指定された文字列を保持するための内部ストアストレージ容量が不足しています。
     *
     * @details     ストアされた文字列には、null終端文字が付与されます。@n
     *              そのため、ストア容量は期待する文字列長 + null終端分を考慮した容量を指定してください。
     */
    Result Store(const typename Lookup::JsonPathType& jsonPath, const char* pValue, int valueLength) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(BaseType::CanAccept(jsonPath) && valueLength > 0, ResultContentNotFound());
        NN_RESULT_THROW_UNLESS(sizeof(m_Value) > static_cast<size_t>(valueLength), ResultBufferNotEnough());

        m_Length = ::nn::util::Strlcpy(m_Value, pValue, sizeof(m_Value));
        BaseType::MarkAccepted();
        NN_RESULT_SUCCESS;
    }

    NN_FORCEINLINE const char* GetValue() const NN_NOEXCEPT
    {
        return m_Value;
    }

    NN_FORCEINLINE int GetLength() const NN_NOEXCEPT
    {
        return m_Length;
    }

    void Reset() NN_NOEXCEPT
    {
        BaseType::Reset();
        m_Value[0] = '\0';
        m_Length = 0;
    }

private:
    char    m_Value[StoreCapacity];
    int     m_Length;
};

//------------------------------------------------------------------------------------------------------
/**
 * @brief   JSONエントリストア( 値 )
 */
template<typename TValue, typename Lookup>
class ValueStore : public EntryStoreBase<Lookup>
{
public:
    typedef EntryStoreBase<Lookup> BaseType;
    typedef TValue ValueType;

    /**
     * @brief       デフォルトコンストラクタ。
     */
    explicit ValueStore() NN_NOEXCEPT : m_Value() {}

    /**
     * @brief       初期値指定コンストラクタ。
     */
    explicit ValueStore(const ValueType& initialValue) NN_NOEXCEPT : m_Value(initialValue) {}

    /**
     * @brief       値データをストアします。
     *
     * @param[in]   jsonPath    @ref ::nn::http::json::JsonPath で表現された JSONパス文字列コンテナ。@n
     *                          @a value が存在するJSONパスを保持している想定です。
     * @param[in]   value       ストア対象値データ。
     *
     * @return      ストア結果を返します。
     * @retval      nn::ResultSuccess               正常にストアされました。
     * @retval      nn::nim::ResultContentNotFound  指定の JsonPath は、このエントリストアが管理すべきパスではありません。
     */
    Result Store(const typename Lookup::JsonPathType& jsonPath, const ValueType value) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(BaseType::CanAccept(jsonPath), ResultContentNotFound());

        m_Value = static_cast<ValueType>(value);
        BaseType::MarkAccepted();
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief       文字列データを変換してストアします。
     *
     * @tparam      Predicate   変換式を指定します。@n
     *                          型は `Result Predicate(TValue* pOut, const char* pValue, int valueLength) NN_NOEXCEPT;` です。
     *
     * @param[in]   jsonPath    @ref ::nn::http::json::JsonPath で表現された JSONパス文字列コンテナ。@n
     *                          @a pValue が存在するJSONパスを保持している想定です。
     * @param[in]   pValue      ストア対象文字列データ先頭アドレス。( null 終端想定 )
     * @param[in]   valueLength ストア対象文字列データ長。( null 終端含まない )
     *
     * @return      ストア結果を返します。
     * @retval      nn::ResultSuccess               正常にストアされました。
     * @retval      nn::nim::ResultContentNotFound  指定の JsonPath は、このエントリストアが管理すべきパスではありません。
     * @retval      nn::nim::ResultBufferNotEnough  指定された文字列を保持するための内部ストアストレージ容量が不足しています。
     *
     * @details     ストアされた文字列には、null終端文字が付与されます。@n
     *              そのため、ストア容量は期待する文字列長 + null終端分を考慮した容量を指定してください。
     */
    template<typename Predicate>
    Result Store(const typename Lookup::JsonPathType& jsonPath, const char* pValue, int valueLength, Predicate predicate) NN_NOEXCEPT
    {
        ValueType value;
        NN_RESULT_THROW_UNLESS(BaseType::CanAccept(jsonPath) && valueLength > 0, ResultContentNotFound());
        NN_RESULT_DO(predicate(&value, pValue, valueLength));

        m_Value = static_cast<ValueType>(value);
        BaseType::MarkAccepted();
        NN_RESULT_SUCCESS;
    }

    NN_FORCEINLINE ValueType GetValue() const NN_NOEXCEPT
    {
        return m_Value;
    }

    void Reset(const ValueType& source) NN_NOEXCEPT
    {
        BaseType::Reset();
        m_Value = source;
    }

    void Reset() NN_NOEXCEPT
    {
        BaseType::Reset();
        m_Value = {};
    }

private:
    ValueType   m_Value;
};

//------------------------------------------------------------------------------------------------------
namespace EntryStoreUtil
{

//------------------------------------------------------------------------------------------------------
//! @brief      無視可能なリザルトをフィルタします。
//! @return     無視できるリザルトの場合、@ref nn::ResultSuccess() が返ります。
NN_FORCEINLINE Result FilterIgnorable(const Result& result) NN_NOEXCEPT
{
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(ResultContentNotFound)
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

//------------------------------------------------------------------------------------------------------
}   // ~EntryStoreUtil

//------------------------------------------------------------------------------------------------------
/**
 * @brief   ::nn::http::json::ImportJsonByRapidJson() 用キャンセル概念型。
 */
class Canceler
{
public:
    Canceler() NN_NOEXCEPT {}

    virtual void Cancel() NN_NOEXCEPT {}
    virtual bool IsCancelled() const NN_NOEXCEPT
    {
        return false;
    }
};

//------------------------------------------------------------------------------------------------------
namespace Stream {
namespace Base {

//------------------------------------------------------------------------------------------------------
/**
 * @brief   RapidJson 用代理出力ストリーム基底実装クラス。
 *
 * @tparam  TStream     出力ストリーム型。
 *
 * @details このクラス及び、派生先クラスは、テンプレートメソッドから呼び出される事を想定した静的インターフェースクラスでの運用を想定しています。@n
 *          Output対象ストリーム実体への参照を保持して代替呼び出しを行います。
 */
template<typename TStream, typename Ch = char>
class OutputProxy
{
    NN_DISALLOW_COPY(OutputProxy);

public:
    //! @name output interface.
    //! @{
    Ch * PutBegin() NN_NOEXCEPT
    {
        return m_pStream->PutBegin();
    }

    void Put(Ch value) NN_NOEXCEPT
    {
        m_pStream->Put(value);
    }

    void Flush() NN_NOEXCEPT
    {
        m_pStream->Flush();
    }

    size_t PutEnd(Ch* pValue) NN_NOEXCEPT
    {
        return m_pStream->PutEnd(pValue);
    }
    //! @}

    /**
     * @brief       コンストラクタ
     *
     * @param[in]   pStream     出力ストリーム実体。
     */
    explicit OutputProxy(TStream* pStream) NN_NOEXCEPT
        : m_pStream(pStream)
    {
        NN_SDK_ASSERT_NOT_NULL(pStream);
    }

private:
    TStream* const  m_pStream;
};

//------------------------------------------------------------------------------------------------------
/**
 * @brief   RapidJson 用代理入力ストリーム基底実装クラス。
 *
 * @tparam  TStream     入力ストリーム型。
 *
 * @details このクラス及び、派生先クラスは、テンプレートメソッドから呼び出される事を想定した静的インターフェースクラスでの運用を想定しています。@n
 *          Input対象ストリーム実体への参照を保持して代替呼び出しを行います。
 */
template<typename TStream, typename Ch = char>
class InputProxy
{
    NN_DISALLOW_COPY(InputProxy);

public:
    Ch Peek() const NN_NOEXCEPT
    {
        return m_pStream->Peek();
    }

    Ch Take() NN_NOEXCEPT
    {
        return m_pStream->Take();
    }

    size_t Tell() const NN_NOEXCEPT
    {
        return m_pStream->Tell();
    }

    //! @brief      ストリームアクセス時にエラー( I/O など )が発生した場合の Result を返します。
    //! @details    ImportJsonByRapidJson でパース完了後に参照され ResultSuccess 以外の場合は、ImportJsonByRapidJson のエラー値として返却されます。
    Result GetResult() const NN_NOEXCEPT
    {
        return m_pStream->GetResult();
    }

    /**
     * @brief       コンストラクタ
     *
     * @param[in]   pStream     入力ストリーム実体。
     */
    explicit InputProxy(TStream* pStream) NN_NOEXCEPT
        : m_pStream(pStream)
    {
        NN_SDK_ASSERT_NOT_NULL(pStream);
    }

private:
    TStream* const m_pStream;
};

//------------------------------------------------------------------------------------------------------
/**
 * @brief   RapidJson 用フルオンメモリ入力ストリーム基底実装クラス。
 *
 * @details このクラス及び、派生先クラスは、テンプレートメソッドから呼び出される事を想定した静的インターフェースクラスでの運用を想定しています。@n
 *          派生先クラスで出力ストリームの実装が必要です。
 */
template<typename Ch = char>
class MemoryInput
{
    NN_DISALLOW_COPY(MemoryInput);

public:
    Ch Peek() const NN_NOEXCEPT
    {
        return m_Pos == m_Bytes ? '\0' : m_pSourceStream[m_Pos];
    }

    Ch Take() NN_NOEXCEPT
    {
        auto c = Peek();
        if (c != '\0')
        {
            ++m_Pos;
        }
        return c;
    }

    size_t Tell() const NN_NOEXCEPT
    {
        return m_Pos;
    }

    //! @brief      ストリームアクセス時にエラー( I/O など )が発生した場合の Result を返します。
    //! @details    ImportJsonByRapidJson でパース完了後に参照され ResultSuccess 以外の場合は、ImportJsonByRapidJson のエラー値として返却されます。
    Result GetResult() const NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief       コンストラクタ
     *
     * @param[in]   pStream     JSONソースストリーム。
     * @param[in]   streamBytes JSONソースストリームサイズ。( 終端 null 含まない )
     */
    explicit MemoryInput(const Ch* pStream, size_t streamBytes) NN_NOEXCEPT
        : m_pSourceStream(pStream)
        , m_Bytes(streamBytes)
        , m_Pos(0u)
    {
        NN_SDK_ASSERT(streamBytes > 0u);
        NN_SDK_ASSERT_NOT_NULL(pStream);
    }

private:
    const Ch* const m_pSourceStream;
    const size_t    m_Bytes;
    size_t          m_Pos;
};

//------------------------------------------------------------------------------------------------------
/**
 * @brief   RapidJson 用バッファリング入力ストリーム基底実装クラス。
 *
 * @details このクラス及び、派生先クラスは、テンプレートメソッドから呼び出される事を想定した静的インターフェースクラスでの運用を想定しています。@n
 *          派生先クラスで出力ストリームの実装が必要です。
 */
template<typename Ch = char>
class BufferedInput
{
    NN_DISALLOW_COPY(BufferedInput);

private:
    Ch* const       m_pBuffer;
    const size_t    m_BufferSize;
    size_t          m_TotalBufferedBytes;
    size_t          m_BufferedBytes;
    size_t          m_PositionInBuffer;

private:
    bool FillBuffer() NN_NOEXCEPT
    {
        m_PositionInBuffer = 0u;
        m_BufferedBytes = OnFill(m_pBuffer, m_BufferSize);
        m_TotalBufferedBytes += m_BufferedBytes;
        return m_BufferedBytes > 0u;
    }

protected:
    /**
     * @brief       コンストラクタ。
     * @param[in]   pBuffer         バッファリングメモリ領域先頭アドレス。
     * @param[in]   bufferSize      バッファリングメモリ領域サイズ( バイト単位 )。
     */
    explicit BufferedInput(Ch* pBuffer, size_t bufferSize) NN_NOEXCEPT
        : m_pBuffer(pBuffer)
        , m_BufferSize(bufferSize)
        , m_TotalBufferedBytes(0u)
        , m_BufferedBytes(0u)
        , m_PositionInBuffer(0u)
    {
        NN_SDK_ASSERT(bufferSize > 0u);
        NN_SDK_ASSERT_NOT_NULL(pBuffer);
    }

    //! @brief      新しいバッファリングデータのフィル要求時に呼び出されます。
    //! @details    新しいデータ要求は、バッファ済データを全て消費した際に発生します。
    virtual void OnFill(void* pBuffer, size_t bufferSize) NN_NOEXCEPT = 0;

public:
    Ch Peek() NN_NOEXCEPT
    {
        if (m_PositionInBuffer >= m_BufferedBytes)
        {
            NN_SDK_ASSERT(m_PositionInBuffer == m_BufferedBytes);
            if (!FillBuffer())
            {
                return '\0';
            }
        }
        return m_pBuffer[m_PositionInBuffer];
    }

    Ch Take() NN_NOEXCEPT
    {
        auto c = Peek();
        ++m_PositionInBuffer;
        return c;
    }

    size_t Tell() const NN_NOEXCEPT
    {
        return (m_TotalBufferedBytes - m_BufferedBytes) + m_PositionInBuffer;
    }

    //! @brief      ストリームアクセス時にエラー( I/O など )が発生した場合の Result を返します。
    //! @details    ImportJsonByRapidJson でパース完了後に参照され ResultSuccess 以外の場合は、ImportJsonByRapidJson のエラー値として返却されます。
    virtual Result GetResult() const NN_NOEXCEPT = 0;
};

//------------------------------------------------------------------------------------------------------
}   // ~Base


//------------------------------------------------------------------------------------------------------
/**
 * @brief   http::json ライブラリの標準的な出力ストリームの定義です。
 */
typedef ::nn::http::json::StringBufferForRapidJson  StandardOutputStream;

//------------------------------------------------------------------------------------------------------
/**
 * @brief   RapidJson 用フルオンメモリ入力、代替出力ストリーム実装クラス。@n
 *          主に、「自身が入力ストリームを作成し、外部で生成された出力ストリームに対して出力を行う経路」で使用します。
 *
 * @tparam  TOutputStream   出力ストリーム型。
 *
 * @details このクラス及び、派生先クラスは、テンプレートメソッドから呼び出される事を想定した静的インターフェースクラスでの運用を想定しています。@n
 *          Output対象ストリーム実体への参照を保持して代替呼び出しを行います。
 */
template<typename TOutputStream, typename TChar = char>
class MemoryInputWithProxyOutput
    : public Stream::Base::MemoryInput<TChar>
    , public Stream::Base::OutputProxy<TOutputStream, TChar>
{
    NN_DISALLOW_COPY(MemoryInputWithProxyOutput);

public:
    typedef TChar Ch;
    typedef Stream::Base::MemoryInput<Ch>                   InputBase;
    typedef Stream::Base::OutputProxy<TOutputStream, Ch>    OutputBase;

    /**
     * @brief       コンストラクタ
     *
     * @param[in]   pStream         JSONソースストリーム。
     * @param[in]   streamBytes     JSONソースストリームサイズ。( 終端 null 含まない )
     * @param[in]   pOutputStream   出力ストリーム実体。
     */
    explicit MemoryInputWithProxyOutput(const char* pStream, size_t streamBytes, TOutputStream* pOutputStream) NN_NOEXCEPT
        : InputBase(pStream, streamBytes), OutputBase(pOutputStream)
    {
    }
};

//------------------------------------------------------------------------------------------------------
/**
 * @brief   RapidJson 用フルオンメモリ入力、内部確保済固定長領域出力ストリームクラス。
 *
 * @tparam  ConstantStringBufferCapacity    "key: value" の単位の検出要素格納バッファ容量サイズ( バイトサイズ )。
 *
 * @details このクラス及び、派生先クラスは、テンプレートメソッドから呼び出される事を想定した静的インターフェースクラスでの運用を想定しています。@n
 *          内部で ConstantOutputStreamCapacity で指定したバイトサイズの領域を確保して出力ストリームとして扱います。
 */
template<size_t ConstantOutputStreamCapacity = 1024>
class MemoryInputWithStaticOutput
    : public Stream::Base::MemoryInput<char>
    , public StandardOutputStream
{
    NN_DISALLOW_COPY(MemoryInputWithStaticOutput);

public:
    typedef char                            Ch;
    typedef Stream::Base::MemoryInput<Ch>   InputBase;
    typedef StandardOutputStream            OutputBase;
    static const size_t OutputStreamCapacity = ConstantOutputStreamCapacity;

    /**
     * @brief       コンストラクタ
     *
     * @param[in]   pStream     JSONソースストリーム。
     * @param[in]   streamBytes JSONソースストリームサイズ。( 終端 null 含まない )
     */
    explicit MemoryInputWithStaticOutput(const char* pStream, size_t streamBytes) NN_NOEXCEPT
        : InputBase(pStream, streamBytes), OutputBase()
    {
        OutputBase::SetStringBuffer(m_OutputBuffer, OutputStreamCapacity);
    }

private:
    Ch m_OutputBuffer[OutputStreamCapacity];
};

//------------------------------------------------------------------------------------------------------
/**
 * @brief   RapidJson 用 ::nn::http::json::StringBufferForRapidJson 出力ストリーム、代替入力ストリームクラス。@n
 *          主に、「自身が出力ストリームを作成し、外部で生成された入力ストリームから入力を行う経路」で使用します。
 *
 * @tparam  ConstantStringBufferCapacity    "key: value" の単位の検出要素格納バッファ容量サイズ( バイトサイズ )。
 *
 * @details このクラス及び、派生先クラスは、テンプレートメソッドから呼び出される事を想定した静的インターフェースクラスでの運用を想定しています。@n
 *          Input 対象ストリーム実体への参照を保持して代替呼び出しを行います。@n
 *          内部で ConstantOutputStreamCapacity で指定したバイトサイズの領域を確保して出力ストリームとして扱います。
 */
template<typename TInputStream, size_t ConstantOutputStreamCapacity = 1024>
class ProxyInputStream
    : public Stream::Base::InputProxy<TInputStream, char>
    , public StandardOutputStream
{
    NN_DISALLOW_COPY(ProxyInputStream);

public:
    typedef char                                        Ch;
    typedef Stream::Base::InputProxy<TInputStream, Ch>  InputBase;
    typedef StandardOutputStream                        OutputBase;
    static const size_t OutputStreamCapacity = ConstantOutputStreamCapacity;

    /**
     * @brief       コンストラクタ
     *
     * @param[in]   pInputStream    入力ストリーム実体。
     */
    explicit ProxyInputStream(TInputStream* pInputStream) NN_NOEXCEPT
        : InputBase(pInputStream), OutputBase()
    {
        OutputBase::SetStringBuffer(m_OutputBuffer, OutputStreamCapacity);
    }

private:
    Ch m_OutputBuffer[OutputStreamCapacity];
};

//------------------------------------------------------------------------------------------------------
/**
 * @brief   RapidJson ( nn::http::json ) 用 代替出力ストリーム実装クラス、代替入力ストリームクラス呼び出しプロキシ実装クラス。@n
 *          「外部生成された入力ストリーム及び出力ストリームに対して入出力を行う経路」で使用します。
 *
 * @tparam  TInputStream    入力ストリーム型。
 * @tparam  TOutputStream   出力ストリーム型。
 *
 * @details このクラス及び、派生先クラスは、テンプレートメソッドから呼び出される事を想定した静的インターフェースクラスでの運用を想定しています。@n
 *          Input / Output 対象ストリーム実体への参照を保持して代替呼び出しを行います。
 */
template<typename TInputStream, typename TOutputStream, typename TChar = char>
class ProxyStream
    : public Stream::Base::InputProxy<TInputStream, TChar>
    , public Stream::Base::OutputProxy<TOutputStream, TChar>
{
    NN_DISALLOW_COPY(ProxyStream);

public:
    typedef TChar                                            Ch;
    typedef Stream::Base::InputProxy<TInputStream, Ch>      InputBase;
    typedef Stream::Base::OutputProxy<TOutputStream, Ch>    OutputBase;

    /**
     * @brief       コンストラクタ
     *
     * @param[in]   pInputStream    入力ストリーム実体。
     * @param[in]   pOutputStream   出力ストリーム実体。
     */
    explicit ProxyStream(TInputStream* pInputStream, TOutputStream* pOutputStream) NN_NOEXCEPT
        : InputBase(pInputStream), OutputBase(pOutputStream)
    {
    }
};

}   // ~Stream

}   // ~HttpJson

}}}
