﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Result.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/nim/srv/nim_FileNetworkInstallTaskMeta.h>

namespace nn { namespace nim { namespace srv {

    Result FileNetworkInstallTaskMeta::Create(const char* path, ncm::ApplicationId appId, const ncm::ContentMetaKey keyList[], int count, ncm::StorageId storage, Bit32 config) NN_NOEXCEPT
    {
        auto header = MakeHeader(appId, count, storage, config);

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Write | fs::OpenMode_AllowAppend));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        NN_RESULT_DO(fs::WriteFile(file, 0, &header, sizeof(header), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
        NN_RESULT_DO(fs::WriteFile(file, sizeof(header), keyList, sizeof(ncm::ContentMetaKey) * count, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }

    Result FileNetworkInstallTaskMeta::Initialize(const char* path) NN_NOEXCEPT
    {
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Read | fs::OpenMode_Write | fs::OpenMode_AllowAppend));
        m_File = file;
        NN_RESULT_DO(fs::ReadFile(*m_File, 0, &m_Header, sizeof(m_Header)));

        NN_RESULT_SUCCESS;
    }

    Result FileNetworkInstallTaskMeta::Add(const ncm::ContentMetaKey keyList[], int count) NN_NOEXCEPT
    {
        bool written = false;
        for (int i = 0; i < count; i++)
        {
            auto& key = keyList[i];

            bool hasId;
            NN_RESULT_DO(HasId(&hasId, key.id));
            if (hasId)
            {
                continue;
            }

            NN_RESULT_DO(fs::WriteFile(*m_File, sizeof(m_Header) + sizeof(ncm::ContentMetaKey) * m_Header.contentMetaCount, &key, sizeof(key), fs::WriteOption::MakeValue(0)));
            m_Header.contentMetaCount++;

            written = true;
        }

        if (written)
        {
            NN_RESULT_DO(fs::WriteFile(*m_File, 0, &m_Header, sizeof(m_Header), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
        }

        NN_RESULT_SUCCESS;
    }

    Result FileNetworkInstallTaskMeta::SetAttribute(Bit64 attribute) NN_NOEXCEPT
    {
        m_Header.attribute = attribute;
        return fs::WriteFile(*m_File, 0, &m_Header, sizeof(m_Header), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush));
    }

    FileNetworkInstallTaskMeta::~FileNetworkInstallTaskMeta() NN_NOEXCEPT
    {
        if (m_File)
        {
            fs::CloseFile(*m_File);
        }
    }

    Result FileNetworkInstallTaskMeta::GetContentMeta(ncm::ContentMetaKey* outValue, int index) NN_NOEXCEPT
    {
        return fs::ReadFile(*m_File, sizeof(m_Header) + sizeof(ncm::ContentMetaKey) * index, outValue, sizeof(*outValue));
    }

    Result FileNetworkInstallTaskMeta::HasId(bool* outValue, Bit64 id) NN_NOEXCEPT
    {
        int count = CountContentMeta();
        for (int i = 0; i < count; i++)
        {
            ncm::ContentMetaKey key;
            NN_RESULT_DO(GetContentMeta(&key, i));
            if (key.id == id)
            {
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        }

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

}}}
