﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>

#include "nim_StringUtil.h"
#include "nim_DynamicRightsRevokeReason.h"

namespace nn { namespace nim { namespace srv {
namespace DynamicRights {

//-----------------------------------------------------------------------------
namespace {
//-----------------------------------------------------------------------------
// デバッグコード
#if !defined(NN_SDK_BUILD_RELEASE)
#define DEBUG_TRACE(...) NN_DETAIL_NIM_TRACE( "[DynamicRights::RevokeReason] " __VA_ARGS__ )
#else
#define DEBUG_TRACE(...)    static_cast<void>(0)
#endif
//-----------------------------------------------------------------------------

//! @brief      レスポンスアダプタ。
//! @details    各要素の型は http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=325012215 に基づきます。
class RevokeReasonAdaptor : public HttpJson::Canceler
{
    typedef HttpJson::EntryLookup<6, 64>                        EntryLookup;
    typedef HttpJson::ValueStore<bool, EntryLookup>             StoreIsRevoked;
    typedef HttpJson::ValueStore<Dragons::Reason, EntryLookup>  StoreReason;
    typedef HttpJson::ValueStore<int64_t, EntryLookup>          StoreDate;

    struct StoreResponse
    {
        StoreIsRevoked  isRevoked;
        StoreReason     reason;
        StoreDate       date;       //!< 未検出時は無期限扱い

        bool CreateLookupPath() NN_NOEXCEPT
        {
            return (
                isRevoked.CreateLookupPath("$.inactivated_reasons[0].is_inactivated") &&
                reason.CreateLookupPath("$.inactivated_reasons[0].inactivated_reason") &&
                date.CreateLookupPath("$.inactivated_reasons[0].inactivated_date")
            );
        }

        void Reset() NN_NOEXCEPT
        {
            isRevoked.Reset(false);
            reason.Reset(Dragons::Reason::Unknown);
            date.Reset();
        }
    };

public:
    typedef AsyncRevokeReasonImpl::ValueType    ValueType;
    typedef EntryLookup::JsonPathType           JsonPathType;

    explicit RevokeReasonAdaptor() NN_NOEXCEPT
    {
        m_Store.Reset();
        NN_ABORT_UNLESS(m_Store.CreateLookupPath());
    }

    //! @brief      内部異常値の取得。
    //!
    //! @details    本クラスインスタンスをキャンセラ登録する事で内部異常が発生した場合には、@ref ::nn::http::json::ImportJsonByRapidJson() の返値に http::ResultCanceled が返ります。@n
    //!             その場合、本メソッドで異常値を取得し適切に処理してください。@n
    //!             異常状態が発生していない場合は、ResultSuccess が返されます。
    Result GetFailure() const NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    ValueType GetRevokeReason() const NN_NOEXCEPT
    {
        return GetRevokeReasonFrom(m_Store.reason.GetValue());
    }

    // パース中に文字列値をもつJSONパスが見つかった時に呼ばれます。
    void Update(const JsonPathType& jsonPath, const char* pValue, int valueLength) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(HttpJson::EntryStoreUtil::FilterIgnorable(
            m_Store.reason.Store(jsonPath, pValue, valueLength,
            [](StoreReason::ValueType* pOut, const char* pValue, int valueLength) NN_NOEXCEPT->Result{
                *pOut = Dragons::ParseReasonFrom(pValue, valueLength);
                NN_RESULT_SUCCESS;
            })
        ));
    }

    // パース中に std::numeric_limit<int64_t>::max 以下の整数値をもつJSONパスが見つかった時に呼ばれます。
    void Update(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(HttpJson::EntryStoreUtil::FilterIgnorable(m_Store.date.Store(jsonPath, value)));
    }

    void Update(const JsonPathType& jsonPath, bool value) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(HttpJson::EntryStoreUtil::FilterIgnorable(m_Store.isRevoked.Store(jsonPath, value)));
    }

    // 以下検出対象外
    void NotifyObjectBegin(const JsonPathType&) NN_NOEXCEPT {}
    void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT {}
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, double) NN_NOEXCEPT {}
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT {}

    // 内部処理で失敗があったらパースをキャンセルする。
    virtual bool IsCancelled() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

private:
    StoreResponse   m_Store;
};

//-----------------------------------------------------------------------------
}   // ~unnamed
//-----------------------------------------------------------------------------

//!============================================================================
//! @name AsyncRevokeReasonImpl 実装
//
// NOTE:
//  処理結果リソース( m_Value )は非同期処理部との排他は行っていません。
//  これは、Shimレイヤの nim::AsyncResult の Get() アクセスにおいて Wait() される事を前提想定した実装です。
//  非同期に Get() を呼び出す要件になった場合は排他が必要です。
//-----------------------------------------------------------------------------
AsyncRevokeReasonImpl::AsyncRevokeReasonImpl() NN_NOEXCEPT
    : AsyncBase(this, GetSharedThreadAllocator()), Executor()
    , m_Value(::nn::nim::RevokeReason::Unknown)
{
}

//-----------------------------------------------------------------------------
AsyncRevokeReasonImpl::~AsyncRevokeReasonImpl() NN_NOEXCEPT
{
    Join();
}

//-----------------------------------------------------------------------------
Result AsyncRevokeReasonImpl::Initialize(DeviceContext* pDeviceContext, const AccountId& accountId, const ELicenseId& eLicenseId) NN_NOEXCEPT
{
    NN_RESULT_DO(Executor::Initialize(pDeviceContext));
    m_ELicenseId = eLicenseId;
    m_AccountId = accountId;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncRevokeReasonImpl::GetSize(::nn::sf::Out<std::uint64_t> outValue) NN_NOEXCEPT
{
    NN_RESULT_DO(Executor::GetResult());

    *outValue = sizeof(ValueType);
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncRevokeReasonImpl::Get(const ::nn::sf::OutBuffer& buffer) const NN_NOEXCEPT
{
    NN_RESULT_DO(Executor::GetResult());

    const size_t requiredSize = sizeof(ValueType);
    NN_RESULT_THROW_UNLESS(requiredSize <= buffer.GetSize(), ResultBufferNotEnough());
    std::memcpy(buffer.GetPointerUnsafe(), &m_Value, requiredSize);
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncRevokeReasonImpl::OnQueryAccessProfile(AccessProfile* pOut, char* pOutPathUrl, size_t availablePathUrlCapacity) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);
    constexpr char base[] = "/v1/elicenses/inactivated_reason";
    NN_ABORT_UNLESS(sizeof(base) <= availablePathUrlCapacity && availablePathUrlCapacity <= static_cast<size_t>(std::numeric_limits<int>::max()));

    const auto capacity = static_cast<int>(availablePathUrlCapacity);
    NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutPathUrl, base, capacity) < capacity);

    pOut->uid = ::nn::account::InvalidUid;
    pOut->naId = m_AccountId;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncRevokeReasonImpl::OnSetupRequestBody(ConnectionType* pConnection) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConnection);

    // ポストデータボディ作成。
    const char PostBase[] =
    "{"
        "\"elicense_ids\": [\"%s\"]"
    "}";

    char data[128];
    char stringForId[ELicenseId::StringSize + 1] = {};
    const size_t postLength = ::nn::util::SNPrintf(data, sizeof(data), PostBase, m_ELicenseId.ToString(stringForId, sizeof(stringForId)));
    NN_ABORT_UNLESS(postLength < sizeof(data));
    NN_RESULT_DO(pConnection->SetTransactionPostFields(data, postLength, true));    // クローンコピーするので CURL 経由でヒープ使います。

    // "Content-Length", "Content-Type" ヘッダ。
    NN_ABORT_UNLESS(::nn::util::SNPrintf(data, sizeof(data), "Content-Length:%zu", postLength) < sizeof(data));
    NN_RESULT_DO(pConnection->SetTransactionHeader(data));
    NN_RESULT_DO(pConnection->SetTransactionHeader("Content-Type:application/json"));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncRevokeReasonImpl::OnResolveResponse(InputStream* pInputStream) NN_NOEXCEPT
{
    // { "key": "value" } 要素は、1024 byte に収まる想定の Insitu パース用ストリーム。
    HttpJson::Stream::ProxyInputStream<InputStream, 1024> stream(pInputStream);

    RevokeReasonAdaptor adaptor;
    NN_RESULT_TRY(::nn::http::json::ImportJsonByRapidJson<::nn::http::json::DefaultJsonErrorMap>(adaptor, stream, static_cast<HttpJson::Canceler*>(&adaptor)))
    NN_RESULT_CATCH_CONVERT(::nn::http::ResultCanceled, adaptor.GetFailure())
    NN_RESULT_END_TRY;

    m_Value = adaptor.GetRevokeReason();
    NN_RESULT_SUCCESS;
}

}   // ~DynamicRights
}}} // ~nn::nim::srv
