﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>

#include "nim_DynamicRightsNotificationToken.h"
#include "nim_StringUtil.h"

namespace nn { namespace nim { namespace srv {
namespace DynamicRights {

//-----------------------------------------------------------------------------
namespace {
//-----------------------------------------------------------------------------
// デバッグコード
#if !defined(NN_SDK_BUILD_RELEASE)
#define DEBUG_TRACE(...) NN_DETAIL_NIM_TRACE( "[DynamicRights::NotificationToken] " __VA_ARGS__ )
#else
#define DEBUG_TRACE(...)    static_cast<void>(0)
#endif
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
}   // ~unnamed
//-----------------------------------------------------------------------------

//!============================================================================
//! @name AsyncRegisterNotificationTokenImpl 実装
//
// NOTE:
//  Shimレイヤの nim::AsyncResult の Get() アクセスにおいて Wait() される事を前提想定した実装です。
//  非同期に Get() を呼び出す要件になった場合は排他が必要です。
//-----------------------------------------------------------------------------
AsyncRegisterNotificationTokenImpl::AsyncRegisterNotificationTokenImpl() NN_NOEXCEPT
    : AsyncBase(this, GetSharedThreadAllocator()), Executor()
{
}

//-----------------------------------------------------------------------------
AsyncRegisterNotificationTokenImpl::~AsyncRegisterNotificationTokenImpl() NN_NOEXCEPT
{
    Join();
}

//-----------------------------------------------------------------------------
Result AsyncRegisterNotificationTokenImpl::Initialize(DeviceContext* pDeviceContext, const ::nn::npns::NotificationToken& notificationToken) NN_NOEXCEPT
{
    NN_RESULT_DO(Executor::Initialize(pDeviceContext));
    m_Token = notificationToken;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncRegisterNotificationTokenImpl::OnQueryAccessProfile(AccessProfile* pOut, char* pOutPathUrl, size_t availablePathUrlCapacity) NN_NOEXCEPT
{
    // AccessProfile の各メンバは初期値設定されており、本 API は初期値のままで良い。
    NN_UNUSED(pOut);
    NN_SDK_ASSERT_NOT_NULL(pOut);
    constexpr char base[] = "/v1/notification_token";
    NN_ABORT_UNLESS(sizeof(base) <= availablePathUrlCapacity && availablePathUrlCapacity <= static_cast<size_t>(std::numeric_limits<int>::max()));

    const auto capacity = static_cast<int>(availablePathUrlCapacity);
    NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutPathUrl, base, capacity) < capacity);
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncRegisterNotificationTokenImpl::OnSetupRequestBody(ConnectionType* pConnection) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConnection);

    // ポストデータボディ作成。
    // npns::NotificationToken::data[] が null 終端されてる想定。( 参考: nim_AsyncNotificationTokenImpl.cpp )
    const char PostBase[] = "{\"notification_token\":\"%s\"}";
    char data[sizeof(PostBase) + sizeof(::nn::npns::NotificationToken)];
    const size_t postLength = ::nn::util::SNPrintf(data, sizeof(data), PostBase, m_Token.data);
    NN_ABORT_UNLESS(postLength < sizeof(data));

    DEBUG_TRACE("Post filed generate complete, (length: %zu)=>`%s`\n", postLength, data);
    NN_SDK_ASSERT(postLength == std::strlen(data)); // NOTE: デバッグ用の保険検証

    // クローンコピーするので CURL 経由でヒープ使います。( ヘッダでも slist でヒープ使う )
    NN_RESULT_DO(pConnection->SetTransactionPostFields(data, postLength, true));

    // "Content-Length", "Content-Type" ヘッダ。
    NN_ABORT_UNLESS(::nn::util::SNPrintf(data, sizeof(data), "Content-Length:%zu", postLength) < sizeof(data));
    NN_RESULT_DO(pConnection->SetTransactionHeader(data));
    NN_RESULT_DO(pConnection->SetTransactionHeader("Content-Type:application/json"));
    NN_RESULT_DO(pConnection->SetTransactionCustomMethod("PUT"));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncRegisterNotificationTokenImpl::OnResolveResponse(InputStream* pInputStream) NN_NOEXCEPT
{
    // 解析すべきレスポンスストリームはなし。
    NN_UNUSED(pInputStream);
    NN_RESULT_SUCCESS;
}

}   // ~DynamicRights
}}} // ~nn::nim::srv
