﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/nim/srv/nim_DeviceContext.h>
#include <nn/nsd/nsd_ApiForMenu.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/factory/settings_DeviceCertificate.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/spl/spl_Api.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include "nim_DebugUtil.h"
#include "nim_DeviceCertificate.h"
#include "nim_StringUtil.h"

namespace nn { namespace nim { namespace srv {
    namespace {
        Result GetEciDeviceCertificate(EciDeviceCertificate* outValue) NN_NOEXCEPT
        {
#if defined NN_BUILD_CONFIG_OS_WIN
            auto result = GetEciDeviceCertificateDebug(outValue);
            if (result.IsFailure())
            {
                NN_DETAIL_NIM_TRACE("[DeviceContext] Failed to get debug eci device cert. Use 0 instead\n");
                std::memset(outValue, 0, sizeof(*outValue));
            }
            NN_RESULT_SUCCESS;
#else
            NN_RESULT_DO(settings::factory::GetEciDeviceCertificate(
                reinterpret_cast<settings::factory::EccB233DeviceCertificate*>(outValue)));

            NN_RESULT_SUCCESS;
#endif
        }

        const char* GetPlatformString() NN_NOEXCEPT
        {
#if defined NN_BUILD_CONFIG_SPEC_GENERIC
            return "Generic";
#elif defined NN_BUILD_CONFIG_SPEC_NX
            return "NX";
#else
            return "Unknown";
#endif
        }
    }

    DeviceContext::~DeviceContext() NN_NOEXCEPT
    {
        if (m_SslContext)
        {
            m_SslContext->Destroy();
        }
    }

    Result DeviceContext::Initialize() NN_NOEXCEPT
    {
        EciDeviceCertificate cert;
        NN_RESULT_DO(GetEciDeviceCertificate(&cert));
        char deviceIdString[32] = {};
        std::memcpy(deviceIdString, cert.name.id, sizeof(cert.name.id));
        m_DeviceId = NN_NIM_STR_TO_ULL(deviceIdString, nullptr, 16);

        NN_DETAIL_NIM_TRACE("[DeviceContext] Device ID 0x%016llx\n", m_DeviceId);

#if defined NN_BUILD_CONFIG_OS_HORIZON
        NN_RESULT_THROW_UNLESS((m_DeviceId & 0x00FFFFFFFFFFFFFF) == spl::GetDeviceIdLow(), ResultInvalidEciDeviceCertificateId());
#endif

        m_SslContext.emplace();
        bool isSslContextCreated = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (!isSslContextCreated)
            {
                m_SslContext = util::nullopt;
            }
        };

        NN_RESULT_DO(m_SslContext->Create(ssl::Context::SslVersion_Auto));
        isSslContextCreated = true;

        ssl::CertStoreId certStoreId;
        auto result = m_SslContext->RegisterInternalPki(&certStoreId, ssl::Context::InternalPki_DeviceClientCertDefault);
        if (result.IsFailure())
        {
            NN_DETAIL_NIM_WARN("[DeviceContext] Failed to register internal device client certificate\n");
        }

        {
            settings::system::FirmwareVersion firmwareVersion;
            settings::system::GetFirmwareVersion(&firmwareVersion);


            nsd::EnvironmentIdentifier eid;
            nsd::GetEnvironmentIdentifier(&eid);

            // ユーザーエージェントを変更する場合は、以下の[ユーザータイトル配信／仕様／ユーザーエージェント]の内容も修正してください。
            // https://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=330988844
            util::SNPrintf(m_PrivateUserAgentHeader, sizeof(m_PrivateUserAgentHeader),
                "User-Agent: NintendoSDK Firmware/%s-%u.%u (platform:%s; did:%016llx; eid:%s)",
                firmwareVersion.displayVersion, firmwareVersion.majorRelstep, firmwareVersion.minorRelstep,
                GetPlatformString(), m_DeviceId, eid.value);

            util::SNPrintf(m_PublicUserAgentHeader, sizeof(m_PublicUserAgentHeader),
                "User-Agent: NintendoSDK Firmware/%s-%u.%u (platform:%s; eid:%s)",
                firmwareVersion.displayVersion, firmwareVersion.majorRelstep, firmwareVersion.minorRelstep,
                GetPlatformString(), eid.value);
        }

        NN_DETAIL_NIM_TRACE("[DeviceContext] Private %s\n", m_PrivateUserAgentHeader);
        NN_DETAIL_NIM_TRACE("[DeviceContext] Public  %s\n", m_PublicUserAgentHeader);

        NN_RESULT_SUCCESS;
    }
}}}
