﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/crypto.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_ThreadAllocator.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringView.h>
#include "nim_AsyncTemporaryFileBase.h"
#include "nim_SystemUpdateMetaUtil.h"

namespace nn { namespace nim { namespace srv {
    namespace {
        void ParseETag(ETag* out, const void* header, size_t headerSize) NN_NOEXCEPT
        {
            nn::util::string_view sv(reinterpret_cast<const char*>(header), headerSize);
            auto etagIndex = sv.find("ETag:");
            if (etagIndex == nn::util::string_view::npos)
            {
                return;
            }

            auto etagValueIndex = sv.find("\"", etagIndex + 5);
            if (etagValueIndex == nn::util::string_view::npos)
            {
                return;
            }

            auto etagValueEndIndex = sv.find("\"", etagValueIndex + 1);
            if (etagValueEndIndex == nn::util::string_view::npos)
            {
                return;
            }

            auto length = etagValueEndIndex - etagValueIndex - 1;
            std::memcpy(out->data, &sv.data()[etagValueIndex + 1], std::min(sizeof(out->data), length));
            out->data[length] = '\0';
        }

    } // namespace

    AsyncTemporaryFileBase::AsyncTemporaryFileBase(ThreadAllocator* threadAllocator) NN_NOEXCEPT : AsyncResultImpl<AsyncTemporaryFileBase>(this, threadAllocator), m_IsDownloaded(), m_ETag({}) {}

    AsyncTemporaryFileBase::~AsyncTemporaryFileBase() NN_NOEXCEPT
    {
        AsyncResultImpl<AsyncTemporaryFileBase>::Join();

        if (m_File)
        {
            fs::FlushFile(*m_File);
            fs::CloseFile(*m_File);
            fs::DeleteFile(m_Path);
        }
    }

    Result AsyncTemporaryFileBase::Initialize(DeviceContext* deviceContext, const Url& url, const char* path, const ETag& eTag) NN_NOEXCEPT
    {
        m_DeviceId = deviceContext->GetDeviceId();
        NN_RESULT_DO(m_Connection.Initialize(deviceContext));
        NN_RESULT_DO(fs::CreateFile(path, 0));
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Read | fs::OpenMode_Write | fs::OpenMode_AllowAppend));
        m_Url = url;
        m_Path.Assign(path);
        m_File = file;
        m_ETag = eTag;
        NN_RESULT_SUCCESS;
    }

    Result AsyncTemporaryFileBase::Execute() NN_NOEXCEPT
    {
        // ncm で発生する fs アクセスの優先度を設定する
        // 注意:    IPC 先で fs アクセスが発生するときに影響するので、
        //          他のプロセスへの IPC を行うときには注意
        fs::SetPriorityRawOnCurrentThread(fs::PriorityRaw_Background);
        NN_RESULT_DO(CheckSystemUpdateMetaForContentDelivery(&m_Connection, m_DeviceId));

        const char* headerList[2];
        const char* acceptHeader("Accept:application/json");
        char ifNoneMatchHeader[64];

        int headerCount = 1;
        headerList[0] = acceptHeader;

        if (!m_ETag.IsEmpty())
        {
            util::SNPrintf(ifNoneMatchHeader, sizeof(ifNoneMatchHeader), "If-None-Match: \"%s\"", m_ETag.data);
            headerList[1] = ifNoneMatchHeader;
            headerCount++;
        }

        size_t written = 0;
        ETag& eTag = m_ETag;
        NN_RESULT_DO(m_Connection.Get(m_Url,
            [&eTag](const void* buffer, size_t size) -> Result
            {
                ParseETag(&eTag, buffer, size);
                NN_RESULT_SUCCESS;
            },
            [this, &written](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_DO(fs::WriteFile(*m_File, written, buffer, size, fs::WriteOption()));
                written += size;

                NN_RESULT_SUCCESS;
            },
            headerList, headerCount));

        m_IsDownloaded = true;

        NN_RESULT_SUCCESS;
    }

    Result AsyncTemporaryFileBase::Cancel() NN_NOEXCEPT
    {
        m_Connection.Cancel();
        NN_RESULT_SUCCESS;
    }

    Result AsyncTemporaryFileBase::GetSize(sf::Out<int64_t> outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_IsDownloaded, ResultTemporaryFileNotDownloaded());
        return fs::GetFileSize(outValue.GetPointer(), *m_File);
    }

    Result AsyncTemporaryFileBase::Read(sf::Out<uint64_t> outValue, int64_t offset, sf::OutBuffer outBuffer) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_IsDownloaded, ResultTemporaryFileNotDownloaded());
        size_t read;
        NN_RESULT_DO(fs::ReadFile(&read, *m_File, offset, outBuffer.GetPointerUnsafe(), outBuffer.GetSize()));
        *outValue = static_cast<uint64_t>(read);
        NN_RESULT_SUCCESS;
    }

    Result AsyncTemporaryFileBase::GetETag(nn::sf::Out<nn::nim::ETag> outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_IsDownloaded, ResultTemporaryFileNotDownloaded());
        std::memcpy(outValue.GetPointer(), &m_ETag, sizeof(m_ETag));
        NN_RESULT_SUCCESS;
    }

}}}
