﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/account/account_CachedNintendoAccountInfo.h>
#include <nn/account/account_Result.h>
#include <nn/ec/ec_Result.h>
#include <nn/ec/system/ec_DeviceAuthenticationTypes.h>
#include <nn/es/es_Api.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/nim/srv/nim_HttpConnection.h>
#include <nn/nim/srv/nim_NetworkInstallUrl.h>
#include <nn/nsd/nsd_ApiForNasService.h>
#include <nn/util/util_TFormatString.h>

#include "nim_AsyncDeviceAuthenticationTokenImpl.h"
#include "nim_AsyncDeviceLinkImpl.h"
#include "nim_EcSystemThreadAllocator.h"
#include "nim_EShopUtil.h"
#include "nim_Json.h"

namespace nn { namespace nim { namespace srv {
    namespace {
        // デバイスリンクスレッドが１つであることに依存している
        NN_ALIGNAS(4096) char g_RequestWork[4096];

        struct NintendoAccountAuthorizationCode
        {
            char data[account::NintendoAccountAuthorizationCodeLengthMax + 1];
        };

        struct NintendoAccountIdToken
        {
            char data[account::NintendoAccountIdTokenLengthMax + 1];
        };

        Result RequestDeviceLinkImpl(char* readBuffer, size_t readBufferSize, HttpConnection* connection,
            const Url& url, const char* postData, const char* headerList[], int headerCount) NN_NOEXCEPT
        {
            size_t readSize = 0;
            auto writeCallback = [&readBuffer, &readSize, readBufferSize](const void* buffer, size_t bufferSize) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + bufferSize < readBufferSize, ResultUnexpectedResponseDeviceAuthenticationTokenTooLong());
                std::memcpy(&readBuffer[readSize], buffer, bufferSize);
                readSize += bufferSize;
                NN_RESULT_SUCCESS;
            };

            auto result = postData ?
                connection->Post(url, postData, writeCallback, headerList, headerCount) :
                connection->Get(url, writeCallback, headerList, headerCount);
            readBuffer[readSize] = '\0';
            NN_DETAIL_NIM_TRACE("[RequestDeviceLinkImpl] response %s\n", readBuffer);

            if (result <= ResultHttpStatus())
            {
                NN_RESULT_TRY(ToEShopResultResponseDestructively(readBuffer))
                    NN_RESULT_CATCH(ResultInvalidJson) { NN_RESULT_THROW(result); }
                NN_RESULT_END_TRY
            }
            NN_RESULT_DO(result);

            NN_RESULT_SUCCESS;
        }

        Result ParseVirtualAccountResponse(ec::system::VirtualAccountId* outValue, char* readBuffer) NN_NOEXCEPT
        {
            nne::rapidjson::Document document;
            NN_RESULT_THROW_UNLESS(!document.ParseInsitu(readBuffer).HasParseError(), ResultUnexpectedResponseVirtualAccountParseError());

            auto idObj = document.FindMember("id");
            NN_RESULT_THROW_UNLESS(idObj != document.MemberEnd() && idObj->value.IsString(), ResultUnexpectedResponseVirtualAccountParseError());

            char* endPtr;
            auto vaId = std::strtoul(idObj->value.GetString(), &endPtr, 10);
            NN_RESULT_THROW_UNLESS(*endPtr == '\0', ResultUnexpectedResponseVirtualAccountParseError());

            *outValue = vaId;
            NN_RESULT_SUCCESS;
        }

        Result GetVirtualAccountId(ec::system::VirtualAccountId* outValue,
            HttpConnection* connection,
            const NintendoAccountIdToken& naIdToken) NN_NOEXCEPT
        {
            Url url;
            MakeVirtualAccountUrl(&url);

            char naIdTokenHeader[128 + sizeof(naIdToken.data)];
            util::TSNPrintf(naIdTokenHeader, sizeof(naIdTokenHeader), "Authorization: Bearer %s", naIdToken.data);

            const char* HeaderList[] = {
                "Accept: application/json",
                "Content-Type: application/json",
                naIdTokenHeader
            };

            char buffer[1024];
            NN_RESULT_DO(RequestDeviceLinkImpl(buffer, sizeof(buffer), connection, url, nullptr, HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])));

            ec::system::VirtualAccountId vaId;
            NN_RESULT_DO(ParseVirtualAccountResponse(&vaId, buffer));

            *outValue = vaId;
            NN_RESULT_SUCCESS;
        }

        Result UnlinkDevice(HttpConnection* connection,
            account::NintendoAccountId naUserId,
            const ec::system::DeviceAuthenticationToken& daToken) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(naUserId);

            Url url;
            MakeUnlinkDeviceUrl(&url);

            char authTokenHeader[128 + sizeof(daToken.data)];
            util::TSNPrintf(authTokenHeader, sizeof(authTokenHeader), "X-DeviceAuthorization: Bearer %s", daToken.data);

            const char* HeaderList[] = {
                "Accept: application/json",
                "Content-Type: application/x-www-form-urlencoded",
                authTokenHeader,
            };

            char naUserIdBody[64];
            util::TSNPrintf(naUserIdBody, sizeof(naUserIdBody), "na_user_id=%016llx", naUserId);

            char buffer[1024];
            NN_RESULT_DO(RequestDeviceLinkImpl(buffer, sizeof(buffer), connection, url, naUserIdBody, HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])));

            NN_RESULT_SUCCESS;
        }

        Result UnlinkDeviceAll(HttpConnection* connection,
            Bit64 deviceId,
            const ec::system::DeviceAuthenticationToken& daToken) NN_NOEXCEPT
        {
            Url url;
            MakeUnlinkDeviceAllUrl(&url, deviceId);

            char authTokenHeader[128 + sizeof(daToken.data)];
            util::TSNPrintf(authTokenHeader, sizeof(authTokenHeader), "X-DeviceAuthorization: Bearer %s", daToken.data);

            const char* HeaderList[] = {
                "Accept: application/json",
                "Content-Type: application/x-www-form-urlencoded",
                authTokenHeader,
            };

            char buffer[1024];
            NN_RESULT_DO(RequestDeviceLinkImpl(buffer, sizeof(buffer), connection, url, "", HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])));

            NN_RESULT_SUCCESS;
        }

        Result LinkDevice(HttpConnection* connection,
            const NintendoAccountIdToken& naIdToken,
            const ec::system::DeviceAuthenticationToken& daToken) NN_NOEXCEPT
        {
            Url url;
            MakeLinkDeviceUrl(&url);

            char authTokenHeader[128 + sizeof(daToken.data)];
            util::TSNPrintf(authTokenHeader, sizeof(authTokenHeader), "X-DeviceAuthorization: Bearer %s", daToken.data);

            char naIdTokenHeader[128 + sizeof(naIdToken.data)];
            util::TSNPrintf(naIdTokenHeader, sizeof(naIdTokenHeader), "Authorization: Bearer %s", naIdToken.data);

            const char* HeaderList[] = {
                "Accept: application/json",
                "Content-Type: application/json",
                authTokenHeader,
                naIdTokenHeader
            };

            char buffer[1024];
            NN_RESULT_DO(RequestDeviceLinkImpl(buffer, sizeof(buffer), connection, url, "", HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])));

            NN_RESULT_SUCCESS;
        }

        Result CreateVirtualAccount(HttpConnection* connection,
            const NintendoAccountAuthorizationCode& authCode) NN_NOEXCEPT
        {
            nsd::NasServiceSetting shopSetting;
            NN_RESULT_DO(nsd::GetNasServiceSetting(&shopSetting, nsd::NasServiceNameOfNxShop));

            char redirectUrl[nsd::NasServiceSetting::RedirectUri::Size];
            std::strcpy(redirectUrl, shopSetting.redirectUri.value);
            NN_RESULT_DO(HttpConnection::Escape(redirectUrl, sizeof(redirectUrl), redirectUrl));

            Url url;
            MakeCreateVirtualAccountUrl(&url);

            char postFields[1024];
            util::TSNPrintf(postFields, sizeof(postFields), "auth_code=%s&redirect_url=%s", authCode.data, redirectUrl);

            const char* HeaderList[] = {
                "Accept: application/json",
                "Content-Type: application/x-www-form-urlencoded",
            };

            char buffer[1024];
            NN_RESULT_DO(RequestDeviceLinkImpl(buffer, sizeof(buffer), connection, url, postFields, HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])));

            NN_RESULT_SUCCESS;
        }

        Result ParseDeviceLinkStatusResponse(ec::system::DeviceLinkStatus* outValue, char* buffer) NN_NOEXCEPT
        {
            nne::rapidjson::Document document;
            if (document.ParseInsitu(buffer).HasParseError())
            {
                NN_RESULT_THROW(nim::ResultInvalidJson());
            }

            if (document.Empty())
            {
                outValue->hasDeviceLink = false;
                NN_RESULT_SUCCESS;
            }

            auto itr = document.Begin();
            auto idObj = itr->FindMember("id");
            NN_RESULT_THROW_UNLESS(idObj != itr->MemberEnd(), nim::ResultInvalidJson());

            char* endPtr;
            uint64_t deviceId = std::strtol(idObj->value.GetString(), &endPtr, 16);
            NN_RESULT_THROW_UNLESS(*endPtr == '\0', nim::ResultInvalidJson());

            outValue->hasDeviceLink = true;
            outValue->linkedDeviceId = deviceId;
            NN_RESULT_SUCCESS;
        }

        Result DeviceLinkStatus(ec::system::DeviceLinkStatus* outValue,
            HttpConnection* connection,
            const NintendoAccountIdToken& naIdToken) NN_NOEXCEPT
        {
            Url url;
            MakeDeviceLinkStatusUrl(&url);

            char naIdTokenHeader[128 + sizeof(naIdToken.data)];
            util::TSNPrintf(naIdTokenHeader, sizeof(naIdTokenHeader), "Authorization: Bearer %s", naIdToken.data);

            const char* HeaderList[] = {
                "Accept: application/json",
                naIdTokenHeader
            };

            char buffer[1024];
            NN_RESULT_DO(RequestDeviceLinkImpl(buffer, sizeof(buffer), connection, url, nullptr, HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])));
            NN_RESULT_DO(ParseDeviceLinkStatusResponse(outValue, buffer));

            NN_RESULT_SUCCESS;
        }

    } // namespace

    AsyncUnlinkDeviceImpl::AsyncUnlinkDeviceImpl() NN_NOEXCEPT : AsyncResultImpl<AsyncUnlinkDeviceImpl>(this, &GetEcSystemThreadAllocator()), m_Uid(account::InvalidUid), m_VaStore() {}

    Result AsyncUnlinkDeviceImpl::Initialize(DeviceContext* deviceContext, VirtualAccountStore* vaStore, account::Uid uid) NN_NOEXCEPT
    {
        ec::system::VirtualAccountId vaId;
        NN_RESULT_THROW_UNLESS(vaStore->Get(&vaId, uid).IsSuccess(), ec::ResultAccountNotDeviceLinked());

        account::NetworkServiceAccountManager manager;
        NN_RESULT_DO(account::GetNetworkServiceAccountManager(&manager, uid));
        NN_RESULT_TRY(manager.GetNintendoAccountId(&m_NaUserId))
            NN_RESULT_CATCH(account::ResultNetworkServiceAccountUnavailable)
            {
                // NA ID を取得できない場合は機器認証解除をスキップしチケットの削除のみ行う
                NN_DETAIL_NIM_TRACE("[AsyncUnlinkDeviceImpl] Nintendo Account ID unavailable. Skip communication with the server.\n");
                m_NaUserId = account::InvalidNintendoAccountId;
            }
        NN_RESULT_END_TRY
        NN_RESULT_DO(m_HttpConnection.Initialize(deviceContext));

        m_Uid = uid;
        m_VaId = vaId;
        m_VaStore = vaStore;
        NN_RESULT_SUCCESS;
    }

    Result AsyncUnlinkDeviceImpl::Initialize(DeviceContext* deviceContext, account::NintendoAccountId naUserId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(naUserId);
        m_NaUserId = naUserId;
        NN_RESULT_DO(m_HttpConnection.Initialize(deviceContext));
        NN_RESULT_SUCCESS;
    }

    Result AsyncUnlinkDeviceImpl::Execute() NN_NOEXCEPT
    {
        // Unlink が絶妙なタイミングでキャンセルされて、サーバ上は
        // Unlink しているがチケットが消えていないという状況を防ぐ
        // ために、チケットを先に消してから Unlink する

        // 過去の API 動作を残すための処置
        if (m_VaStore)
        {
            es::DeletePersonalizedTicket(m_VaId);
        }

        if (m_NaUserId)
        {
            ec::system::DeviceAuthenticationToken daToken = {};
            NN_RESULT_DO(GetShopDeviceAuthenticationToken(&daToken, &m_Cancelable));
            NN_RESULT_DO(UnlinkDevice(&m_HttpConnection, m_NaUserId, daToken));
        }

        // 過去の API 動作を残すための処置
        if (m_VaStore)
        {
            NN_RESULT_DO(m_VaStore->Unregister(m_Uid));
        }

        NN_RESULT_SUCCESS;
    }

    Result AsyncUnlinkDeviceImpl::Cancel() NN_NOEXCEPT
    {
        if (m_NaUserId)
        {
            m_HttpConnection.Cancel();
            m_Cancelable.Cancel();
        }
        NN_RESULT_SUCCESS;
    }

    AsyncUnlinkDeviceAllImpl::AsyncUnlinkDeviceAllImpl() NN_NOEXCEPT : AsyncResultImpl<AsyncUnlinkDeviceAllImpl>(this, &GetEcSystemThreadAllocator()), m_VaStore() {}

    Result AsyncUnlinkDeviceAllImpl::Initialize(DeviceContext* deviceContext, VirtualAccountStore* vaStore) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_HttpConnection.Initialize(deviceContext));
        m_DeviceId = deviceContext->GetDeviceId();

        m_VaStore = vaStore;
        NN_RESULT_SUCCESS;
    }

    Result AsyncUnlinkDeviceAllImpl::Execute() NN_NOEXCEPT
    {
        ec::system::DeviceAuthenticationToken daToken = {};
        NN_RESULT_DO(GetShopDeviceAuthenticationToken(&daToken, &m_Cancelable));

        NN_RESULT_DO(UnlinkDeviceAll(&m_HttpConnection, m_DeviceId, daToken));
        m_VaStore->Clear();

        NN_RESULT_SUCCESS;
    }

    Result AsyncUnlinkDeviceAllImpl::Cancel() NN_NOEXCEPT
    {
        m_HttpConnection.Cancel();
        m_Cancelable.Cancel();
        NN_RESULT_SUCCESS;
    }

    AsyncLinkDeviceImpl::AsyncLinkDeviceImpl() NN_NOEXCEPT : AsyncResultImpl<AsyncLinkDeviceImpl>(this, &GetEcSystemThreadAllocator()), m_VaStore() {}

    Result AsyncLinkDeviceImpl::Initialize(DeviceContext* deviceContext, VirtualAccountStore* vaStore, account::Uid uid) NN_NOEXCEPT
    {
        account::NetworkServiceAccountManager manager;
        NN_RESULT_DO(account::GetNetworkServiceAccountManager(&manager, uid));
        account::NintendoAccountId naId;
        NN_RESULT_TRY(manager.GetNintendoAccountId(&naId))
            NN_RESULT_CATCH_CONVERT(account::ResultNetworkServiceAccountUnavailable, ec::ResultNintendoAccountNotLinked())
        NN_RESULT_END_TRY
        NN_RESULT_DO(m_HttpConnection.Initialize(deviceContext));

        nsd::NasServiceSetting shopSetting;
        NN_RESULT_DO(nsd::GetNasServiceSetting(&shopSetting, nsd::NasServiceNameOfNxShop));

        account::NintendoAccountAuthorizationRequestParameters params = {
            "nxeshop",
            " ",
            " "
        };

        NN_RESULT_DO(manager.CreateNintendoAccountAuthorizationRequest(
            &m_AuthRequest,
            shopSetting.clientId,
            shopSetting.redirectUri.value,
            nsd::NasServiceSetting::RedirectUri::Size,
            params,
            g_RequestWork,
            sizeof(g_RequestWork)));

        m_Uid = uid;
        m_VaStore = vaStore;
        NN_RESULT_SUCCESS;
    }

    Result AsyncLinkDeviceImpl::Execute() NN_NOEXCEPT
    {
        NintendoAccountIdToken naIdToken;
        {
            os::SystemEvent event;
            NN_RESULT_DO(m_AuthRequest.GetSystemEvent(&event));
            event.Wait();
            NN_RESULT_DO(m_AuthRequest.GetResult());
            size_t tokenSize;
            NN_RESULT_DO(m_AuthRequest.GetIdToken(&tokenSize, naIdToken.data, sizeof(naIdToken.data) - 1));
            naIdToken.data[tokenSize] = '\0';
        }

        ec::system::VirtualAccountId vaId;
        NN_RESULT_DO(GetVirtualAccountId(&vaId, &m_HttpConnection, naIdToken));
        NN_RESULT_DO(m_VaStore->Register(m_Uid, vaId));

        ec::system::DeviceAuthenticationToken daToken = {};
        NN_RESULT_DO(GetShopDeviceAuthenticationToken(&daToken, &m_Cancelable));
        NN_RESULT_DO(LinkDevice(&m_HttpConnection, naIdToken, daToken));

        NN_RESULT_SUCCESS;
    }

    Result AsyncLinkDeviceImpl::Cancel() NN_NOEXCEPT
    {
        NN_RESULT_DO(m_AuthRequest.Cancel());
        m_HttpConnection.Cancel();
        m_Cancelable.Cancel();
        NN_RESULT_SUCCESS;
    }

    AsyncCreateVirtualAccountImpl::AsyncCreateVirtualAccountImpl() NN_NOEXCEPT : AsyncResultImpl<AsyncCreateVirtualAccountImpl>(this, &GetEcSystemThreadAllocator()) {}

    Result AsyncCreateVirtualAccountImpl::Initialize(DeviceContext* deviceContext, account::Uid uid) NN_NOEXCEPT
    {
        account::NetworkServiceAccountManager manager;
        NN_RESULT_DO(account::GetNetworkServiceAccountManager(&manager, uid));
        account::NintendoAccountId naId;
        NN_RESULT_TRY(manager.GetNintendoAccountId(&naId))
            NN_RESULT_CATCH_CONVERT(account::ResultNetworkServiceAccountUnavailable, ec::ResultNintendoAccountNotLinked())
        NN_RESULT_END_TRY

        NN_RESULT_DO(manager.LoadCachedNintendoAccountInfo(&m_AccountInfo, g_RequestWork, sizeof(g_RequestWork)));

        NN_RESULT_DO(m_HttpConnection.Initialize(deviceContext));

        nsd::NasServiceSetting shopSetting;
        NN_RESULT_DO(nsd::GetNasServiceSetting(&shopSetting, nsd::NasServiceNameOfNxShop));

        account::NintendoAccountAuthorizationRequestParameters params = {
            "nxeshop",
            " ",
            " "
        };

        NN_RESULT_DO(manager.CreateNintendoAccountAuthorizationRequest(
            &m_AuthRequest,
            shopSetting.clientId,
            shopSetting.redirectUri.value,
            nsd::NasServiceSetting::RedirectUri::Size,
            params,
            g_RequestWork,
            sizeof(g_RequestWork)));

        m_Uid = uid;
        NN_RESULT_SUCCESS;
    }

    Result AsyncCreateVirtualAccountImpl::Execute() NN_NOEXCEPT
    {
        NintendoAccountAuthorizationCode authCode;
        {
            os::SystemEvent event;
            NN_RESULT_DO(m_AuthRequest.GetSystemEvent(&event));
            event.Wait();
            NN_RESULT_DO(m_AuthRequest.GetResult());
            size_t authCodeSize;
            NN_RESULT_DO(m_AuthRequest.GetAuthorizationCode(&authCodeSize, authCode.data, sizeof(authCode.data) - 1));
            authCode.data[authCodeSize] = '\0';
        }

        NN_RESULT_DO(CreateVirtualAccount(&m_HttpConnection, authCode));

        NN_RESULT_SUCCESS;
    }

    Result AsyncCreateVirtualAccountImpl::Cancel() NN_NOEXCEPT
    {
        m_HttpConnection.Cancel();
        NN_RESULT_SUCCESS;
    }

    AsyncDeviceLinkStatusImpl::AsyncDeviceLinkStatusImpl() NN_NOEXCEPT : AsyncValueImpl<AsyncDeviceLinkStatusImpl, ec::system::DeviceLinkStatus>(this, &GetEcSystemThreadAllocator()) {}

    Result AsyncDeviceLinkStatusImpl::Initialize(DeviceContext* deviceContext, account::Uid uid) NN_NOEXCEPT
    {
        account::NetworkServiceAccountManager manager;
        NN_RESULT_DO(account::GetNetworkServiceAccountManager(&manager, uid));
        account::NintendoAccountId naId;
        NN_RESULT_TRY(manager.GetNintendoAccountId(&naId))
            NN_RESULT_CATCH_CONVERT(account::ResultNetworkServiceAccountUnavailable, ec::ResultNintendoAccountNotLinked())
        NN_RESULT_END_TRY
        NN_RESULT_DO(m_HttpConnection.Initialize(deviceContext));

        nsd::NasServiceSetting shopSetting;
        NN_RESULT_DO(nsd::GetNasServiceSetting(&shopSetting, nsd::NasServiceNameOfNxShop));

        account::NintendoAccountAuthorizationRequestParameters params = {
            "nxeshop",
            " ",
            " "
        };

        NN_RESULT_DO(manager.CreateNintendoAccountAuthorizationRequest(
            &m_AuthRequest,
            shopSetting.clientId,
            shopSetting.redirectUri.value,
            nsd::NasServiceSetting::RedirectUri::Size,
            params,
            g_RequestWork,
            sizeof(g_RequestWork)));

        m_Uid = uid;
        NN_RESULT_SUCCESS;
    }

    Result AsyncDeviceLinkStatusImpl::ExecuteAndValue(ec::system::DeviceLinkStatus* outValue) NN_NOEXCEPT
    {
        NintendoAccountIdToken idToken;
        {
            os::SystemEvent event;
            NN_RESULT_DO(m_AuthRequest.GetSystemEvent(&event));
            event.Wait();
            NN_RESULT_DO(m_AuthRequest.GetResult());
            size_t idTokenSize;
            NN_RESULT_DO(m_AuthRequest.GetIdToken(&idTokenSize, idToken.data, sizeof(idToken.data) - 1));
            idToken.data[idTokenSize] = '\0';
        }

        NN_RESULT_DO(DeviceLinkStatus(outValue, &m_HttpConnection, idToken));

        NN_RESULT_SUCCESS;
    }

    Result AsyncDeviceLinkStatusImpl::CancelImpl() NN_NOEXCEPT
    {
        m_HttpConnection.Cancel();
        NN_RESULT_SUCCESS;
    }
}}}
