﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/nim/srv/nim_NetworkInstallUrl.h>
#include <nn/os/os_TickApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SystemThreadDefinition.h>
#include "nim_Json.h"
#include "nim_AsyncDeviceAuthenticationTokenImpl.h"
#include "nim_EShopUtil.h"

namespace nn { namespace nim { namespace srv {

    namespace
    {
        static const int DeviceAuthenticationTokenThreadCount = 1;
        static const size_t StackSize = 16 * 1024;

        os::ThreadType g_DeviceAuthenticationTokenTaskThreadList[DeviceAuthenticationTokenThreadCount];
        NN_OS_ALIGNAS_THREAD_STACK char g_DeviceAuthenticationTokenTaskStack[StackSize * DeviceAuthenticationTokenThreadCount];

        ThreadAllocator g_DeviceAuthenticationTokenTaskThreadAllocator(g_DeviceAuthenticationTokenTaskThreadList, DeviceAuthenticationTokenThreadCount, NN_SYSTEM_THREAD_PRIORITY(nim, DeviceAuthenticationTokenTask), g_DeviceAuthenticationTokenTaskStack, sizeof(g_DeviceAuthenticationTokenTaskStack), StackSize, NN_SYSTEM_THREAD_NAME(nim, DeviceAuthenticationTokenTask));
    }

    AsyncDeviceAuthenticationTokenImpl::AsyncDeviceAuthenticationTokenImpl() NN_NOEXCEPT :
    AsyncValueImpl<AsyncDeviceAuthenticationTokenImpl, ec::system::DeviceAuthenticationToken>(this, &g_DeviceAuthenticationTokenTaskThreadAllocator),
        m_pCachedTokenInfo(nullptr), m_RemainingTime(0), m_IsCacheEnabled(false) {}

    Result AsyncDeviceAuthenticationTokenImpl::Initialize(DeviceAuthenticationTokenInfoCache* cachedTokenInfo) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(cachedTokenInfo != nullptr);

        m_pCachedTokenInfo = cachedTokenInfo;
        NN_RESULT_SUCCESS;
    }

    void AsyncDeviceAuthenticationTokenImpl::EnableCachedToken(TimeSpan remainingTime) NN_NOEXCEPT
    {
        m_RemainingTime = remainingTime;
        m_IsCacheEnabled = true;
    }

    Result AsyncDeviceAuthenticationTokenImpl::ExecuteAndValue(ec::system::DeviceAuthenticationToken* outValue) NN_NOEXCEPT
    {
        // キャッシュしているデバイス認証トークンを返却してもよく、かつキャッシュしているあれば、デバイス認証トークンが失効するまでの時間を計算し
        // 許容する残り有効時間よりも長い場合は、キャッシュしているトークンを返却する。
        if (CanReturnCachedDeviceAuthenticationToken())
        {
            DeviceAuthenticationTokenInfo tokenInfo;
            bool isCached = m_pCachedTokenInfo->Get(&tokenInfo);

            if (isCached)
            {
                // 現在時刻を取得
                TimeSpan currentTime = os::GetSystemTick().ToTimeSpan();

                // デバイス認証トークンが失効するまでの時間を計算する(負の値ならば失効している)
                TimeSpan timeUntilExpiration = tokenInfo.expireTime - currentTime;

                // デバイス認証トークンが失効しておらず、デバイス認証トークンが失効するまでの時間が許容する残り有効時間よりも長い場合はキャッシュしているトークンを返却する。
                if (timeUntilExpiration > m_RemainingTime)
                {
                    NN_DETAIL_NIM_TRACE("[AsyncDeviceAuthenticationTokenImpl] Cached device authentication token returned. \n");
                    *outValue = tokenInfo.token;
                    NN_RESULT_SUCCESS;
                }
            }
        }

        TimeSpan expiration;
        NN_RESULT_DO(GetShopDeviceAuthenticationToken(&expiration, outValue, &m_Cancelable));

        // 取得したトークンをキャッシュする
        if (m_pCachedTokenInfo != nullptr)
        {
            DeviceAuthenticationTokenInfo tokenInfo;
            tokenInfo.token = *outValue;
            tokenInfo.expireTime = expiration;
            m_pCachedTokenInfo->Save(tokenInfo);
        }

        NN_RESULT_SUCCESS;
    }

    Result AsyncDeviceAuthenticationTokenImpl::CancelImpl() NN_NOEXCEPT
    {
        m_Cancelable.Cancel();
        NN_RESULT_SUCCESS;
    }

    // トークンのキャッシュを参照可能であること、デバイス認証トークンを返却してよい設定であるかを確認する
    bool AsyncDeviceAuthenticationTokenImpl::CanReturnCachedDeviceAuthenticationToken() NN_NOEXCEPT
    {
        return (m_pCachedTokenInfo != nullptr && m_IsCacheEnabled);
    }
}}}
