﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/service/nifm_GeneralServiceInterface.h>
#include <nn/nifm/detail/service/nifm_ServiceProvider.h>
#include <nn/nifm/detail/core/nifm_RequestManager.h>
#include <nn/nifm/detail/core/nifm_ConnectionSelector.h>
#include <nn/nifm/detail/core/networkInterface/nifm_WirelessInterface.h>
#include <nn/nifm/detail/core/networkInterface/nifm_EthernetInterface.h>
#include <nn/nifm/detail/service/nifm_RequestImpl.h>
#include <nn/nifm/detail/service/nifm_ScanRequestImpl.h>
#include <nn/nifm/detail/service/nifm_TemporaryNetworkProfileImpl.h>

#include <nn/nifm/detail/util/nifm_CancelFlagHolder.h>
#include <nn/nifm/detail/util/nifm_SettingsUtility.h>
#include <nn/nifm/detail/util/nifm_SfUtility.h>

#include <nn/nifm/nifm_TypesNetworkProfile.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_ExpHeapAllocator.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/settings/system/settings_WirelessLan.h>


namespace nn
{
namespace nifm
{
namespace detail
{

GeneralServiceInterface::GeneralServiceInterface(
    ServiceProvider* pServiceProvider,
    const Capability& capability,
    nn::sf::ExpHeapAllocator* pExpHeapAllocator,
    ClientId clientId,
    nn::Bit64 processId) NN_NOEXCEPT
    : m_pServiceProvider(pServiceProvider),
      m_Capability(capability),
      m_pExpHeapAllocator(pExpHeapAllocator),
      m_ClientId(clientId),
      m_ProcessId(processId)
{
}

GeneralServiceInterface::~GeneralServiceInterface() NN_NOEXCEPT
{
}

nn::Result GeneralServiceInterface::GetClientId(nn::sf::Out<ClientId> outClientId) const NN_NOEXCEPT
{
    outClientId.Set(m_ClientId);

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::CreateScanRequest(nn::sf::Out<nn::sf::SharedPointer<IScanRequest>> outIScanRequest) NN_NOEXCEPT
{
    ConnectionSelector* pConnectionSelector = &m_pServiceProvider->m_ConnectionSelector;

    auto pIScanRequest = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::CreateSharedEmplaced<IScanRequest, ScanRequestImpl>(
        m_pExpHeapAllocator, pConnectionSelector, m_Capability
    );

    NN_RESULT_THROW_UNLESS(pIScanRequest != nullptr, ResultResourceNotFound());

    outIScanRequest.Set(pIScanRequest);

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::GetAllowedChannels(nn::sf::OutArray<int16_t> outChannelArray, nn::sf::Out<int> outCount) const NN_NOEXCEPT
{
    int16_t outChannels[WirelessChannelsCountMax];
    NN_RESULT_DO(WirelessInterface::GetAllowedChannels(outChannels, outCount.GetPointer()));

    int minCount = std::min(*outCount, static_cast<int>(outChannelArray.GetLength()));
    for (int i = 0; i < minCount; ++i)
    {
        outChannelArray[i] = outChannels[i];
    }

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::GetScanDataOld(nn::sf::OutArray<detail::sf::AccessPointDataOld> outAccessPointDataArray, nn::sf::Out<int32_t> outCount ) const NN_NOEXCEPT
{
    AccessPointList<FrameHeap<16 * 1024>> accessPointList;
    NN_RESULT_DO(m_pServiceProvider->m_ConnectionSelector.GetScanData(&accessPointList));

    const WirelessAccessPoint* pWirelessAccessPointList[200];

    int fullCount = 0;
    for (const auto& accessPoint : accessPointList)
    {
        pWirelessAccessPointList[fullCount] = static_cast<const WirelessAccessPoint*>(&accessPoint);
        ++fullCount;

        if (fullCount >= NN_ARRAY_SIZE(pWirelessAccessPointList))
        {
            break;
        }
    }

    std::sort(&pWirelessAccessPointList[0], &pWirelessAccessPointList[fullCount],
        [](const WirelessAccessPoint* pLhs, const WirelessAccessPoint* pRhs) { return pLhs->GetPriority() < pRhs->GetPriority(); });

    int count = 0;
    int outLength = static_cast<int>(outAccessPointDataArray.GetLength());     // キャストしても元より小さくしかならないので安全なはず
    for (int i = 0; i < fullCount; ++i)
    {
        AccessPointData accessPointData;
        pWirelessAccessPointList[i]->TryExport(&accessPointData);

        if (accessPointData.isSupported)
        {
            if (count < outLength)
            {
                ConvertAccessPointDataToSfFromNifm(&outAccessPointDataArray[count], accessPointData);
            }
            ++count;
        }
    }

    outCount.Set(count);

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::GetScanData(nn::sf::OutArray<detail::sf::AccessPointData> outAccessPointDataArray, nn::sf::Out<int32_t> outCount) const NN_NOEXCEPT
{
    AccessPointList<FrameHeap<16 * 1024>> accessPointList;
    NN_RESULT_DO(m_pServiceProvider->m_ConnectionSelector.GetScanData(&accessPointList));

    const WirelessAccessPoint* pWirelessAccessPointList[200];

    int fullCount = 0;
    for (const auto& accessPoint : accessPointList)
    {
        pWirelessAccessPointList[fullCount] = static_cast<const WirelessAccessPoint*>(&accessPoint);
        ++fullCount;

        if (fullCount >= NN_ARRAY_SIZE(pWirelessAccessPointList))
        {
            break;
        }
    }

    std::sort(&pWirelessAccessPointList[0], &pWirelessAccessPointList[fullCount],
        [](const WirelessAccessPoint* pLhs, const WirelessAccessPoint* pRhs) { return pLhs->GetPriority() < pRhs->GetPriority(); });

    int count = 0;
    int outLength = static_cast<int>(outAccessPointDataArray.GetLength());     // キャストしても元より小さくしかならないので安全なはず
    for (int i = 0; i < fullCount; ++i)
    {
        AccessPointData accessPointData;
        if (count < outLength)
        {
            pWirelessAccessPointList[i]->TryExport(&accessPointData);
            ConvertAccessPointDataToSfFromNifm(&outAccessPointDataArray[count], accessPointData);
        }
        ++count;
    }

    outCount.Set(count);

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::CreateRequest(nn::sf::Out<nn::sf::SharedPointer<IRequest>> outIRequest, int32_t requirementPreset) NN_NOEXCEPT
{
    RequestManager* pRequestManager = &m_pServiceProvider->m_RequestManager;

    auto pIRequest = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::CreateSharedEmplaced<IRequest, RequestImpl>(
        m_pExpHeapAllocator, pRequestManager, m_Capability, static_cast<RequirementPreset>(requirementPreset), m_ClientId, m_ProcessId
    );

    NN_RESULT_THROW_UNLESS(pIRequest != nullptr, ResultResourceNotFound());
    NN_RESULT_THROW_UNLESS(pIRequest.GetImpl().IsAvailable(), ResultResourceNotFound());

    outIRequest.Set(pIRequest);

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::GetCurrentNetworkProfile( nn::sf::Out<sf::NetworkProfileData> outNetworkProfileData ) const NN_NOEXCEPT
{
    NetworkProfileData profile;
    nn::Result result = m_pServiceProvider->m_ConnectionSelector.GetCurrentNetworkProfile(&profile);

    if( result.IsSuccess() )
    {
        ConvertNetworkProfileDataToSfFromNifm( outNetworkProfileData.GetPointer(), profile );
    }

    NN_RESULT_THROW(result);
}

nn::Result GeneralServiceInterface::EnumerateNetworkInterfaces( nn::sf::OutArray<sf::NetworkInterfaceInfo> outNetworkInterfaceInfoArray, nn::sf::Out<int> outCount, nn::Bit32 filterFlags ) const NN_NOEXCEPT
{
    const size_t length = outNetworkInterfaceInfoArray.GetLength();
    if( length >= static_cast<size_t>(INT_MAX) )
    {
        // キャストの都合で大きすぎる入力は却下
        NN_RESULT_THROW(nn::nifm::ResultInvalidSize());   // TODO
    }
    if( (filterFlags & ~(EnumerateNetworkInterfacesFilter_Ieee80211 | EnumerateNetworkInterfacesFilter_Ethernet | EnumerateNetworkInterfacesFilter_Extended)) != 0 )
    {
        // 知らないフラグを指定していたら却下
        NN_RESULT_THROW(nn::nifm::ResultInvalidArgument());   // TODO
    }

    NetworkInterfaceInfo networkInterfaceInfo[NetworkInterfaceCountMax];
    int validInterfaceCount = 0;
    nn::Result result = m_pServiceProvider->m_ConnectionSelector.EnumerateNetworkInterfaces( networkInterfaceInfo, &validInterfaceCount, NN_ARRAY_SIZE(networkInterfaceInfo) );

    if( result.IsSuccess() )
    {
        NN_SDK_ASSERT(validInterfaceCount <= NetworkInterfaceCountMax);     // NetworkInterfaceCountMax は有効な NIC の数を超えないように定義されていなければならない

        static const int AllMacAddressCountMax = 4;   // TODO: [TORIAEZU]
        const int inCount = static_cast<int>(length);
        int outCount2 = 0;

        // 無線 NIC の列挙
        if( (filterFlags & EnumerateNetworkInterfacesFilter_Ieee80211) != 0 )
        {
            MacAddress macAddressList[AllMacAddressCountMax];
            int count;
            WirelessInterface::GetAllMacAddresses(macAddressList, &count, NN_ARRAY_SIZE(macAddressList));

            //NN_SDK_ASSERT(count <= AllMacAddressCountMax);
            count = std::min(count, AllMacAddressCountMax);

            for( int i = 0; i < count; ++i )
            {
                bool isAvailable = false;
                for( int j = 0; j < validInterfaceCount; ++j )
                {
                    if( macAddressList[i] == networkInterfaceInfo[j].macAddress )
                    {
                        isAvailable = true;
                        break;
                    }
                }

                if( isAvailable || ((filterFlags & EnumerateNetworkInterfacesFilter_Extended) != 0) )
                {
                    if( outCount2 < inCount )
                    {
                        outNetworkInterfaceInfoArray[outCount2].type = NetworkInterfaceType_Ieee80211;
                        outNetworkInterfaceInfoArray[outCount2].macAddress = macAddressList[i];
                        outNetworkInterfaceInfoArray[outCount2].isAvailable = isAvailable;
                    }

                    ++outCount2;
                }
            }
        }

        // 有線 NIC の列挙
        if( (filterFlags & EnumerateNetworkInterfacesFilter_Ethernet) != 0 )
        {
            MacAddress macAddressList[AllMacAddressCountMax];
            int count;
            m_pServiceProvider->m_EthernetInterfaceManager.GetAllMacAddresses(macAddressList, &count, NN_ARRAY_SIZE(macAddressList));

            //NN_SDK_ASSERT(count <= AllMacAddressCountMax);
            count = std::min(count, AllMacAddressCountMax);

            for( int i = 0; i < count; ++i )
            {
                bool isAvailable = false;
                for( int j = 0; j < validInterfaceCount; ++j )
                {
                    if( macAddressList[i] == networkInterfaceInfo[j].macAddress )
                    {
                        isAvailable = true;
                        break;
                    }
                }

                if( isAvailable || ((filterFlags & EnumerateNetworkInterfacesFilter_Extended) != 0) )
                {
                    if( outCount2 < inCount )
                    {
                        outNetworkInterfaceInfoArray[outCount2].type = NetworkInterfaceType_Ethernet;
                        outNetworkInterfaceInfoArray[outCount2].macAddress = macAddressList[i];
                        outNetworkInterfaceInfoArray[outCount2].isAvailable = isAvailable;
                    }

                    ++outCount2;
                }
            }
        }

        outCount.Set(outCount2);
    }

    NN_RESULT_THROW(result);
}

nn::Result GeneralServiceInterface::EnumerateNetworkProfiles( nn::sf::OutArray<sf::NetworkProfileBasicInfo> outNetworkProfileBasicInfoArray, nn::sf::Out<int> outCount, nn::Bit8 typeFlags ) const NN_NOEXCEPT
{
    if ((typeFlags & ~(NetworkProfileType_User | NetworkProfileType_SsidList | NetworkProfileType_Temporary)) != 0)
    {
        // 知らないフラグを指定していたら却下
        NN_RESULT_THROW(nn::nifm::ResultInvalidArgument());   // TODO
    }
    else if ((typeFlags & NetworkProfileType_SsidList) && !m_Capability.isSsidListAcquirable)
    {
        NN_RESULT_THROW(ResultNoCapability());
    }

    int totalCopyCount = 0;
    int totalProfileCount = 0;

    const auto& networkProfilemanager = m_pServiceProvider->m_NetworkProfileManager;

    if ((typeFlags & NetworkProfileType_User) != 0)
    {
        int profileCount = networkProfilemanager.GetNetworkProfileCount(NetworkProfileBase::ProfileType::UserProfile);
        totalProfileCount += profileCount;
        int copyCount = std::min<int>(static_cast<int>(outNetworkProfileBasicInfoArray.GetLength()) - totalCopyCount, profileCount);

        for (int i = 0; i < copyCount; ++i)
        {
            NetworkProfileBasicInfo networkProfileBasicInfo;
            if(networkProfilemanager.GetNetworkProfileBasicInfo(&networkProfileBasicInfo, NetworkProfileBase::ProfileType::UserProfile, i).IsSuccess())
            {
                ConvertNetworkProfileBasicInfoToSfFromNifm(&outNetworkProfileBasicInfoArray[totalCopyCount + i], networkProfileBasicInfo);
            }
        }
        totalCopyCount += copyCount;
    }

    if ((typeFlags & NetworkProfileType_SsidList) != 0)
    {
        int profileCount = networkProfilemanager.GetNetworkProfileCount(NetworkProfileBase::ProfileType::SsidListProfile);
        totalProfileCount += profileCount;
        int copyCount = std::min<int>(static_cast<int>(outNetworkProfileBasicInfoArray.GetLength()) - totalCopyCount, profileCount);

        for (int i = 0; i < copyCount; ++i)
        {
            NetworkProfileBasicInfo networkProfileBasicInfo;
            if(networkProfilemanager.GetNetworkProfileBasicInfo(&networkProfileBasicInfo, NetworkProfileBase::ProfileType::SsidListProfile, i).IsSuccess())
            {
                ConvertNetworkProfileBasicInfoToSfFromNifm(&outNetworkProfileBasicInfoArray[totalCopyCount + i], networkProfileBasicInfo);
            }
        }
        totalCopyCount += copyCount;
    }

    if ((typeFlags & NetworkProfileType_Temporary) != 0)
    {
        int profileCount = networkProfilemanager.GetNetworkProfileCount(NetworkProfileBase::ProfileType::TemporaryProfile);
        totalProfileCount += profileCount;
        int copyCount = std::min<int>(static_cast<int>(outNetworkProfileBasicInfoArray.GetLength()) - totalCopyCount, profileCount);

        for (int i = 0; i < copyCount; ++i)
        {
            NetworkProfileBasicInfo networkProfileBasicInfo;
            if(networkProfilemanager.GetNetworkProfileBasicInfo(&networkProfileBasicInfo, NetworkProfileBase::ProfileType::TemporaryProfile, i).IsSuccess())
            {
                ConvertNetworkProfileBasicInfoToSfFromNifm(&outNetworkProfileBasicInfoArray[totalCopyCount + i], networkProfileBasicInfo);
            }
        }
        totalCopyCount += copyCount;
    }

    outCount.Set(totalProfileCount);

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::GetNetworkProfile( nn::sf::Out<sf::NetworkProfileData> outNetworkProfileData, const nn::util::Uuid& id ) const NN_NOEXCEPT
{
    NetworkProfileData profile;

    const auto& networkProfileManager = m_pServiceProvider->m_NetworkProfileManager;
    NN_RESULT_DO(networkProfileManager.GetNetworkProfileData(&profile, id));

    NN_RESULT_THROW_UNLESS(profile.id != nn::util::InvalidUuid, ResultNetworkProfileNotFound());
    NN_RESULT_THROW_UNLESS(profile.networkProfileType != NetworkProfileType_SsidList || m_Capability.isSsidListAcquirable, ResultNoCapability());

    if (!m_Capability.isPassphraseAcquirable)
    {
        profile.wirelessSetting.security.sharedKey.length = 0;
        std::memset(profile.wirelessSetting.security.sharedKey.keyMaterial, 0, SharedKey::KeyMaterialSize);
        std::memset(profile.ipSetting.proxy.authentication.password, 0, AuthenticationSetting::PasswordSize);
    }

    ConvertNetworkProfileDataToSfFromNifm(outNetworkProfileData.GetPointer(), profile);

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::SetNetworkProfile( nn::sf::Out<nn::util::Uuid> outId, const sf::NetworkProfileData& networkProfileData ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Capability.isProfileEditable, ResultNoCapability());

    auto& networkProfileManager = m_pServiceProvider->m_NetworkProfileManager;

    NetworkProfileData profile;
    ConvertNetworkProfileDataToNifmFromSf( &profile, networkProfileData );

    nn::util::Uuid id;
    profile.networkProfileType = NetworkProfileType_User;
    NN_RESULT_DO(networkProfileManager.AddUserNetworkProfile(&id, profile));

    NN_RESULT_DO(networkProfileManager.ExportUserNetworkProfile());
    outId.Set(id);

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::RemoveNetworkProfile( const nn::util::Uuid& id ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Capability.isProfileEditable, ResultNoCapability());

    auto& networkProfileManager = m_pServiceProvider->m_NetworkProfileManager;

    NN_RESULT_DO(networkProfileManager.RemoveNetworkProfile(NetworkProfileBase::ProfileType::UserProfile, id));
    NN_RESULT_DO(networkProfileManager.ExportUserNetworkProfile());

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::GetCurrentIpAddress( nn::sf::Out<IpV4Address> outIpAddress ) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pServiceProvider->m_ConnectionSelector.GetCurrentIpAddress( outIpAddress.GetPointer() ));
}

nn::Result GeneralServiceInterface::GetCurrentIpConfigInfo( nn::sf::Out<IpAddressSetting> outIpAddressSetting, nn::sf::Out<DnsSetting> outDnsSetting) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pServiceProvider->m_ConnectionSelector.GetCurrentIpConfigInfo( outIpAddressSetting.GetPointer(), outDnsSetting.GetPointer() ));
}

nn::Result GeneralServiceInterface::GetCurrentAccessPointOld( nn::sf::Out<detail::sf::AccessPointDataOld> outAccessPointData ) NN_NOEXCEPT
{
    AccessPointData accessPoint;
    nn::Result result = m_pServiceProvider->m_ConnectionSelector.GetCurrentAccessPoint( &accessPoint );

    if( result.IsSuccess() )
    {
        ConvertAccessPointDataToSfFromNifm( outAccessPointData.GetPointer(), accessPoint );
    }

    NN_RESULT_THROW(result);
}

nn::Result GeneralServiceInterface::GetCurrentAccessPoint(nn::sf::Out<detail::sf::AccessPointData> outAccessPointData) NN_NOEXCEPT
{
    AccessPointData accessPoint;
    nn::Result result = m_pServiceProvider->m_ConnectionSelector.GetCurrentAccessPoint(&accessPoint);

    if (result.IsSuccess())
    {
        ConvertAccessPointDataToSfFromNifm(outAccessPointData.GetPointer(), accessPoint);
    }

    NN_RESULT_THROW(result);
}

nn::Result GeneralServiceInterface::CreateTemporaryNetworkProfile(nn::sf::Out<nn::sf::SharedPointer<INetworkProfile>> outINetworkProfile, nn::sf::Out<nn::util::Uuid> outNetworkProfileId, const sf::NetworkProfileData& networkProfileData) NN_NOEXCEPT
{
    NetworkProfileData profile;
    ConvertNetworkProfileDataToNifmFromSf( &profile, networkProfileData );
    profile.id = nn::util::InvalidUuid; // ID を持つ接続設定の上書き防止

    NetworkProfileManager* pNetworkProfileManager = &m_pServiceProvider->m_NetworkProfileManager;
    nn::util::Uuid id;
    NN_RESULT_DO(pNetworkProfileManager->AddTemporaryNetworkProfile(&id, profile));
    NN_SDK_ASSERT_NOT_EQUAL(nn::util::InvalidUuid, id);

    {
        auto pINetworkProfile = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::CreateSharedEmplaced<INetworkProfile, TemporaryNetworkProfileImpl>(
            m_pExpHeapAllocator, pNetworkProfileManager, id, m_Capability);

        if( pINetworkProfile != nullptr )
        {
            outNetworkProfileId.Set(id);
            outINetworkProfile.Set(pINetworkProfile);
        }
        else
        {
            nn::Result result = pNetworkProfileManager->RemoveNetworkProfile(NetworkProfileBase::ProfileType::TemporaryProfile, id);
            NN_SDK_ASSERT(result.IsSuccess()); // ここでの失敗は削除対象が存在しない場合のみ
            NN_UNUSED(result);
            NN_RESULT_THROW(nn::nifm::ResultOutOfHeap()); // TORIAEZU
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::SetWirelessCommunicationEnabled(bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Capability.isWirelessCommunicationControllable, ResultNoCapability());

    m_pServiceProvider->m_RequestManager.UpdateAggregatedRequest();
    NN_RESULT_DO(m_pServiceProvider->m_WirelessInterface.SetEnabled(isEnabled));

    if (!isEnabled)
    {
        CancelFlagHolder::GetSingleton().RequireCancel<CancelFlagHolder::ReasonFlag::WirelessNetworkInterfaceNoLongerAvailable>();
    }

    nn::settings::system::SetWirelessLanEnabled(isEnabled);

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::SetWirelessCommunicationEnabledForTest(bool isEnabled) NN_NOEXCEPT
{
    bool isCommunicationControlEnabledForTest = false;

    NN_RESULT_DO(IsCommunicationControlEnabledForTest(&isCommunicationControlEnabledForTest));

    NN_RESULT_THROW_UNLESS(isCommunicationControlEnabledForTest, ResultDevelopmentOnly());

    m_pServiceProvider->m_RequestManager.UpdateAggregatedRequest();
    NN_RESULT_DO(m_pServiceProvider->m_WirelessInterface.SetEnabled(isEnabled));

    if (!isEnabled)
    {
        CancelFlagHolder::GetSingleton().RequireCancel<CancelFlagHolder::ReasonFlag::WirelessNetworkInterfaceNoLongerAvailable>();
    }

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::IsWirelessCommunicationEnabled(nn::sf::Out<bool> outIsEnabled) NN_NOEXCEPT
{
    outIsEnabled.Set(m_pServiceProvider->m_WirelessInterface.IsEnabled());

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::GetInternetConnectionStatus(nn::sf::Out<sf::InternetConnectionStatus> outConnectionStatus) NN_NOEXCEPT
{
    //

    InternetConnectionStatus connectionInfo;
    TotalNetworkResourceInfo totalNetworkResourceInfo;
    NN_RESULT_DO(m_pServiceProvider->m_ConnectionSelector.GetTotalNetworkResourceInfo(&totalNetworkResourceInfo));

    if (totalNetworkResourceInfo.networkResourceState == NetworkResourceState::ToBePrepared ||
        totalNetworkResourceInfo.networkResourceState == NetworkResourceState::Available)
    {
        connectionInfo.networkInterfaceType = totalNetworkResourceInfo.networkInterfaceType;

        if (totalNetworkResourceInfo.networkType == NetworkType_Internet)
        {
            if (totalNetworkResourceInfo.networkInterfaceType == NetworkInterfaceType_Ieee80211)
            {
                AccessPointData accessPoint;
                nn::Result result = m_pServiceProvider->m_ConnectionSelector.GetCurrentAccessPoint(&accessPoint);
                if (result.IsSuccess())
                {
                    connectionInfo.linkLevel = accessPoint.linkLevel;
                }
            }
            connectionInfo.internetAvailability = totalNetworkResourceInfo.internetAvailability;
        }
        else
        {
            NN_RESULT_THROW(ResultNotInternetConnection());
        }
    }
    else
    {
        NN_RESULT_THROW(ResultNotConnected());
    }

    ConvertInternetConnectionStatusToSfFromNifm(outConnectionStatus.GetPointer(), connectionInfo);

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::SetEthernetCommunicationEnabled(bool isEnabled) NN_NOEXCEPT
{
    m_pServiceProvider->m_RequestManager.UpdateAggregatedRequest();
    NN_RESULT_DO(m_pServiceProvider->m_EthernetInterfaceManager.SetEnabled(isEnabled));

    if (!isEnabled)
    {
        CancelFlagHolder::GetSingleton().RequireCancel<CancelFlagHolder::ReasonFlag::EthernetNetworkInterfaceNoLongerAvailable>();
    }

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::SetEthernetCommunicationEnabledForTest(bool isEnabled) NN_NOEXCEPT
{
    bool isCommunicationControlEnabledForTest = false;

    NN_RESULT_DO(IsCommunicationControlEnabledForTest(&isCommunicationControlEnabledForTest));

    NN_RESULT_THROW_UNLESS(isCommunicationControlEnabledForTest, ResultDevelopmentOnly());

    m_pServiceProvider->m_RequestManager.UpdateAggregatedRequest();
    NN_RESULT_DO(m_pServiceProvider->m_EthernetInterfaceManager.SetEnabled(isEnabled));

    if (!isEnabled)
    {
        CancelFlagHolder::GetSingleton().RequireCancel<CancelFlagHolder::ReasonFlag::EthernetNetworkInterfaceNoLongerAvailable>();
    }

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::IsEthernetCommunicationEnabled(nn::sf::Out<bool> outIsEnabled) NN_NOEXCEPT
{
    outIsEnabled.Set(m_pServiceProvider->m_EthernetInterfaceManager.IsEnabled());

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::IsAnyInternetRequestAccepted(nn::sf::Out<bool> outIsAnyInternetRequestAccepted, ClientId clientId) NN_NOEXCEPT
{
    outIsAnyInternetRequestAccepted.Set(m_pServiceProvider->m_RequestManager.IsAnyInternetRequestAccepted(clientId));

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::IsAnyForegroundRequestAccepted(nn::sf::Out<bool> outIsAnyForegroundRequestAccepted) NN_NOEXCEPT
{
    outIsAnyForegroundRequestAccepted.Set(m_pServiceProvider->m_RequestManager.IsAnyForegroundRequestAccepted());

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::PutToSleep() NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pServiceProvider->m_PowerStateCoordinator.PutToSleep());
}

nn::Result GeneralServiceInterface::WakeUp() NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pServiceProvider->m_PowerStateCoordinator.WakeUp());
}

nn::Result GeneralServiceInterface::Shutdown() NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pServiceProvider->m_PowerStateCoordinator.Shutdown());
}

nn::Result GeneralServiceInterface::SetExclusiveClient(const ClientId& clientId) NN_NOEXCEPT
{
    // TODO: デバッグフラグ等のチェック

    NN_RESULT_THROW(m_pServiceProvider->m_RequestManager.SetExclusiveClient(clientId));
}

nn::Result GeneralServiceInterface::GetSsidListVersion(nn::sf::Out<SsidListVersion> outSsidListVersion) const NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pServiceProvider->m_NetworkProfileManager.GetSsidListVersion(outSsidListVersion.GetPointer()));
}

nn::Result GeneralServiceInterface::GetDefaultIpSetting(nn::sf::Out<IpSettingData> outIpSettingData) const NN_NOEXCEPT
{
    m_pServiceProvider->m_NetworkProfileManager.GetDefaultIpSetting(outIpSettingData.GetPointer());

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::SetDefaultIpSetting(const IpSettingData& ipSettingData) NN_NOEXCEPT
{
    m_pServiceProvider->m_NetworkProfileManager.SetDefaultIpSetting(ipSettingData);

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::GetTelemetorySystemEventReadableHandle(nn::sf::Out<nn::sf::NativeHandle> systemEventReadableHandle) const NN_NOEXCEPT
{
    systemEventReadableHandle.Set(nn::sf::NativeHandle(m_pServiceProvider->m_ConnectionSelector.GetTelemetrySystemEventReadableHandle(), false));

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::GetTelemetryInfo(nn::sf::Out<TelemetryInfo> outTelemetryInfo) const NN_NOEXCEPT
{
    m_pServiceProvider->m_ConnectionSelector.GetTelemetryInfo(outTelemetryInfo.GetPointer());

    NN_RESULT_SUCCESS;
}

nn::Result GeneralServiceInterface::ConfirmSystemAvailability() const NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pServiceProvider->m_ConnectionSelector.ConfirmInterfaceHealth());
}

nn::Result GeneralServiceInterface::SetBackgroundRequestEnabled(bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Capability.isRequestEnabledSettable, ResultNoCapability());

    NN_RESULT_THROW(m_pServiceProvider->m_RequestManager.SetBackgroundRequestEnabled(isEnabled));
}

}
}
}
