﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/profile/nifm_IpSetting.h>

#include <nn/util/util_TFormatString.h>
#include <nn/util/util_StringUtil.h>

#include <locale>
#include <cstring>


namespace nn
{
namespace nifm
{
namespace detail
{

IpSetting::IpSetting() NN_NOEXCEPT
{
    Clear();
}

IpSetting::IpSetting( const IpSettingData& setting ) NN_NOEXCEPT
{
    m_Setting = setting;
}

IpSetting::~IpSetting() NN_NOEXCEPT
{
}

bool IpSetting::IsIpAddressSettingAuto() const NN_NOEXCEPT
{
    return m_Setting.ip.isAuto;
}

IpV4Address* IpSetting::GetIpAddress(IpV4Address* pOutIpAddress) const NN_NOEXCEPT
{
    memcpy( pOutIpAddress, &m_Setting.ip.ipAddress,  NN_ARRAY_SIZE(m_Setting.ip.ipAddress.data) );
    return pOutIpAddress;
}

Bit8* IpSetting::GetIpAddress(Bit8* pOutIpAddress) const NN_NOEXCEPT
{
    memcpy( pOutIpAddress, &m_Setting.ip.ipAddress.data, NN_ARRAY_SIZE(m_Setting.ip.ipAddress.data) );
    return pOutIpAddress;
}

char* IpSetting::GetIpAddressString(char* pOutIpAddressString) const NN_NOEXCEPT
{
    ConvertIpV4AddressToString( pOutIpAddressString, &m_Setting.ip.ipAddress );
    return pOutIpAddressString;
}

IpV4Address* IpSetting::GetSubnetMask(IpV4Address* pOutSubnetMask) const NN_NOEXCEPT
{
    memcpy( pOutSubnetMask, &m_Setting.ip.subnetMask, NN_ARRAY_SIZE(m_Setting.ip.subnetMask.data) );
    return pOutSubnetMask;
}

Bit8* IpSetting::GetSubnetMask(Bit8* pOutSubnetMask) const NN_NOEXCEPT
{
    memcpy( pOutSubnetMask, &m_Setting.ip.subnetMask.data, NN_ARRAY_SIZE(m_Setting.ip.subnetMask.data) );
    return pOutSubnetMask;
}

char* IpSetting::GetSubnetMaskString(char* pOutSubnetMaskString) const NN_NOEXCEPT
{
    ConvertIpV4AddressToString( pOutSubnetMaskString, &m_Setting.ip.subnetMask );
    return pOutSubnetMaskString;
}

IpV4Address* IpSetting::GetDefaultGateway(IpV4Address* pOutDefaultGateway) const NN_NOEXCEPT
{
    memcpy( pOutDefaultGateway, &m_Setting.ip.defaultGateway, NN_ARRAY_SIZE(m_Setting.ip.defaultGateway.data) );
    return pOutDefaultGateway;
}

Bit8* IpSetting::GetDefaultGateway(Bit8* pOutDefaultGateway) const NN_NOEXCEPT
{
    memcpy( pOutDefaultGateway, &m_Setting.ip.defaultGateway.data, NN_ARRAY_SIZE(m_Setting.ip.defaultGateway.data) );
    return pOutDefaultGateway;
}

char* IpSetting::GetDefaultGatewayString(char* pOutDefaultGatewayString) const NN_NOEXCEPT
{
    ConvertIpV4AddressToString( pOutDefaultGatewayString, &m_Setting.ip.defaultGateway );
    return pOutDefaultGatewayString;
}

bool IpSetting::IsDnsSettingAuto() const NN_NOEXCEPT
{
    return m_Setting.dns.isAuto;
}

IpV4Address* IpSetting::GetPreferredDns(IpV4Address* pOutDns) const NN_NOEXCEPT
{
    memcpy(pOutDns, &m_Setting.dns.preferredDns, NN_ARRAY_SIZE(m_Setting.dns.preferredDns.data) );
    return pOutDns;
}

Bit8* IpSetting::GetPreferredDns(Bit8* pOutDns) const NN_NOEXCEPT
{
    memcpy(pOutDns, &m_Setting.dns.preferredDns.data, NN_ARRAY_SIZE(m_Setting.dns.preferredDns.data) );
    return pOutDns;
}

char* IpSetting::GetPreferredDnsString(char* pOutDnsString) const NN_NOEXCEPT
{
    ConvertIpV4AddressToString( pOutDnsString, &m_Setting.dns.preferredDns );
    return pOutDnsString;
}

IpV4Address* IpSetting::GetAlternateDns(IpV4Address* pOutDns) const NN_NOEXCEPT
{
    memcpy(pOutDns, &m_Setting.dns.alternateDns, NN_ARRAY_SIZE(m_Setting.dns.alternateDns.data) );
    return pOutDns;
}

Bit8* IpSetting::GetAlternateDns(Bit8* pOutDns) const NN_NOEXCEPT
{
    memcpy(pOutDns, &m_Setting.dns.alternateDns.data, NN_ARRAY_SIZE(m_Setting.dns.alternateDns.data) );
    return pOutDns;
}

char* IpSetting::GetAlternateDnsString(char* pOutDnsString) const NN_NOEXCEPT
{
    ConvertIpV4AddressToString( pOutDnsString, &m_Setting.dns.alternateDns );
    return pOutDnsString;
}

bool IpSetting::IsProxyEnabled() const NN_NOEXCEPT
{
    return m_Setting.proxy.isEnabled;
}

uint16_t IpSetting::GetProxyPort() const NN_NOEXCEPT
{
    return m_Setting.proxy.port;
}

char* IpSetting::GetProxyHostname(char *pOutHostname) const NN_NOEXCEPT
{
    nn::util::Strlcpy(pOutHostname, m_Setting.proxy.proxy, NN_ARRAY_SIZE(m_Setting.proxy.proxy) );
    return pOutHostname;
}

bool IpSetting::IsProxyAuthEnabled() const NN_NOEXCEPT
{
    return m_Setting.proxy.authentication.isEnabled;
}

char* IpSetting::GetProxyUsername(char *pOutUsername) const NN_NOEXCEPT
{
    nn::util::Strlcpy(pOutUsername, m_Setting.proxy.authentication.username, NN_ARRAY_SIZE(m_Setting.proxy.authentication.username) );
    return pOutUsername;
}

char* IpSetting::GetProxyPassword(char *pOutPassword) const NN_NOEXCEPT
{
    nn::util::Strlcpy(pOutPassword, m_Setting.proxy.authentication.password, NN_ARRAY_SIZE(m_Setting.proxy.authentication.password) );
    return pOutPassword;
}

uint16_t IpSetting::GetMtu() const NN_NOEXCEPT
{
    return m_Setting.mtu;
}


void IpSetting::SetIpAddressSettingAuto(bool isAuto) NN_NOEXCEPT
{
    m_Setting.ip.isAuto = isAuto;
}

void IpSetting::SetIpAddress(const IpV4Address* pIpAddress) NN_NOEXCEPT
{
    memcpy(&m_Setting.ip.ipAddress, pIpAddress, NN_ARRAY_SIZE(m_Setting.ip.ipAddress.data) );
}

void IpSetting::SetIpAddress(const Bit8* pIpAddress) NN_NOEXCEPT
{
    memcpy(&m_Setting.ip.ipAddress.data, pIpAddress, NN_ARRAY_SIZE(m_Setting.ip.ipAddress.data) );
}

void IpSetting::SetSubnetMask(const IpV4Address* pSubnetMask)NN_NOEXCEPT
{
    memcpy(&m_Setting.ip.subnetMask, pSubnetMask, NN_ARRAY_SIZE(m_Setting.ip.subnetMask.data) );
}

void IpSetting::SetSubnetMask(const Bit8* pSubnetMask)NN_NOEXCEPT
{
    memcpy(&m_Setting.ip.subnetMask.data, pSubnetMask, NN_ARRAY_SIZE(m_Setting.ip.subnetMask.data) );
}

void IpSetting::SetDefaultGateway(const IpV4Address* pDefaultGateway) NN_NOEXCEPT
{
    memcpy(&m_Setting.ip.defaultGateway, pDefaultGateway, NN_ARRAY_SIZE(m_Setting.ip.defaultGateway.data) );
}

void IpSetting::SetDefaultGateway(const Bit8* pDefaultGateway) NN_NOEXCEPT
{
    memcpy(&m_Setting.ip.defaultGateway.data, pDefaultGateway, NN_ARRAY_SIZE(m_Setting.ip.defaultGateway.data) );
}

void IpSetting::SetDnsSettingAuto(bool isAuto) NN_NOEXCEPT
{
    m_Setting.dns.isAuto = isAuto;
}

void IpSetting::SetPreferredDns(const IpV4Address* pDns) NN_NOEXCEPT
{
    memcpy(&m_Setting.dns.preferredDns, pDns, NN_ARRAY_SIZE(m_Setting.dns.preferredDns.data) );
}

void IpSetting::SetPreferredDns(const Bit8* pDns) NN_NOEXCEPT
{
    memcpy(&m_Setting.dns.preferredDns.data, pDns, NN_ARRAY_SIZE(m_Setting.dns.preferredDns.data) );
}

void IpSetting::SetAlternateDns(const IpV4Address* pDns) NN_NOEXCEPT
{
    memcpy(&m_Setting.dns.alternateDns, pDns, NN_ARRAY_SIZE(m_Setting.dns.alternateDns.data) );
}

void IpSetting::SetAlternateDns(const Bit8* pDns) NN_NOEXCEPT
{
    memcpy(&m_Setting.dns.alternateDns.data, pDns, NN_ARRAY_SIZE(m_Setting.dns.alternateDns.data) );
}

void IpSetting::SetProxyEnabled(bool isEnabled) NN_NOEXCEPT
{
    m_Setting.proxy.isEnabled = isEnabled;
}

void IpSetting::SetProxyPort(uint16_t port) NN_NOEXCEPT
{
    m_Setting.proxy.port = port;
}

void IpSetting::SetProxyHostname(const char *pHostname) NN_NOEXCEPT
{
    nn::util::Strlcpy(m_Setting.proxy.proxy, pHostname, NN_ARRAY_SIZE(m_Setting.proxy.proxy) );
}

void IpSetting::SetProxyAuthEnabled(const bool isEnabled) NN_NOEXCEPT
{
    m_Setting.proxy.authentication.isEnabled = isEnabled;
}

void IpSetting::SetProxyUsername(const char *pUsername) NN_NOEXCEPT
{
    nn::util::Strlcpy(m_Setting.proxy.authentication.username, pUsername, NN_ARRAY_SIZE(m_Setting.proxy.authentication.username) );
}

void IpSetting::SetProxyPassword(const char *pPassword) NN_NOEXCEPT
{
    nn::util::Strlcpy(m_Setting.proxy.authentication.password, pPassword, NN_ARRAY_SIZE(m_Setting.proxy.authentication.password) );
}

void IpSetting::SetMtu(uint16_t mtu) NN_NOEXCEPT
{
    m_Setting.mtu = mtu;
}


void IpSetting::Clear() NN_NOEXCEPT
{
    m_Setting.ip.isAuto = true;
    memset( &m_Setting.ip.ipAddress.data, 0, NN_ARRAY_SIZE(m_Setting.ip.ipAddress.data) );
    memset( &m_Setting.ip.subnetMask.data, 0, NN_ARRAY_SIZE(m_Setting.ip.subnetMask.data) );
    memset( &m_Setting.ip.defaultGateway.data, 0, NN_ARRAY_SIZE(m_Setting.ip.defaultGateway.data) );
    m_Setting.dns.isAuto = true;
    memset( &m_Setting.dns.preferredDns.data, 0, NN_ARRAY_SIZE(m_Setting.dns.preferredDns.data) );
    memset( &m_Setting.dns.alternateDns.data, 0, NN_ARRAY_SIZE(m_Setting.dns.alternateDns.data) );
    m_Setting.proxy.isEnabled = false;
    m_Setting.proxy.port = 0;
    m_Setting.proxy.proxy[0] = '\0';
    m_Setting.proxy.authentication.isEnabled = false;
    m_Setting.proxy.authentication.username[0] = '\0';
    m_Setting.proxy.authentication.password[0] = '\0';
    m_Setting.mtu = DefaultMtu;
}

bool IpSetting::operator==(const IpSetting& rh) const NN_NOEXCEPT
{
    return m_Setting.ip.isAuto == rh.m_Setting.ip.isAuto &&
           memcmp( &m_Setting.ip.ipAddress.data, &rh.m_Setting.ip.ipAddress.data, NN_ARRAY_SIZE(m_Setting.ip.ipAddress.data) ) == 0 &&
           memcmp( &m_Setting.ip.subnetMask.data, &rh.m_Setting.ip.subnetMask.data,NN_ARRAY_SIZE(m_Setting.ip.subnetMask.data) ) == 0 &&
           memcmp( &m_Setting.ip.defaultGateway.data, &rh.m_Setting.ip.defaultGateway.data, NN_ARRAY_SIZE(m_Setting.ip.defaultGateway.data) ) == 0 &&
           m_Setting.dns.isAuto == rh.m_Setting.dns.isAuto &&
           memcmp( &m_Setting.dns.preferredDns.data, &rh.m_Setting.dns.preferredDns.data, NN_ARRAY_SIZE(m_Setting.dns.preferredDns.data) ) == 0 &&
           memcmp( &m_Setting.dns.alternateDns.data, &rh.m_Setting.dns.alternateDns.data, NN_ARRAY_SIZE(m_Setting.dns.alternateDns.data) ) == 0 &&
           m_Setting.proxy.isEnabled == rh.m_Setting.proxy.isEnabled &&
           m_Setting.proxy.port == rh.m_Setting.proxy.port &&
           util::Strncmp( m_Setting.proxy.proxy, rh.m_Setting.proxy.proxy, NN_ARRAY_SIZE(m_Setting.proxy.proxy) ) == 0 &&
           m_Setting.proxy.authentication.isEnabled == rh.m_Setting.proxy.authentication.isEnabled &&
           util::Strncmp( m_Setting.proxy.authentication.username, rh.m_Setting.proxy.authentication.username, NN_ARRAY_SIZE(m_Setting.proxy.authentication.username) ) == 0 &&
           util::Strncmp( m_Setting.proxy.authentication.password, rh.m_Setting.proxy.authentication.password, NN_ARRAY_SIZE(m_Setting.proxy.authentication.password) ) == 0 &&
           m_Setting.mtu == rh.m_Setting.mtu;
}

bool IpSetting::operator!=(const IpSetting& rh) const NN_NOEXCEPT
{
    return !(*this == rh);
}

IpSetting::operator IpSettingData() const NN_NOEXCEPT
{
    return m_Setting;
}

IpSetting& IpSetting::operator=( const IpSetting& rh ) NN_NOEXCEPT
{
    // TODO:
    memcpy( &m_Setting, &rh.m_Setting, sizeof(m_Setting) );
    return *this;
}

void IpSetting::ConvertIpV4AddressToString( char* pOutStr, const IpV4Address* pAddress ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutStr );
    NN_SDK_ASSERT_NOT_NULL( pAddress );

    nn::util::SNPrintf( pOutStr, sizeof("xxx.xxx.xxx.xxx"), "%u.%u.%u.%u", pAddress->data[0], pAddress->data[1], pAddress->data[2], pAddress->data[3] );
}

void IpSetting::Export(IpSettingData* pOutIpSetting) const NN_NOEXCEPT
{
    std::memcpy(pOutIpSetting, &m_Setting, sizeof(*pOutIpSetting));
}

}
}
}
