﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/nifm_UserRequest.h>
#include <nn/nifm/detail/core/nifm_RequestManager.h>
#include <nn/nifm/detail/util/nifm_SignalObject.h>

#include <nn/util/util_StringUtil.h>
#include <nn/util/util_LockGuard.h>
#include <nn/result/result_HandlingUtility.h>

#include <mutex>


namespace nn
{
namespace nifm
{
namespace detail
{

namespace
{
    const int DeprecatedConnectionConfirmationOption_NotRequired = ConnectionConfirmationOption_Prohibited + 1;

    /*
     * 最高優先度(0)
     *   （システム）
     * UserHighPriorityTop(64)
     *   （アプリ/システム）
     *    90 <- 一般アプリ(Local)
     * UserHighPriorityBottom(94)
     *   （システム）
     * UserLowPriorityTop(128) <- 一般アプリ(Internet)
     *   （アプリ/システム）
     * UserLowPriorityBottom(158)
     *   （システム）
     * 最低優先度(255)
     */

    const uint8_t UserHighPriorityTop = 64;
    const uint8_t UserLowPriorityTop = 128;

    static const uint8_t rawPriorityTable[] = {
        // High Priority(0-15 -> 64-94)
        UserHighPriorityTop + 0 /* 0:64*/, UserHighPriorityTop + 2 /* 1:66*/, UserHighPriorityTop + 4 /* 2:68*/, UserHighPriorityTop + 6 /* 3:70*/,
        UserHighPriorityTop + 8 /* 4:72*/, UserHighPriorityTop + 10/* 5:74*/, UserHighPriorityTop + 12/* 6:76*/, UserHighPriorityTop + 14/* 7:78*/,
        UserHighPriorityTop + 16/* 8:80*/, UserHighPriorityTop + 18/* 9:82*/, UserHighPriorityTop + 20/*10:84*/, UserHighPriorityTop + 22/*11:86*/,
        UserHighPriorityTop + 24/*12:88*/, UserHighPriorityTop + 26/*13:90*/, UserHighPriorityTop + 28/*14:92*/, UserHighPriorityTop + 30/*15:94*/,
        // Low Priority(16-31 -> 128-158)
        UserLowPriorityTop + 0 /*16:128*/, UserLowPriorityTop + 2 /*17:130*/, UserLowPriorityTop + 4 /*18:132*/, UserLowPriorityTop + 6 /*19:134*/,
        UserLowPriorityTop + 8 /*20:136*/, UserLowPriorityTop + 10/*21:138*/, UserLowPriorityTop + 12/*22:140*/, UserLowPriorityTop + 14/*23:142*/,
        UserLowPriorityTop + 16/*24:144*/, UserLowPriorityTop + 18/*25:146*/, UserLowPriorityTop + 20/*26:148*/, UserLowPriorityTop + 22/*27:150*/,
        UserLowPriorityTop + 24/*28:152*/, UserLowPriorityTop + 26/*29:154*/, UserLowPriorityTop + 28/*30:156*/, UserLowPriorityTop + 30/*31:158*/,
    };

    inline uint8_t ConvertPriorityToRawPriority(uint8_t priority)
    {
        NN_SDK_REQUIRES(priority >= UserRequestPriorityTop);
        NN_SDK_REQUIRES(priority <= UserRequestPriorityBottom);

        NN_STATIC_ASSERT(sizeof(rawPriorityTable) == 32);

        return rawPriorityTable[priority];
    }

    static const Requirement RequirementNone =
    {
        0,                                          // priority
        true,                                       // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        false,                                      // _isGreedy
        true,                                       // _isSharable
        false,                                      // _isForeground
        false,                                      // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_None,                           // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Invalid        // _connectionConfirmationOption
    };

    /****************
     * アプリ向け
     ****************/

    // インターネット通信：汎用
    static const Requirement RequirementApplicationInternetGeneric =
    {
        ConvertPriorityToRawPriority(20),           // _priority(136)
        true,                                       // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        false,                                      // _isGreedy
        false,                                      // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        true,                                       // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Required       // _connectionConfirmationOption
    };

    // インターネット通信：広帯域
    static const Requirement RequirementApplicationInternetBestEffort =
    {
        ConvertPriorityToRawPriority(16),           // _priority(128)
        true,                                       // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        true,                                       // _isGreedy
        true,                                       // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        true,                                       // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Required       // _connectionConfirmationOption
    };

    // ローカル通信：汎用
    static const Requirement RequirementApplicationLocalGeneric =
    {
        ConvertPriorityToRawPriority(13),           // _priority(90)
        false,                                      // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        true,                                       // _isGreedy
        false,                                      // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_Local,                          // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Invalid        // _connectionConfirmationOption
    };

    // すれちがい通信：汎用
    static const Requirement RequirementApplicationNeighborDetectionGeneric =
    {
        ConvertPriorityToRawPriority(22),           // _priority(140)
        false,                                      // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        true,                                       // _isGreedy
        true,                                       // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_NeighborDetection,              // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Invalid        // _connectionConfirmationOption
    };

    // スキャン : 汎用
    static const Requirement RequirementApplicationPassiveScan =
    {
        ConvertPriorityToRawPriority(11),           // _priority(86)
        true,                                       // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        true,                                       // _isGreedy
        false,                                      // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_None,                           // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Invalid        // _connectionConfirmationOption
    };

    /**********************************************/
    /*  これ以下の preset の使用は、権限が必要    */
    /**********************************************/

    /*********************************************
     * システムアプレット/ライブラリアプレット向け
     *********************************************/

    // アプレット（インターネット通信）：汎用
    static const Requirement RequirementAppletGeneric =
    {
        63,                                         // _priority
        true,                                       // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        false,                                      // _isGreedy
        true,                                       // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        true,                                       // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Required       // _connectionConfirmationOption
    };

    // アプレット（インターネット通信）：一部 BG 通信を制限する
    static const Requirement RequirementAppletInternetBestEffort =
    {
        61,                                         // _priority
        true,                                       // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        true,                                       // _isGreedy
        true,                                       // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        true,                                       // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Required       // _connectionConfirmationOption
    };

    // アプレット（インターネット通信）：一部 BG 通信を制限するかつ，Persistent
    static const Requirement RequirementAppletInternetBestEffortPersistent =
    {
        61,                                         // _priority
        true,                                       // _isRejectable
        true,                                       // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        true,                                       // _isGreedy
        true,                                       // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        true,                                       // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Required       // _connectionConfirmationOption
    };

    // アプレット（インターネット通信）：ネットワーク接続アプレット
    static const Requirement RequirementAppletNetworkConnection =
    {
        59,                                         // _priority
        true,                                       // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        true,                                       // _isSustainable
        false,                                      // _isGreedy
        true,                                       // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        true,                                       // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Required       // _connectionConfirmationOption
    };

    // アプレット（インターネット通信）：Web 認証アプレット
    static const Requirement RequirementAppletWifiWebAuth =
    {
        59,                                         // _priority
        true,                                       // _isRejectable
        true,                                       // _isPersistent
        true,                                       // _isInstant
        true,                                       // _isSustainable
        false,                                      // _isGreedy
        true,                                       // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Prohibited     // _connectionConfirmationOption
    };

    // アプレット（ローカル通信）：汎用
    static const Requirement RequirementApplicationLocalForApplet =
    {
        55,                                         // _priority
        false,                                      // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        true,                                       // _isGreedy
        false,                                      // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_Local,                          // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Invalid        // _connectionConfirmationOption
    };

    // アプレット（インターネット通信）：Persistent かつ、すべての要求と共存可能
    static const Requirement RequirementAppletContinuous =
    {
        65,                                         // _priority
        true,                                       // _isRejectable
        true,                                       // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        false,                                      // _isGreedy
        true,                                       // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        true,                                       // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Required       // _connectionConfirmationOption
    };

    // アプレット（インターネット接続）：DevMenu/DevMenuSystem
    static const Requirement RequirementAppletDevMenu =
    {
        141,                                        // _priority
        true,                                       // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        false,                                      // _isGreedy
        true,                                       // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        true,                                       // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Forced         // _connectionConfirmationOption
    };

    // アプレット（すれちがい通信）: Encounter
    static const Requirement RequirementAppletNeighborDetection =
    {
        77,                                         // _priority
        false,                                      // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        true,                                       // _isGreedy
        true,                                       // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_NeighborDetection,              // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Invalid        // _connectionConfirmationOption
    };

    // アプレット（スキャン） : netConnect
    static const Requirement RequirementAppletPassiveScan =
    {
        51,                                         // _priority
        true,                                       // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        true,                                       // _isGreedy
        false,                                      // _isSharable
        true,                                       // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_None,                           // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Invalid        // _connectionConfirmationOption
    };

    /***********************
     * システムプロセス向け
     ***********************/

    // 単発のインターネット利用要求
    static const Requirement RequirementSystemProcessGeneric =
    {
        161,                                        // _priority
        true,                                       // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        false,                                      // _isGreedy
        false,                                      // _isSharable
        false,                                      // _isForeground
        false,                                      // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Required       // _connectionConfirmationOption
    };

    // Persistent なインターネット利用要求
    static const Requirement RequirementSystemProcessPersistent =
    {
        171,                                        // _priority
        true,                                       // _isRejectable
        true,                                       // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        false,                                      // _isGreedy
        false,                                      // _isSharable
        false,                                      // _isForeground
        false,                                      // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Required       // _connectionConfirmationOption
    };

    // Persistent かつ、すべての要求と共存可能なインターネット利用要求
    static const Requirement RequirementSystemProcessContinuous =
    {
        181,                                        // _priority
        true,                                       // _isRejectable
        true,                                       // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        false,                                      // _isGreedy
        true,                                       // _isSharable
        false,                                      // _isForeground
        false,                                      // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Required       // _connectionConfirmationOption
    };

    // 単発の、すべての要求と共存可能なインターネット利用要求
    static const Requirement RequirementSystemProcessNotGreedySharable =
    {
        163,                                        // _priority
        true,                                       // _isRejectable
        false,                                      // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        false,                                      // _isGreedy
        true,                                       // _isSharable
        false,                                      // _isForeground
        false,                                      // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Required       // _connectionConfirmationOption
    };

    // 無線接続時にスリープに入っても維持される、 Persistent かつ、すべての要求と共存可能なインターネット利用要求
    static const Requirement RequirementNpns =
    {
        181,                                        // _priority
        true,                                       // _isRejectable
        true,                                       // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        false,                                      // _isGreedy
        true,                                       // _isSharable
        false,                                      // _isForeground
        false,                                      // _isAutomaticSwitchProhibited
        true,                                       // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Required       // _connectionConfirmationOption
    };

    // すれちがい通信：プロセス向け
    static const Requirement RequirementNeighborDetection =
    {
        191,                                        // _priority
        false,                                      // _isRejectable
        true,                                       // _isPersistent
        true,                                       // _isInstant
        false,                                      // _isSustainable
        true,                                       // _isGreedy
        true,                                       // _isSharable
        false,                                      // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        true,                                       // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_NeighborDetection,              // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Invalid        // _connectionConfirmationOption
    };

    // TODO:
    /***************
     * 内部利用向け
     ***************/

    // 接続先指定（内部利用）
    static const Requirement RequirementSustainConnection =
    {
        159,                                        // _priority
        true,                                       // _isRejectable
        false,                                      // _isPersistent
        false,                                      // _isInstant
        false,                                      // _isSustainable
        false,                                      // _isGreedy
        true,                                       // _isSharable
        false,                                      // _isForeground
        true,                                       // _isAutomaticSwitchProhibited
        false,                                      // _isKeptInSleep
        false,                                      // _isInternalContextUpdatable
        NetworkType_Internet,                       // _networkType
        nn::util::InvalidUuid,                      // _profileId
        ConnectionConfirmationOption_Prohibited     // _connectionConfirmationOption
    };
}

UserRequest::UserRequest(
    RequestManager* pRequestManager,
    RequirementPreset requirementPreset,
    ISignalObject* pSignalObject,
    ClientId clientId,
    nn::Bit64 processId) NN_NOEXCEPT
    : m_pRequestManager(pRequestManager),
      m_pSignalObject(pSignalObject),
      m_ClientId(clientId),
      m_ClientProcessId(processId),
      m_RequestState(RequestState_Free),
      m_Result(nn::nifm::ResultNotInitialized()),   // TODO
      m_LastRevision(0),
      m_SocketDescriptor(-1),
      m_IsTemporaryExcluded(false),
      m_AdditionalInfo()
{
    SetRequirementPreset(requirementPreset);
}

UserRequest::~UserRequest() NN_NOEXCEPT
{
}

nn::Result UserRequest::SetRequirement(const Requirement& requirement) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_RequestState == RequestState_Free, ResultInvalidRequestState());

    NN_RESULT_THROW_UNLESS(requirement._networkType >= NetworkType_None, ResultOutOfRange());
    NN_RESULT_THROW_UNLESS(requirement._networkType < NetworkType_Count, ResultOutOfRange());

    if (requirement._networkType == NetworkType_Internet)
    {
        NN_RESULT_THROW_UNLESS(
            requirement._connectionConfirmationOption >= ConnectionConfirmationOption_Invalid,
            ResultOutOfRange());
        NN_RESULT_THROW_UNLESS(
            requirement._connectionConfirmationOption < ConnectionConfirmationOption_Count,
            ResultOutOfRange());
    }
    else
    {
        NN_RESULT_THROW_UNLESS(
            requirement._connectionConfirmationOption == ConnectionConfirmationOption_Invalid,
            ResultOutOfRange());
    }

    m_Requirement = requirement;

    if (m_Requirement._connectionConfirmationOption == DeprecatedConnectionConfirmationOption_NotRequired)
    {
        // NotRequired の廃止用の処理
        m_Requirement._connectionConfirmationOption = ConnectionConfirmationOption_Prohibited;
    }

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::GetRequirement(Requirement* pOutRequirement) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    *pOutRequirement = m_Requirement;

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::SetRequirementPreset(RequirementPreset requirementPreset) NN_NOEXCEPT
{
    switch( requirementPreset )
    {
    case RequirementPreset_None:
        NN_RESULT_THROW(SetRequirement(RequirementNone));

    case RequirementPreset_InternetGeneric:
        NN_RESULT_THROW(SetRequirement(RequirementApplicationInternetGeneric));

    case RequirementPreset_InternetBestEffort:
        NN_RESULT_THROW(SetRequirement(RequirementApplicationInternetBestEffort));

    case RequirementPreset_LocalGeneric:
        NN_RESULT_THROW(SetRequirement(RequirementApplicationLocalGeneric));

    case RequirementPreset_NeighborDetection:
        NN_RESULT_THROW(SetRequirement(RequirementApplicationNeighborDetectionGeneric));

    case RequirementPreset_InternetForApplet:
        NN_RESULT_THROW(SetRequirement(RequirementAppletGeneric));

    case RequirementPreset_InternetBestEffortForApplet:
        NN_RESULT_THROW(SetRequirement(RequirementAppletInternetBestEffort));

    case RequirementPreset_InternetBestEffortForAppletPersistent:
        NN_RESULT_THROW(SetRequirement(RequirementAppletInternetBestEffortPersistent));

    case RequirementPreset_InternetForNetConnectApplet:
        NN_RESULT_THROW(SetRequirement(RequirementAppletNetworkConnection));

    case RequirementPreset_InternetForWifiWebAuthApplet:
        NN_RESULT_THROW(SetRequirement(RequirementAppletWifiWebAuth));

    case RequirementPreset_LocalForApplet:
        NN_RESULT_THROW(SetRequirement(RequirementApplicationLocalForApplet));

    case RequirementPreset_InternetForAppletContinuous:
        NN_RESULT_THROW(SetRequirement(RequirementAppletContinuous));

    case RequirementPreset_InternetForDevMenu:
        NN_RESULT_THROW(SetRequirement(RequirementAppletDevMenu));

    case RequirementPreset_NeighborDetectionForApplet:
        NN_RESULT_THROW(SetRequirement(RequirementAppletNeighborDetection));

    case RequirementPreset_InternetForSystemProcess:
        NN_RESULT_THROW(SetRequirement(RequirementSystemProcessGeneric));

    case RequirementPreset_InternetForSystemProcessPersistent:
        NN_RESULT_THROW(SetRequirement(RequirementSystemProcessPersistent));

    case RequirementPreset_InternetForSystemProcessContinuous:
        NN_RESULT_THROW(SetRequirement(RequirementSystemProcessContinuous));

    case RequirementPreset_InternetForSystemProcessSharable:
        NN_RESULT_THROW(SetRequirement(RequirementSystemProcessNotGreedySharable));

    case RequirementPreset_InternetForNpns:
        NN_RESULT_THROW(SetRequirement(RequirementNpns));

    case RequirementPreset_NeighborDetectionForSystemProcess:
        NN_RESULT_THROW(SetRequirement(RequirementNeighborDetection));

    case RequirementPreset_Scan:
        NN_RESULT_THROW(SetRequirement(RequirementApplicationPassiveScan));

    case RequirementPreset_ScanForApplet:
        NN_RESULT_THROW(SetRequirement(RequirementAppletPassiveScan));

    case RequirementPreset_SustainConnection:
        NN_RESULT_THROW(SetRequirement(RequirementSustainConnection));

    default:
        NN_DETAIL_NIFM_ERROR("Unknown requirement preset(%d). Set None instead.\n", requirementPreset);
        NN_RESULT_THROW(SetRequirement(RequirementNone));
    }
}

nn::Result UserRequest::SetRequirementByRevision(uint32_t revision) NN_NOEXCEPT
{
    Requirement requirement;
    NN_RESULT_DO(m_pRequestManager->GetRequirement(&requirement, revision));

    // 他の利用要求からの内容をコピーしたものは，テレメトリコンテキスト更新に影響しない
    requirement._isInternalContextUpdatable = false;

    NN_RESULT_THROW(SetRequirement(requirement));
}

nn::Result UserRequest::SetPriority(uint8_t priority) NN_NOEXCEPT
{
    NN_RESULT_THROW(SetRawPriority(ConvertPriorityToRawPriority(priority)));
}

nn::Result UserRequest::SetRawPriority(uint8_t rawPriority) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_RequestState == RequestState_Free, ResultInvalidRequestState());

    m_Requirement._priority = rawPriority;

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::SetNetworkProfileId(const nn::util::Uuid& id) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_RequestState == RequestState_Free, ResultInvalidRequestState());

    m_Requirement._profileId = id;

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::SetRejectable(bool isRejectableRequest) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_RequestState == RequestState_Free, ResultInvalidRequestState());

    m_Requirement._isRejectable = isRejectableRequest;

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::SetConnectionConfirmationOption(ConnectionConfirmationOption connectionConfirmationOption) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_RequestState == RequestState_Free, ResultInvalidRequestState());

    m_Requirement._connectionConfirmationOption =
        connectionConfirmationOption == DeprecatedConnectionConfirmationOption_NotRequired ?
        ConnectionConfirmationOption_Prohibited :   // NotRequired の廃止用の処理
        connectionConfirmationOption;

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::SetPersistent(bool isPersistent) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_RequestState == RequestState_Free, ResultInvalidRequestState());

    m_Requirement._isPersistent = isPersistent;

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::SetInstant(bool isInstant) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_RequestState == RequestState_Free, ResultInvalidRequestState());

    m_Requirement._isInstant = isInstant;

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::SetSustainable(bool isSustainable, uint8_t priority) NN_NOEXCEPT
{
    NN_UNUSED(priority);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_RequestState == RequestState_Free, ResultInvalidRequestState());

    m_Requirement._isSustainable = isSustainable;

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::SetGreedy(bool isGreedy) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_RequestState == RequestState_Free, ResultInvalidRequestState());

    m_Requirement._isGreedy = isGreedy;

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::SetSharable(bool isSharable) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_RequestState == RequestState_Free, ResultInvalidRequestState());

    m_Requirement._isSharable = isSharable;

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::SetKeptInSleep(bool isKeptInSleep) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_RequestState == RequestState_Free, ResultInvalidRequestState());

    m_Requirement._isKeptInSleep = isKeptInSleep;

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::RegisterSocketDescriptor(int socketDescriptor) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_RequestState == RequestState_Accepted, ResultInvalidRequestState());

    // TODO: [TORIAEZU] とりあえず同時に登録できるのは1個まで
    NN_RESULT_THROW_UNLESS(m_SocketDescriptor < 0, ResultOutOfResource());

    m_SocketDescriptor = socketDescriptor;

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::UnregisterSocketDescriptor(int socketDescriptor) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_RequestState == RequestState_Accepted, ResultInvalidRequestState());

    // TODO: [TORIAEZU] とりあえず同時に登録できるのは1個まで
    NN_RESULT_THROW_UNLESS(socketDescriptor == m_SocketDescriptor || socketDescriptor == -1, ResultInvalidArgument());

    m_SocketDescriptor = -1;

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::SetClientId(ClientId clientId) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_ClientId = clientId;

    NN_RESULT_SUCCESS;
}

const Requirement& UserRequest::GetRequirement() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_Requirement;
}

ConnectionConfirmationOption UserRequest::GetPresentConnectionConfirmationOption() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    if (m_RequestState == RequestState_OnHold)
    {
        return m_Requirement._connectionConfirmationOption;
    }

    return ConnectionConfirmationOption_Invalid;
}

nn::Result UserRequest::GetResult() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW(m_Result);
}

RequestState UserRequest::GetRequestState() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_RequestState;
}

uint32_t UserRequest::GetRevision() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_LastRevision;
}

int UserRequest::GetSocketDescriptor() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_SocketDescriptor;
}

Bit64 UserRequest::GetClientProcessId() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_ClientProcessId;
}

ClientId UserRequest::GetClientId() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_ClientId;
}

nn::Result UserRequest::GetAdditionalInfo(AdditionalInfo* pOutAdditionalInfo, uint32_t *pOutRevision) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_SDK_ASSERT_NOT_NULL(pOutAdditionalInfo);
    NN_SDK_ASSERT_NOT_NULL(pOutRevision);

    // 現状では Reject 時にのみ AdditionalInfo を保存している
    NN_RESULT_THROW_UNLESS(m_RequestState == RequestState_Free || (m_RequestState == RequestState_OnHold && m_Requirement._isPersistent)
        , ResultInvalidRequestState());
    NN_SDK_ASSERT(m_Result.IsFailure());

    *pOutRevision = m_LastRevision;
    *pOutAdditionalInfo = m_AdditionalInfo;

    NN_RESULT_SUCCESS;
}

nn::Result UserRequest::Receive(uint32_t lastRevision) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    switch( m_RequestState )
    {
    case RequestState_Free:
        m_RequestState = RequestState_OnHold;
        m_Result = nn::nifm::ResultProcessing();
        m_LastRevision = lastRevision;
        NN_DETAIL_NIFM_INFO("Received: %d\n", m_LastRevision);
        NN_RESULT_SUCCESS;

    case RequestState_OnHold:
    case RequestState_Blocking:
        NN_RESULT_SUCCESS;

    case RequestState_Accepted:
        Signal();
        NN_RESULT_SUCCESS;

    default:
        NN_RESULT_THROW(ResultInvalidRequestState());
    }
}

nn::Result UserRequest::Refuse(nn::Result result) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    switch (m_RequestState)
    {
    case RequestState_Free:
    case RequestState_OnHold:
    case RequestState_Blocking:
    case RequestState_Accepted:
        m_RequestState = RequestState_Free;
        m_Result = result;
        std::memset(&m_AdditionalInfo, 0, sizeof(m_AdditionalInfo));
        Signal();
        NN_DETAIL_NIFM_INFO("Refused(%d-%d): revision:%d\n", result.GetModule(), result.GetDescription(), m_LastRevision);
        NN_RESULT_SUCCESS;

    default:
        NN_RESULT_THROW(ResultInvalidRequestState());
    }
}

nn::Result UserRequest::Reject(nn::Result result) NN_NOEXCEPT
{
    AdditionalInfo additionalInfo = {};
    NN_RESULT_THROW(Reject(result, additionalInfo));
}

nn::Result UserRequest::Reject(nn::Result result, const AdditionalInfo& additionalInfo) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    switch( m_RequestState )
    {
    case RequestState_Free:
    case RequestState_Blocking:
        NN_RESULT_SUCCESS;


    case RequestState_Accepted:
        if (m_Requirement._isRejectable)
        {
            if (m_Requirement._isPersistent)
            {
                m_RequestState = RequestState_OnHold;
                m_IsTemporaryExcluded = true;
                NN_DETAIL_NIFM_INFO("Rejected(%d-%d) -> Receive (Persistent): %d\n", result.GetModule(), result.GetDescription(), m_LastRevision);
            }
            else
            {
                // release ビルド以外では常に，非 Persistent な要求の却下のログを表示
                m_RequestState = RequestState_Free;
                NN_DETAIL_NIFM_TRACE_V3("Rejected(%d-%d): revision:%d\n", result.GetModule(), result.GetDescription(), m_LastRevision);
            }
            std::memset(&m_AdditionalInfo, 0, sizeof(m_AdditionalInfo));
        }
        else
        {
            m_RequestState = RequestState_Blocking;
            NN_DETAIL_NIFM_INFO("Blocked: %d\n", m_LastRevision);
        }

        m_Result = result;
        Signal();
        NN_RESULT_SUCCESS;

    case RequestState_OnHold:
        if (m_IsTemporaryExcluded)
        {
            NN_RESULT_SUCCESS;
        }
        else if (m_Requirement._isPersistent)
        {
            m_IsTemporaryExcluded = true;
            NN_DETAIL_NIFM_INFO("Rejected(%d-%d) -> Receive (Persistent): %d\n", result.GetModule(), result.GetDescription(), m_LastRevision);
        }
        else
        {
            // release ビルド以外では常に，非 Persistent な要求の却下のログを表示
            m_RequestState = RequestState_Free;
            NN_DETAIL_NIFM_TRACE_V3("Rejected(%d-%d): revision:%d\n", result.GetModule(), result.GetDescription(), m_LastRevision);
        }
        m_Result = result;
        m_AdditionalInfo = additionalInfo; // 同時に reject される全ての利用要求に情報が格納される
        Signal();
        NN_RESULT_SUCCESS;

    default:
        NN_RESULT_THROW(ResultInvalidRequestState());
    }
}

nn::Result UserRequest::Accept() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    switch( m_RequestState )
    {
    case RequestState_OnHold:
        m_SocketDescriptor = -1;    // 外部仕様は却下時破棄だが、タイミング依存で登録が残る可能性がある（受理状態とアトミックでないため）ので受理時に上書きする
        m_RequestState = RequestState_Accepted;
        m_Result = nn::ResultSuccess();
        Signal();
        NN_DETAIL_NIFM_INFO("Accepted: %d\n", m_LastRevision);
        NN_RESULT_SUCCESS;

    default:
        NN_RESULT_THROW(ResultInvalidRequestState());
    }
}

bool UserRequest::IsTemporaryExcluded() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_IsTemporaryExcluded;
}

void UserRequest::ResetTemporaryExcludedFlag() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_IsTemporaryExcluded = false;
}

void UserRequest::Dump() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    char connectionConfirmationOptionStr[16] = {};

    switch (m_Requirement._connectionConfirmationOption)
    {
    case nn::nifm::ConnectionConfirmationOption_Prohibited:
        nn::util::SNPrintf(connectionConfirmationOptionStr, sizeof(connectionConfirmationOptionStr), "Prohibited");
        break;
    case nn::nifm::ConnectionConfirmationOption_Preferred:
        nn::util::SNPrintf(connectionConfirmationOptionStr, sizeof(connectionConfirmationOptionStr), "Preferred");
        break;
    case nn::nifm::ConnectionConfirmationOption_Required:
        nn::util::SNPrintf(connectionConfirmationOptionStr, sizeof(connectionConfirmationOptionStr), "Required");
        break;
    case nn::nifm::ConnectionConfirmationOption_Forced:
        nn::util::SNPrintf(connectionConfirmationOptionStr, sizeof(connectionConfirmationOptionStr), "Forced");
        break;
    default:
        nn::util::SNPrintf(connectionConfirmationOptionStr, sizeof(connectionConfirmationOptionStr), "%d", m_Requirement._connectionConfirmationOption);
        break;
    }

    char requestStateStr[16] = {};
    switch (m_RequestState)
    {
    case RequestState_Invalid:
        nn::util::SNPrintf(requestStateStr, sizeof(requestStateStr), "Invalid");
        break;
    case RequestState_Free:
        nn::util::SNPrintf(requestStateStr, sizeof(requestStateStr), "Free");
        break;
    case RequestState_OnHold:
        nn::util::SNPrintf(requestStateStr, sizeof(requestStateStr), "OnHold");
        break;
    case RequestState_Accepted:
        nn::util::SNPrintf(requestStateStr, sizeof(requestStateStr), "Accepted");
        break;
    case RequestState_Blocking:
        nn::util::SNPrintf(requestStateStr, sizeof(requestStateStr), "Blocking");
        break;
    default:
        nn::util::SNPrintf(requestStateStr, sizeof(requestStateStr), "%d", m_RequestState);
        break;
    }

    char networkTypeStr[16];
    switch (m_Requirement._networkType)
    {
    case NetworkType_None:
        nn::util::SNPrintf(networkTypeStr, sizeof(networkTypeStr), "None");
        break;
    case NetworkType_Internet:
        nn::util::SNPrintf(networkTypeStr, sizeof(networkTypeStr), "Internet");
        break;
    case NetworkType_Local:
        nn::util::SNPrintf(networkTypeStr, sizeof(networkTypeStr), "Local");
        break;
    case NetworkType_NeighborDetection:
        nn::util::SNPrintf(networkTypeStr, sizeof(networkTypeStr), "Neighbor");
        break;
    default:
        nn::util::SNPrintf(networkTypeStr, sizeof(networkTypeStr), "%d", m_Requirement._networkType);
        break;
    }

    char uuidString[nn::util::Uuid::StringSize];
    NN_UNUSED(uuidString);

    NN_DETAIL_NIFM_INFO_V1("  revision: %d\n", m_LastRevision);
    NN_DETAIL_NIFM_INFO_V1("  state: %s\n", requestStateStr);
    NN_DETAIL_NIFM_INFO_V1("  excluded: %s\n", m_IsTemporaryExcluded ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("  cid: %d(%s)\n", m_ClientId.value, m_ClientId.IsValid() ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("    requirement priority: %d\n", m_Requirement._priority);
    NN_DETAIL_NIFM_INFO_V1("    requirement rejectable: %s\n", m_Requirement._isRejectable ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("    requirement persistent: %s\n", m_Requirement._isPersistent ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("    requirement instant: %s\n", m_Requirement._isInstant ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("    requirement sustainable: %s\n", m_Requirement._isSustainable ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("    requirement greedy: %s\n", m_Requirement._isGreedy ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("    requirement sharable: %s\n", m_Requirement._isSharable ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("    requirement foreground: %s\n", m_Requirement._isForeground ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("    requirement automaticSwitchProhibited: %s\n", m_Requirement._isAutomaticSwitchProhibited ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("    requirement keptInSleep: %s\n", m_Requirement._isKeptInSleep ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("    requirement internalContextUpdatable: %s\n", m_Requirement._isInternalContextUpdatable ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("    requirement network type: %s\n", networkTypeStr);
    NN_DETAIL_NIFM_INFO_V1("    requirement profile id: %s\n", nn::util::Uuid(m_Requirement._profileId).ToString(uuidString, nn::util::Uuid::StringSize));
    NN_DETAIL_NIFM_INFO_V1("    connectionConfirmationOption: %s\n", connectionConfirmationOptionStr);
}

void UserRequest::Signal() NN_NOEXCEPT
{
    if (m_pSignalObject != nullptr)
    {
        m_pSignalObject->Signal();
    }
}

}
}
}
