﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/nifm_DispatchLoop.h>
#include <nn/nifm/detail/core/nifm_RequestManager.h>
#include <nn/nifm/detail/core/nifm_ConnectionSelector.h>
#include <nn/nifm/detail/core/networkInterface/nifm_EthernetInterfaceManager.h>
#include <nn/nifm/detail/core/nifm_PowerStateCoordinator.h>

#include <nn/nifm/detail/util/nifm_CancelFlagHolder.h>
#include <nn/nifm/detail/util/nifm_DebugUtility.h>

#include <nn/nn_SystemThreadDefinition.h>

#include <nn/bgtc/bgtc_StateControlApi.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nn/srepo/srepo_StateNotifier.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_LockGuard.h>

namespace nn
{
namespace nifm
{
namespace detail
{

void DispatchLoop::DispatchLoopThreadFunction(void* pContext) NN_NOEXCEPT
{
    reinterpret_cast<DispatchLoop*>(pContext)->Dispatch();
}

DispatchLoop::DispatchLoop(
    RequestManager* pRequestManager,
    ConnectionSelector* pConnectionSelector,
    EthernetInterfaceManager* pEthernetInterfaceManager,
    NetworkProfileManager* pNetworkProfileManager,
    PowerStateCoordinator* pPowerStateCoordinator) NN_NOEXCEPT
    : m_TriggerLoopEvent(nn::os::EventClearMode_AutoClear),
      m_TriggerLoopEventHandler(m_TriggerLoopEvent),
      m_pRequestManager(pRequestManager),
      m_pConnectionSelector(pConnectionSelector),
      m_pEthernetInterfaceManager(pEthernetInterfaceManager),
      m_pNetworkProfileManager(pNetworkProfileManager),
      m_pPowerStateCoordinator(pPowerStateCoordinator),
      m_AggregatedRequest(),
      m_TotalNetworkResourceInfo(),
      m_DispatchImpl([this]() NN_NOEXCEPT{ DispatchEthernetInterfaceStandingBy(); }),
      m_IsPartial(false),
      m_IsToExit(false)
{
}

DispatchLoop::~DispatchLoop() NN_NOEXCEPT
{
}

void DispatchLoop::Dispatch() NN_NOEXCEPT
{
    nn::os::MultiWaitType multiWait;

    nn::os::InitializeMultiWait(&multiWait);
    m_TriggerLoopEventHandler.LinkTo(&multiWait);
    m_pRequestManager->GetSubmitEventHandler().LinkTo(&multiWait);
    m_pConnectionSelector->GetScanTimerEventHandler().LinkTo(&multiWait);
    m_pConnectionSelector->GetConnectionEventHandler().LinkTo(&multiWait);
    m_pNetworkProfileManager->GetNetworkProfileEventHandler().LinkTo(&multiWait);
    m_pEthernetInterfaceManager->GetPluggingEventHandler().LinkTo(&multiWait);
    m_pPowerStateCoordinator->GetDispatchTriggerEventHandler().LinkTo(&multiWait);

    m_pConnectionSelector->GetTotalNetworkResourceInfo(&m_TotalNetworkResourceInfo);

    while (NN_STATIC_CONDITION(true))
    {
        nn::os::MultiWaitHolderType *pHolder = nn::os::TimedWaitAny(&multiWait, nn::TimeSpan::FromSeconds(DispatchIntervalInSeconds));

        if (pHolder == nullptr)
        {
            m_pRequestManager->UpdateAggregatedRequest();
        }

        // MultiWait にたまっているすべてのイベントを消化してから進む
        while (pHolder != nullptr)
        {
            if (m_pEthernetInterfaceManager->GetPluggingEventHandler().ExecuteCallbackIfSignaled(pHolder))
            {
                // USB Ether アダプターの挿抜イベントに対応したコールバックが処理された
                NN_DETAIL_NIFM_INFO("Handled Ethernet interface plugging.\n");
            }
            else if (m_pConnectionSelector->GetScanTimerEventHandler().ExecuteCallbackIfSignaled(pHolder))
            {
                // スキャン要求のコールバックが処理された
                NN_DETAIL_NIFM_INFO("Handled scan trigger.\n");
            }
            else if (m_pConnectionSelector->GetConnectionEventHandler().ExecuteCallbackIfSignaled(pHolder))
            {
                // Wlan/Eth/Tsc のイベントに対応したコールバックが処理された
                NN_DETAIL_NIFM_INFO("Handled connection status change.\n");

                // 有線がリンクアップしたら継続要求を取り下げる
                auto pEthernetInterface = m_pEthernetInterfaceManager->Find(pHolder);

                if (pEthernetInterface != nullptr && pEthernetInterface->IsLinkUp())
                {
                    m_pRequestManager->CancelSustainedRequests();
                    TriggerNetworkSwitchIfRequired();
                }
            }
            else if (m_TriggerLoopEventHandler.ExecuteCallbackIfSignaled(pHolder))
            {
                // Dispatch ループを回すためのイベントに対応したコールバックが処理された
                NN_DETAIL_NIFM_INFO("Handled trigger for dispatch loop.\n");
            }
            else if (m_pRequestManager->GetSubmitEventHandler().ExecuteCallbackIfSignaled(pHolder))
            {
                // 利用要求の提出、取り下げのイベントに対応したコールバックが処理された
                NN_DETAIL_NIFM_INFO("Handled request submitting/canceling.\n");

                TriggerNetworkSwitchIfRequired();
            }
            else if (m_pNetworkProfileManager->GetNetworkProfileEventHandler().ExecuteCallbackIfSignaled(pHolder))
            {
                // 接続設定の編集・増減イベントに対応したコールバックが処理された
                NN_DETAIL_NIFM_INFO("Handled network profile modified.\n");

                m_pConnectionSelector->HandleNetworkProfileModification();

                TriggerNetworkSwitchIfRequired();
            }
            else if (m_pPowerStateCoordinator->GetDispatchTriggerEventHandler().ExecuteCallbackIfSignaled(pHolder))
            {
                // スリープ状態が遷移してディスパッチがトリガされた
                NN_DETAIL_NIFM_INFO("Handled trigger by power state reason.\n");
            }

            pHolder = nn::os::TryWaitAny(&multiWait);
        }

        if (IsToExit())
        {
            break;
        }

        m_DispatchImpl();
    }
}   // NOLINT(impl/function_size)

void DispatchLoop::Start() NN_NOEXCEPT
{
    void* pThreadStack = reinterpret_cast<void*>(nn::util::align_up(reinterpret_cast<uintptr_t>(m_ThreadStack), nn::os::StackRegionAlignment));
    Result result = nn::os::CreateThread(&m_ThreadType, DispatchLoopThreadFunction, this, pThreadStack, s_StackSize, NN_SYSTEM_THREAD_PRIORITY(nifm, Dispatch));
    if(result.IsSuccess())
    {
        nn::os::SetThreadNamePointer(&m_ThreadType, NN_SYSTEM_THREAD_NAME(nifm, Dispatch));
        nn::os::StartThread(&m_ThreadType);
    }
}

void DispatchLoop::Stop() NN_NOEXCEPT
{
    SetToExit(true);
    m_TriggerLoopEvent.Signal();

    // Started -> Exited の遷移を待ってから削除される
    nn::os::DestroyThread(&m_ThreadType);
}

void DispatchLoop::SetToExit(bool isToExit) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    m_IsToExit = isToExit;
}

bool DispatchLoop::IsToExit() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    return m_IsToExit;
}

void DispatchLoop::TriggerNetworkSwitchIfRequired() NN_NOEXCEPT
{
    // 自動切換が抑制されていなければ無線から有線に切り替えるために切断する
    if (!m_pRequestManager->IsAutomaticSwitchProhibited() &&
        m_pConnectionSelector->IsNicSwitchingRequired() &&
        m_pNetworkProfileManager->IsAnyEthernetNetworkProfileAvailable())
    {
        NN_DETAIL_NIFM_INFO("Switch from Wireless to Ethernet.\n");
        m_pConnectionSelector->EnsureConnectionFree();     // TODO: [TORIAEZU] 複数の接続を扱うようになったら個別に切る
    }
}

void DispatchLoop::AdjustRequestsAndConnections() NN_NOEXCEPT
{
    Dump(m_AggregatedRequest);
    Dump(*m_pRequestManager);

    AdditionalInfo additionalInfo = {};
    nn::Result result = m_pConnectionSelector->Renew(&additionalInfo, m_AggregatedRequest);

    // TODO: ResultNetworkNotFound の詳細化
    // 有線のプロファイルが指定されていた場合、 USB Ethernet アダプターが見つからないエラーにする
    // 無線のプロファイルが指定されていた場合、無線オフかを確認する
    // Ocean 側で差し替えてもらうことも考える

    m_pConnectionSelector->GetTotalNetworkResourceInfo(&m_TotalNetworkResourceInfo);

    Dump(m_TotalNetworkResourceInfo);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    if (m_TotalNetworkResourceInfo.networkResourceState == NetworkResourceState::Available &&
        m_TotalNetworkResourceInfo.networkType == NetworkType_Internet)
    {
        switch (m_TotalNetworkResourceInfo.networkInterfaceType)
        {
        case NetworkInterfaceType_Ethernet:
            nn::bgtc::SetIsUsingSleepUnsupportedDevices(true);
            break;

        case NetworkInterfaceType_Ieee80211:
        default:
            nn::bgtc::SetIsUsingSleepUnsupportedDevices(false);
            break;
        }
    }
    else
    {
        nn::bgtc::SetIsUsingSleepUnsupportedDevices(false);
    }
#endif

    m_pRequestManager->Renew(m_TotalNetworkResourceInfo, m_AggregatedRequest, additionalInfo, result);

    m_pConnectionSelector->ReleaseNetworkResourceLost(m_TotalNetworkResourceInfo);

    if (m_pEthernetInterfaceManager->Renew())
    {
        m_TriggerLoopEvent.Signal();
    }
}

bool DispatchLoop::BalanceRequestsAndConnections() NN_NOEXCEPT
{
    m_pRequestManager->StageAggregatedRequest(&m_AggregatedRequest);

    TotalNetworkResourceInfo totalNetworkResourceInfoPrev = m_TotalNetworkResourceInfo;

    AdjustRequestsAndConnections();

    AggregatedRequestType aggregatedRequestPrev = m_AggregatedRequest;

    m_pRequestManager->UnstageAggregatedRequest(&m_AggregatedRequest);

    bool isBalanced = true
        && m_AggregatedRequest == aggregatedRequestPrev
        && m_TotalNetworkResourceInfo == totalNetworkResourceInfoPrev
        && m_pRequestManager->IsStable();

    if (isBalanced)
    {
        // TODO:
        // 接続が安定しないうちに 有線LAN が有効になるなどのイベントが起こると，即座には ResetTemporaryExcluded されないので
        // インターネット接続が成功しない事があるが、基本的に persistent なシステムプロセスからの要求なので無視している
        m_pRequestManager->UpdateAggregatedRequest();
    }
    else
    {
        m_TriggerLoopEvent.Signal();
    }

    if(isBalanced)
    {
        // バランスが取れたら、srepoへ接続要求の処理結果を通知する

        bool isNetworkResourceStateAvailable = m_TotalNetworkResourceInfo.networkResourceState == NetworkResourceState::Available;
        bool isInternetAvailabilityConfirmed = m_TotalNetworkResourceInfo.internetAvailability == InternetAvailability_Confirmed;

        // m_pRequestManager->IsProhibitedConnectionConfirmationOptionRequestAvailable() は
        // 排他などコストがかかるので、Internet接続されているときに限り呼び出す
        bool isProhibitedConnectionConfirmationOptionRequestAvailable = true
            && isNetworkResourceStateAvailable
            && m_TotalNetworkResourceInfo.networkType == NetworkType_Internet
            && m_pRequestManager->IsProhibitedConnectionConfirmationOptionRequestAvailable()
            ;

        NotifyNetworkStateToSrepo(
            m_TotalNetworkResourceInfo.networkType,
            isNetworkResourceStateAvailable,
            isInternetAvailabilityConfirmed,
            isProhibitedConnectionConfirmationOptionRequestAvailable);
    }

    return isBalanced;
}

void DispatchLoop::RefreshSsidListNetworkProfile() NN_NOEXCEPT
{
    if (m_TotalNetworkResourceInfo.networkResourceState == NetworkResourceState::Available)
    {
        // SSID リストが接続に使用されている場合はあきらめる
        NetworkProfileData networkProfileData;
        if (m_pConnectionSelector->GetCurrentNetworkProfile(&networkProfileData).IsSuccess() &&
            networkProfileData.networkProfileType == NetworkProfileType::NetworkProfileType_SsidList)
        {
            return;
        }
    }

    m_pNetworkProfileManager->ImportSsidListNetworkProfile();
}

void DispatchLoop::DispatchEthernetInterfaceStandingBy() NN_NOEXCEPT
{
    // 起動・ウェイク直後の有線出現待ち
    // 既に接続がある場合はすぐに抜ける
    // ローカル・すれちがい通信を開始したい場合もすぐに抜ける
    // スリープへの遷移中もすぐに抜ける（＝ウェイク直後にもスリープできる）

    auto goalState = m_pPowerStateCoordinator->GetGoalState();

    if(goalState == PowerStateCoordinator::State::FullAwake && !m_pRequestManager->IsFullAwaked())
    {
        // DispatchHandlingDisconnectionDuringSleep で間に合わなかった場合、ここで FullWakeUp される
        m_pRequestManager->FullWakeUp();
    }

    AggregatedRequestType aggregatedRequest;
    m_pRequestManager->GetAggregatedRequest(&aggregatedRequest);

    if (!m_pEthernetInterfaceManager->IsStandingBy() ||
        (aggregatedRequest.networkType == NetworkType_Local || aggregatedRequest.networkType == NetworkType_NeighborDetection) ||
        m_TotalNetworkResourceInfo.networkResourceState == NetworkResourceState::Available ||
        goalState == PowerStateCoordinator::State::Sleep)
    {
        if (goalState == PowerStateCoordinator::State::MinimumAwake)
        {
            m_DispatchImpl = [this]() NN_NOEXCEPT{ DispatchMinimumAwake(); };

            NN_DETAIL_NIFM_TRACE("State is now DispatchMinimumAwake\n");
        }
        else // Sleep/FullAwake
        {
            m_DispatchImpl = [this]() NN_NOEXCEPT{ DispatchFullAwake(); };

            NN_DETAIL_NIFM_TRACE("State is now DispatchFullAwake\n");
        }

        m_TriggerLoopEvent.Signal();
    }
}

void DispatchLoop::DispatchFullAwake() NN_NOEXCEPT
{
    // アウェイク状態の標準的な挙動
    // 利用要求の状態を接続に反映させるように試み、接続の結果をまた利用要求にフィードバックする

    auto isBalanced = BalanceRequestsAndConnections();

    auto goalState = m_pPowerStateCoordinator->GetGoalState();

    if (goalState == PowerStateCoordinator::State::Sleep && isBalanced)
    {
        if (m_TotalNetworkResourceInfo.networkType == NetworkType_Internet)
        {
            NetworkProfileData networkProfile;
            auto result = m_pNetworkProfileManager->GetNetworkProfileData(&networkProfile, m_TotalNetworkResourceInfo.networkProfileId);
            // NpNs
            m_IsPartial =
                m_TotalNetworkResourceInfo.networkInterfaceType == NetworkInterfaceType_Ieee80211 &&
                m_TotalNetworkResourceInfo.networkResourceState == NetworkResourceState::Available &&
                result.IsSuccess() && networkProfile.networkProfileType == NetworkProfileType_User;
        }
        else
        {
            // NeighborDetection
            m_IsPartial =
                m_TotalNetworkResourceInfo.networkType == NetworkType_NeighborDetection &&
                m_TotalNetworkResourceInfo.networkInterfaceType == NetworkInterfaceType_Ieee80211 &&
                m_TotalNetworkResourceInfo.networkResourceState == NetworkResourceState::Available;
        }

        m_pRequestManager->PutToSleep(m_IsPartial); // これ以降は OnHold な利用要求は保留される

        m_DispatchImpl = [this]() NN_NOEXCEPT{ DispatchPuttingRequestManagerToSleep(); };

        NN_DETAIL_NIFM_TRACE("State is now DispatchPuttingRequestManagerToSleep\n");

        m_TriggerLoopEvent.Signal();
    }
    else if (goalState == PowerStateCoordinator::State::Shutdown)
    {
        m_pRequestManager->Shutdown();

        m_DispatchImpl = [this]() NN_NOEXCEPT{ DispatchPuttingRequestManagerToShutdown(); };

        NN_DETAIL_NIFM_TRACE("State is now DispatchPuttingRequestManagerToShutdown\n");

        m_TriggerLoopEvent.Signal();
    }

    CancelFlagHolder::GetSingleton().ClearCancel();
}

void DispatchLoop::DispatchMinimumAwake() NN_NOEXCEPT
{
    NN_DETAIL_NIFM_TRACE("DispatchMinimumAwake, psc.goal=%d\n", m_pPowerStateCoordinator->GetGoalState());

    // アウェイク状態の標準的な挙動
    // 利用要求の状態を接続に反映させるように試み、接続の結果をまた利用要求にフィードバックする

    auto isBalanced = BalanceRequestsAndConnections();

    auto goalState = m_pPowerStateCoordinator->GetGoalState();

    if (goalState == PowerStateCoordinator::State::FullAwake)
    {
        m_pPowerStateCoordinator->AcknowledgeAwake();

        // これ以降は FG の要求が評価される
        m_pRequestManager->FullWakeUp();

        m_DispatchImpl = [this]() NN_NOEXCEPT{ DispatchFullAwake(); };

        NN_DETAIL_NIFM_TRACE("State is now DispatchFullAwake\n");

        m_TriggerLoopEvent.Signal();
    }
    else if (goalState == PowerStateCoordinator::State::Sleep && isBalanced)
    {
        // 無線接続の場合は、スリープ中も維持する属性の要求は残す
        // ただしユーザー接続設定以外を利用中なら、接続先アクセスポイントのリソースを枯渇させないように切断する

        if (m_TotalNetworkResourceInfo.networkType == NetworkType_Internet)
        {
            NetworkProfileData networkProfile;
            auto result = m_pNetworkProfileManager->GetNetworkProfileData(&networkProfile, m_TotalNetworkResourceInfo.networkProfileId);
            // NpNs
            m_IsPartial =
                m_TotalNetworkResourceInfo.networkInterfaceType == NetworkInterfaceType_Ieee80211 &&
                m_TotalNetworkResourceInfo.networkResourceState == NetworkResourceState::Available &&
                result.IsSuccess() && networkProfile.networkProfileType == NetworkProfileType_User;
        }
        else
        {
            // NeighborDetection
            m_IsPartial =
                m_TotalNetworkResourceInfo.networkType == NetworkType_NeighborDetection &&
                m_TotalNetworkResourceInfo.networkInterfaceType == NetworkInterfaceType_Ieee80211 &&
                m_TotalNetworkResourceInfo.networkResourceState == NetworkResourceState::Available;
        }

        m_pRequestManager->PutToSleep(m_IsPartial); // これ以降は OnHold な利用要求は保留される

        m_DispatchImpl = [this]() NN_NOEXCEPT{ DispatchPuttingRequestManagerToSleep(); };

        NN_DETAIL_NIFM_TRACE("State is now DispatchPuttingRequestManagerToSleep\n");

        m_TriggerLoopEvent.Signal();
    }
    else if (goalState == PowerStateCoordinator::State::Shutdown)
    {
        m_pRequestManager->Shutdown();

        m_DispatchImpl = [this]() NN_NOEXCEPT{ DispatchPuttingRequestManagerToShutdown(); };

        NN_DETAIL_NIFM_TRACE("State is now DispatchPuttingRequestManagerToShutdown\n");

        m_TriggerLoopEvent.Signal();
    }

    CancelFlagHolder::GetSingleton().ClearCancel();
}

void DispatchLoop::DispatchPuttingRequestManagerToSleep() NN_NOEXCEPT
{
    // 受理済みで、残す必要がない利用要求が取り下げられるのを待つ
    // Blocking で粘る要求がいる可能性があるので、ディスパッチを繰り返して待つ

    BalanceRequestsAndConnections();

    if (m_pRequestManager->IsReadyToSleep())
    {
        // 接続が残る（かもしれない）ので、明示的にソケットを切断する
        if (m_IsPartial)
        {
            nn::socket::ShutdownAllSockets(false);
        }

        m_DispatchImpl = [this]() NN_NOEXCEPT{ DispatchWaitingForConnectionSelectorsBalance(); };

        NN_DETAIL_NIFM_TRACE("State is now DispatchWaitingForConnectionSelectorsBalance\n");

        m_TriggerLoopEvent.Signal();
    }

    CancelFlagHolder::GetSingleton().ClearCancel();
}

void DispatchLoop::DispatchWaitingForConnectionSelectorsBalance() NN_NOEXCEPT
{
    // 接続状態が、残すべき要求に見合うまで待つ
    // 残していた要求がクライアント側から取り下げられた場合も、それに追随する
    // 接続が予期せぬ切断で失われた場合は、要求にその状態を反映させる
    // 新規に提出された OnHold な要求は、スリープ中に維持する設定でも反映させない (persistent だと失敗しつづけることがあるため）
    // 状態がいつ変化するかわからないので、いったん釣り合ったと思ったら先のステップへ進んでしまう

    if (BalanceRequestsAndConnections())
    {
        m_DispatchImpl = [this]() NN_NOEXCEPT{ DispatchPuttingNetworkInterfacesToSleep(); };

        SocketInfo socketInfo[ConnectionSelector::SocketInfoCountMax];
        int count = m_pRequestManager->GetSocketInfoToKeepInSleep(socketInfo, NN_ARRAY_SIZE(socketInfo));
        if (count > NN_ARRAY_SIZE(socketInfo))
        {
            NN_DETAIL_NIFM_WARN("The count of registered SocketInfo is too much: %d/%d", count, NN_ARRAY_SIZE(socketInfo));
            count = NN_ARRAY_SIZE(socketInfo);
        }

        m_pConnectionSelector->SetTcpSessionInformation(socketInfo, count);
        m_pConnectionSelector->PutToSleep();

        NN_DETAIL_NIFM_TRACE("State is now DispatchPuttingNetworkInterfacesToSleep\n");
    }

    m_TriggerLoopEvent.Signal();

    CancelFlagHolder::GetSingleton().ClearCancel();
}

void DispatchLoop::DispatchPuttingNetworkInterfacesToSleep() NN_NOEXCEPT
{
    // インターフェースがスリープに入って、接続状態が固定されている
    // 接続を変化させるような操作はすでに禁止されている
    // 固定された接続状態を要求に反映

    AdjustRequestsAndConnections();     // クライアント側からの利用要求の取り下げによって切断が発生しないように統合リクエストは更新しない

    NN_DETAIL_NIFM_INFO("Ready to sleep.\n");
    m_DispatchImpl = [this]() NN_NOEXCEPT{ DispatchReadyToSleep(); };

    m_pEthernetInterfaceManager->PutToSleep();

    m_pPowerStateCoordinator->AcknowledgeSleep();

    NN_DETAIL_NIFM_TRACE("State is now DispatchReadyToSleep\n");

    m_TriggerLoopEvent.Signal();
}

void DispatchLoop::DispatchReadyToSleep() NN_NOEXCEPT
{
    // 接続状態は固定されており、要求が変化しても接続に反映させることはないし、要求に反映させるべき変化も発生しない

    if (m_pEthernetInterfaceManager->Renew())
    {
        m_TriggerLoopEvent.Signal();
    }

    auto goalState = m_pPowerStateCoordinator->GetGoalState();

    if (goalState == PowerStateCoordinator::State::MinimumAwake ||
        goalState == PowerStateCoordinator::State::FullAwake)
    {
        m_DispatchImpl = [this]() NN_NOEXCEPT{ DispatchHandlingDisconnectionDuringSleep(); };

        m_pEthernetInterfaceManager->WakeUp();
        m_pConnectionSelector->WakeUp();

        NN_DETAIL_NIFM_TRACE("State is now DispatchHandlingDisconnectionDuringSleep\n");

        m_TriggerLoopEvent.Signal();
    }
}

void DispatchLoop::DispatchHandlingDisconnectionDuringSleep() NN_NOEXCEPT
{
    // スリープ開始時の接続が（本当はスリープ中に切断されているにも拘らず）ウェイク時にも維持されていると誤解すると有線インターフェースの出現待ちをスキップしてしまう
    // そのため、スリープ中に発生した切断があれば、有線インターフェースの出現待ちの前に処理する

    BalanceRequestsAndConnections();    // RequestManager がまだスリープ状態なので、新しい接続が確立されることはない

    auto goalState = m_pPowerStateCoordinator->GetGoalState();

    if (goalState == PowerStateCoordinator::State::MinimumAwake)
    {
        m_pRequestManager->MinimumWakeUp();
    }
    else if(goalState == PowerStateCoordinator::State::FullAwake)
    {
        m_pRequestManager->FullWakeUp();
    }

    m_pPowerStateCoordinator->AcknowledgeAwake();

    // 起動中の自動更新対象であるSSIDリスト再読み込み
    // RequestManager 起床後の DispatchLoop 処理が回ってこないので、ここで再読み込みしても問題ない
    RefreshSsidListNetworkProfile();

    m_DispatchImpl = [this]() NN_NOEXCEPT{ DispatchEthernetInterfaceStandingBy(); };

    NN_DETAIL_NIFM_TRACE("State is now DispatchEthernetInterfaceStandingBy\n");

    m_TriggerLoopEvent.Signal();
}

void DispatchLoop::DispatchPuttingRequestManagerToShutdown() NN_NOEXCEPT
{
    // 受理済みの利用要求が取り下げられるのを待つ
    // Blocking で粘る要求がいる可能性があるので、ディスパッチを繰り返して待つ

    BalanceRequestsAndConnections();

    if (m_pRequestManager->IsReadyToShutdown())
    {
        nn::socket::ShutdownAllSockets(true);

        m_DispatchImpl = [this]() NN_NOEXCEPT{ DispatchWaitingForConnectionFree(); };

        NN_DETAIL_NIFM_TRACE("State is now DispatchWaitingForConnectionFree\n");
    }

    m_TriggerLoopEvent.Signal();

    CancelFlagHolder::GetSingleton().ClearCancel();
}

void DispatchLoop::DispatchWaitingForConnectionFree() NN_NOEXCEPT
{
    // 接続状態が、要求なしの状態に見合う（つまり切断）まで待つ

    if (BalanceRequestsAndConnections())
    {
        m_pPowerStateCoordinator->AcknowledgeShutdown();

        m_DispatchImpl = [this]() NN_NOEXCEPT{ DispatchReadyToShutdown(); };

        NN_DETAIL_NIFM_TRACE("State is now DispatchReadyToShutdown\n");
    }

    m_TriggerLoopEvent.Signal();

    CancelFlagHolder::GetSingleton().ClearCancel();
}

void DispatchLoop::DispatchReadyToShutdown() NN_NOEXCEPT
{
    // Shutdown を待つのみ

    NN_DETAIL_NIFM_TRACE("Waiting for shutdown\n");
}

void DispatchLoop::NotifyNetworkStateToSrepo(
    NetworkType type,
    bool isNetworkResourceStateAvailable,
    bool isInternetAvailabilityConfirmed,
    bool isProhibitedConnectionConfirmationOptionRequestAvailable) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    if(!isNetworkResourceStateAvailable)
    {
        nn::srepo::NotifyCompletedNetworkRequestChanged(nn::srepo::CompletedNetworkRequestType::None);
        return;
    }

    switch(type)
    {
        case NetworkType_Internet:
            {
                if(isProhibitedConnectionConfirmationOptionRequestAvailable)
                {
                    // ConnectionConfirmationOption_Prohibited なリクエストが受理されている場合、LANモードとみなす
                    // 外部接続できる可能性はあるが、レポートとしては LocalAreaNetwork を優先する
                    nn::srepo::NotifyCompletedNetworkRequestChanged(nn::srepo::CompletedNetworkRequestType::LocalAreaNetwork);
                }
                else if(isInternetAvailabilityConfirmed)
                {
                    nn::srepo::NotifyCompletedNetworkRequestChanged(nn::srepo::CompletedNetworkRequestType::WideAreaNetwork);
                }
                else
                {
                    // 想定外だがフェイルセーフで None に
                    nn::srepo::NotifyCompletedNetworkRequestChanged(nn::srepo::CompletedNetworkRequestType::None);
                }
            }
            break;
        case NetworkType_Local:
            nn::srepo::NotifyCompletedNetworkRequestChanged(nn::srepo::CompletedNetworkRequestType::LocalDirectNetwork);
            break;
        case NetworkType_NeighborDetection:
            nn::srepo::NotifyCompletedNetworkRequestChanged(nn::srepo::CompletedNetworkRequestType::NeighborDetection);
            break;
        default:
            nn::srepo::NotifyCompletedNetworkRequestChanged(nn::srepo::CompletedNetworkRequestType::None);
            break;
    }

#else
    NN_UNUSED(type);
    NN_UNUSED(isNetworkResourceStateAvailable);
    NN_UNUSED(isInternetAvailabilityConfirmed);
    NN_UNUSED(isProhibitedConnectionConfirmationOptionRequestAvailable);
#endif
}

}
}
}
