﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/networkInterface/nifm_EthernetInterfaceManagerBase.win32.h>

#include <nn/util/util_LockGuard.h>

#include <iphlpapi.h>

#include <new>
#include <mutex>


namespace nn
{
namespace nifm
{
namespace detail
{

void EthernetInterfaceManagerBase::InterfaceGroupEventCallback::ExecuteImpl() NN_NOEXCEPT
{
    m_pEthernetInterfaceManagerBase->InterfaceGroupEventCallback();
}

void EthernetInterfaceManagerBase::GetAllMacAddresses( MacAddress* pOutMacAddresses, int* pOutCount, int inCount ) const NN_NOEXCEPT
{
    static const NET_IF_MEDIA_CONNECT_STATE MediaConnectStateOrder[] = { MediaConnectStateConnected, MediaConnectStateDisconnected };

    NN_SDK_ASSERT(pOutMacAddresses != nullptr || inCount == 0);
    NN_SDK_ASSERT_NOT_NULL(pOutCount);

    *pOutCount = 0;

    PMIB_IF_TABLE2 pMibIfTable2;
    if( ::GetIfTable2( &pMibIfTable2 ) == NO_ERROR )
    {
        for( const auto mediaConnectState : MediaConnectStateOrder )
        {
            for( uint32_t i = 0; i < pMibIfTable2->NumEntries; ++i )
            {
                const MIB_IF_ROW2& mibIfRow2 = pMibIfTable2->Table[i];
                if( mibIfRow2.Type == IF_TYPE_ETHERNET_CSMACD &&
                    mibIfRow2.InterfaceAndOperStatusFlags.HardwareInterface == TRUE &&
                    mibIfRow2.MediaConnectState == mediaConnectState )
                {
                    if( *pOutCount < inCount )
                    {
                        MacAddress& macAddress = pOutMacAddresses[*pOutCount];
                        memcpy( macAddress.data, mibIfRow2.PhysicalAddress, MacAddress::Size );
                    }
                    ++*pOutCount;
                }
            }
        }

        ::FreeMibTable( pMibIfTable2 );
    }
}

EthernetInterfaceManagerBase::EthernetInterfaceManagerBase() NN_NOEXCEPT
    : m_Mutex(),
      m_EthernetInterface(),
      m_InterfaceGroupEvent(nn::os::EventClearMode_AutoClear),
      m_InterfaceGroupEventCallback(this),
      m_InterfaceGroupEventHandler(m_InterfaceGroupEvent),
      m_IsEnabled(true),
      m_IsAppended(false)
{
    MacAddress macAddresses[1];
    int32_t outCount;

    GetAllMacAddresses( macAddresses, &outCount, 1 );
    NN_DETAIL_NIFM_INFO("Found %d Ethernet interfaces.\n", outCount);
    if( outCount > 0 )
    {
        NN_DETAIL_NIFM_INFO(
            "Registered %02x:%02x:%02x:%02x:%02x:%02x as default Ethernet interface.\n",
            macAddresses[0].data[0], macAddresses[0].data[1], macAddresses[0].data[2],
            macAddresses[0].data[3], macAddresses[0].data[4], macAddresses[0].data[5]
        );
        m_EthernetInterface.Initialize(macAddresses[0]);
        m_EthernetInterface.SetEnabled(m_IsEnabled);
    }

    m_InterfaceGroupEventHandler.Register(&m_InterfaceGroupEventCallback);

    m_InterfaceGroupEvent.Signal();     // 1回だけコールバックを発生させて、そこで登録をおこなう
}

EthernetInterfaceManagerBase::~EthernetInterfaceManagerBase() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsAppended);   // 派生クラスが RemoveAll していれば通る（していなくても通ることもある）
}

bool EthernetInterfaceManagerBase::Renew() NN_NOEXCEPT
{
    return false;
}

EthernetInterface* EthernetInterfaceManagerBase::Find(const nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    // この関数の呼び出し中はインターフェースの増減が発生する操作をしない約束なので、ロックは不要

    // TODO: [TORIAEZU]
    // Windows ではリンク状態変化のハンドリングを Windows API のコールバックに依存していて
    // EventHandler によるハンドリングを経由していないので、シグナルすることもなければ
    // 何を渡されても見つけることもできない

    NN_UNUSED(pMultiWaitHolder);

    return nullptr;
}

void EthernetInterfaceManagerBase::InterfaceGroupEventCallback() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsAppended);   // 1回しか呼ばれないはず

    NN_UTIL_LOCK_GUARD(m_Mutex);

    if( !m_IsAppended )
    {
        if( m_EthernetInterface.IsAvailable() )
        {
            AppendCallback(m_EthernetInterface);
            m_IsAppended = true;
        }
    }
}

nn::Result EthernetInterfaceManagerBase::SetEnabled(bool isEnabled) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_IsEnabled = isEnabled;

    NN_RESULT_DO(m_EthernetInterface.SetEnabled(isEnabled));

    NN_RESULT_SUCCESS;
}

bool EthernetInterfaceManagerBase::IsEnabled() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_IsEnabled;
}

void EthernetInterfaceManagerBase::RemoveAll() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    if( m_IsAppended )
    {
        RemoveCallback(m_EthernetInterface);
        m_IsAppended = false;
    }
}

bool EthernetInterfaceManagerBase::IsStandingBy() NN_NOEXCEPT
{
    return false;
}

nn::Result EthernetInterfaceManagerBase::PutToSleep() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

nn::Result EthernetInterfaceManagerBase::WakeUp() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

}
}
}
