﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/accessPoint/nifm_AccessPointBase.h>

#include <nn/nifm/detail/core/networkInterface/nifm_NetworkInterfaceBase.h>
#include <nn/nifm/detail/core/profile/nifm_NetworkProfileBase.h>

#include <nn/nifm/detail/core/accessPoint/nifm_WirelessAccessPoint.h>
#include <nn/nifm/detail/core/accessPoint/nifm_LocalAccessPoint.h>
#include <nn/nifm/detail/core/accessPoint/nifm_EthernetAccessPoint.h>


namespace nn
{
namespace nifm
{
namespace detail
{

AccessPointBase::AccessPointBase(NetworkInterfaceBase* pNetworkInterfaceBase) NN_NOEXCEPT
    : m_pNetworkInterfaceBase(pNetworkInterfaceBase),
      m_Timestamp(nn::os::GetSystemTick().ToTimeSpan())
{
}

AccessPointBase::~AccessPointBase() NN_NOEXCEPT
{
}

AccessPointBase::AccessPointBase(const AccessPointBase& accessPointBase) NN_NOEXCEPT
    : m_pNetworkInterfaceBase(accessPointBase.m_pNetworkInterfaceBase),
      m_Timestamp(accessPointBase.m_Timestamp)
{
}

nn::Result AccessPointBase::Connect(
    const NetworkProfileBase& networkProfile,
    const AggregatedRequestType& aggregatedRequest) const NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pNetworkInterfaceBase->Connect(*this, networkProfile, aggregatedRequest));
}

nn::Result AccessPointBase::Disconnect(nn::Result connectionResult) const NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pNetworkInterfaceBase->Disconnect(this, connectionResult));
}

nn::Result AccessPointBase::Release() const NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pNetworkInterfaceBase->Release(this));
}

bool AccessPointBase::IsPossiblyAvailable(const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) const NN_NOEXCEPT
{
    if (GetNetworkType() != aggregatedRequest.networkType)
    {
        // ネットワークの種類が要求と一致しない
        return false;
    }

    if (GetNetworkType() != NetworkType_Internet)
    {
        // インターネット接続以外は設定を必要としないのでダミーと（のみ）マッチ
        return networkProfile.GetType() == NetworkProfileBase::ProfileType::DummyProfile;
    }

    if (aggregatedRequest.profileId != nn::util::InvalidUuid)
    {
        if (aggregatedRequest.profileId != networkProfile.GetId())
        {
            // ID 指定があるのに ID が一致しない
            return false;
        }
    }
    else
    {
        if (!networkProfile.GetAutoConnect())
        {
            // ID 指定がないのに自動接続の対象でない
            return false;
        }
    }

    if (GetNetworkInterface().GetNetworkInterfaceType() != networkProfile.GetNetworkInterfaceType())
    {
        // アクセスポイントと設定の NIC タイプが一致しない
        return false;
    }

    if (aggregatedRequest.isGreedy && !networkProfile.GetLargeCapacity())
    {
        // 大容量通信を必要とする要求は、大容量通信を許可しない接続設定では接続できない
        return false;
    }

    return IsPossiblyAvailableImpl(networkProfile, aggregatedRequest);
}

int AccessPointBase::GetPriority() const NN_NOEXCEPT
{
    return 0;
}

bool AccessPointBase::IsPossiblyAvailableImpl(const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) const NN_NOEXCEPT
{
    NN_UNUSED(networkProfile);
    NN_UNUSED(aggregatedRequest);

    return true;
}

bool AccessPointBase::operator==(const WirelessAccessPoint& accessPoint) const NN_NOEXCEPT
{
    NN_UNUSED(accessPoint);
    return false;
}

bool AccessPointBase::operator==(const LocalAccessPoint& accessPoint) const NN_NOEXCEPT
{
    NN_UNUSED(accessPoint);
    return false;
}

bool AccessPointBase::operator==(const EthernetAccessPoint& accessPoint) const NN_NOEXCEPT
{
    NN_UNUSED(accessPoint);
    return false;
}

}
}
}
