﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>

#include "./ngc_ErrnoT.h"

namespace nn { namespace ngc { namespace detail {

/**
 * @brief   pBv から Rank 用辞書を作成します。
 * @param[out]  pOutLvA     作成先辞書、レベル A(256ビット区切り)
 *                          256 ビットごとに、その位置での Rank1 の結果を保存
 * @param[out]  pOutLvB     作成先辞書、レベル B(32 ビット区切り)
 *                          32 ビットごとに、直前の lv_a からの Rank1 の差分を保存
 * @param[in]   wordCount   ブロック数(1 ブロック = 32 ビット)
 * @param[in]   pBv         作成元ビットベクトル
 */
void BuildRankDictionary(uint32_t* pOutLvA, uint8_t* pOutLvB,
                         unsigned int wordCount, const uint32_t* pBv) NN_NOEXCEPT;
void BuildRankDictionary(uint32_t* pOutLvA, uint8_t* pOutLvB,
                         unsigned int wordCount, const uint64_t* pBv) NN_NOEXCEPT;

/**
 * @brief   Rank を求めます
 *          あらかじめ用意した Rank 辞書を利用して O(1) で求められます
 */
unsigned int CalcRank1(unsigned int pos, const uint32_t* pBv, const uint32_t* pLvA,
                       const uint8_t* pLvB) NN_NOEXCEPT;
unsigned int CalcRank1(unsigned int pos, const uint64_t* pBv, const uint32_t* pLvA,
                       const uint8_t* pLvB) NN_NOEXCEPT;

int SelectPos(uint32_t r, size_t p) NN_NOEXCEPT;
int SelectPos(uint64_t r, size_t p) NN_NOEXCEPT;

/**
 * @brief   Select を求めます
 *          nth 番目の 1 ビットの場所を返します。 nth は 0 から開始します。
 */
int CalcSelect1(unsigned int nth, unsigned int size, const uint32_t* pBv,
                                const uint32_t* pLvA, const uint8_t* pLvB) NN_NOEXCEPT;
int CalcSelect1(unsigned int nth, unsigned int size, const uint64_t* pBv,
                                const uint32_t* pLvA, const uint8_t* pLvB) NN_NOEXCEPT;

int CalcSelect0(unsigned int nth, unsigned int size, const uint32_t* pBv,
                                const uint32_t* pLvA, const uint8_t* pLvB) NN_NOEXCEPT;
int CalcSelect0(unsigned int nth, unsigned int size, const uint64_t* pBv,
                                const uint32_t* pLvA, const uint8_t* pLvB) NN_NOEXCEPT;

template <size_t N, class Derived, class BIT>
class SmartBitmapCrtp
{
protected:
    enum
    {
        BlockSize = 8 * sizeof(BIT),
        WordCount = (N + BlockSize - 1) / BlockSize,
        LvACount = (N + 256 - 1) / 256,
        LvBCount = ((N + BlockSize - 1) / BlockSize + 3) & ~0x3
    };

public:
    SmartBitmapCrtp() NN_NOEXCEPT {}
    unsigned int GetBitVectorSize() const NN_NOEXCEPT
    {
        return N;
    }
    const BIT* GetBitVector() const NN_NOEXCEPT
    {
        return This()->m_Bitmap;
    }
    bool Has(unsigned int idx) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(This()->m_Bitmap);
        if (m_IsDirty)
        {
            this->Build();
        }
        if (idx >= N)
        {
            return false;
        }
        unsigned int blk = idx / BlockSize;
        unsigned int ofs = idx % BlockSize;
        return 0 != (This()->m_Bitmap[blk] & (static_cast<BIT>(1) << ofs));
    }
    bool operator[](const unsigned int idx) const NN_NOEXCEPT
    {
        return this->Has(idx);
    }
    unsigned int Rank1(unsigned int idx) const NN_NOEXCEPT
    {
        if (m_IsDirty)
        {
            this->Build();
        }
        unsigned int pos = idx < N ? idx : N - 1;
        return detail::CalcRank1(pos, This()->m_Bitmap, m_pLvA, m_pLvB);
    }
    unsigned int Rank0(unsigned int idx) const NN_NOEXCEPT
    {
        unsigned int rank1 = this->Rank1(idx);
        return idx + 1 - rank1;
    }
    int Select1(unsigned int nth) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(This()->m_Bitmap);
        if (nth >= this->Rank1(N - 1))
        {
            return -1;
        }
        return detail::CalcSelect1(nth, N, This()->m_Bitmap, m_pLvA, m_pLvB);
    }
    int Select0(unsigned int nth) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(This()->m_Bitmap);
        if (nth >= this->Rank0(N - 1))
        {
            return -1;
        }
        return detail::CalcSelect0(nth, N, This()->m_Bitmap, m_pLvA, m_pLvB);
    }
    bool Set(unsigned int idx) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(This()->m_Bitmap);
        if (idx >= N)
        {
            return false;
        }
        unsigned int blk = idx / BlockSize;
        unsigned int ofs = idx % BlockSize;
        if (!(This()->m_Bitmap[blk] & (static_cast<BIT>(1) << ofs)))
        {
            This()->m_Bitmap[blk] |= static_cast<BIT>(1) << ofs;
            m_IsDirty = true;
        }
        return true;
    }
    bool TurnOn(unsigned int idx) NN_NOEXCEPT
    {
        return this->Set(idx);
    }
    bool Unset(unsigned int idx) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(This()->m_Bitmap);
        if (idx >= N)
        {
            return false;
        }
        unsigned int blk = idx / BlockSize;
        unsigned int ofs = idx % BlockSize;
        if (This()->m_Bitmap[blk] & (static_cast<BIT>(1) << ofs))
        {
            This()->m_Bitmap[blk] &= ~(static_cast<BIT>(1) << ofs);
            m_IsDirty = true;
        }
        return true;
    }
    bool Set(unsigned int idx, bool value) NN_NOEXCEPT
    {
        return value ? this->Set(idx) : this->Unset(idx);
    }
    void Reset() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(This()->m_Bitmap);
        this->ResetBase();
        // cast is for armcc
        MemSet(reinterpret_cast<void*>(&This()->m_Bitmap[0]),
                    0,
                    static_cast<size_t>(WordCount * sizeof(This()->m_Bitmap[0])));
    }

protected:
    void ResetBase() NN_NOEXCEPT
    {
        m_IsDirty = true;
        // cast is for armcc
        MemSet(reinterpret_cast<void*>(&m_pLvA[0]),
                    0,
                    static_cast<size_t>(LvACount * sizeof(m_pLvA[0])));
        MemSet(reinterpret_cast<void*>(&m_pLvB[0]),
                    0,
                    static_cast<size_t>(LvBCount * sizeof(m_pLvB[0])));
    }

private:
    Derived* This() NN_NOEXCEPT
    {
        return static_cast<Derived*>(this);
    }
    const Derived* This() const NN_NOEXCEPT
    {
        return static_cast<const Derived*>(this);
    }
    void Build() const NN_NOEXCEPT
    {
        detail::BuildRankDictionary(WordCount, This()->m_Bitmap, m_pLvA, m_pLvB);
        m_IsDirty = false;
    }
    mutable bool m_IsDirty;
    mutable uint32_t m_pLvA[LvACount];
    mutable uint8_t m_pLvB[LvBCount];
};

#if defined(NN_BUILD_CONFIG_CPU_X64) || defined(NN_OS_CPU_ARM_AARCH64)
template <size_t N, class BIT = uint64_t>
#else
template <size_t N, class BIT = uint32_t>
#endif
class SmartBitmap final : public SmartBitmapCrtp<N, SmartBitmap<N, BIT>, BIT>
{
    typedef SmartBitmapCrtp<N, SmartBitmap<N, BIT>, BIT> CrtpBase;

public:
    SmartBitmap() NN_NOEXCEPT
    {
        this->Reset();
    }

private:
    BIT m_Bitmap[CrtpBase::WordCount];
    friend class SmartBitmapCrtp<N, SmartBitmap<N, BIT>, BIT>;
};

#if defined(NN_BUILD_CONFIG_CPU_X64) || defined(NN_OS_CPU_ARM_AARCH64)
template <size_t N, class BIT = uint64_t>
#else
template <size_t N, class BIT = uint32_t>
#endif
class SmartBitmapPtr final : public SmartBitmapCrtp<N, SmartBitmapPtr<N, BIT>, BIT>
{
    typedef SmartBitmapCrtp<N, SmartBitmapPtr<N, BIT>, BIT> CrtpBase;

public:
    enum
    {
        ArraySize = CrtpBase::WordCount
    };
    SmartBitmapPtr() NN_NOEXCEPT {}
    void Init(BIT* bitmap) NN_NOEXCEPT
    {
        this->ResetBase();
        m_Bitmap = bitmap;
    }

private:
    BIT* m_Bitmap;
    friend class SmartBitmapCrtp<N, SmartBitmapPtr<N, BIT>, BIT>;
};

}}} // nn::ngc::detail
