﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "./ngc_AhoCorasick.h"
#include "./ngc_AhoCorasickPrivate.h"
#include "./ngc_BinaryReader.h"
#include "./ngc_BinaryWriter.h"

namespace nn { namespace ngc { namespace detail {

AhoCorasick::~AhoCorasick() NN_NOEXCEPT
{
    if (m_pAllocator)
    {
        // nlib succinct を無理やり移植した弊害
        NN_SDK_LOG("ERROR: Please call AhoCorasick::ReleaseAllocator() before calling AhopCorasick::~AhoCorasick()\n");
        NN_ABORT();
    }
    else if (m_pPrv)
    {
        delete m_pPrv;
    }
}

bool AhoCorasick::SetAllocator(nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT
{
    if (!pAllocator)
    {
        return false;
    }
    m_pAllocator = pAllocator;
    return true;
}

void AhoCorasick::ReleaseAllocator() NN_NOEXCEPT
{
    if (!m_pAllocator)
    {
        return;
    }
    if (m_pPrv)
    {
        m_pPrv->lenHolder.ReleaseAllocator();
        m_pPrv->reporttreeHolder.ReleaseAllocator();
        m_pPrv->gotoarcHolder.ReleaseAllocator();
        m_pPrv->failuretreeHolder.ReleaseAllocator();

        m_pPrv->lenHolder.~Map();
        m_pPrv->reporttreeHolder.~Map();
        m_pPrv->gotoarcHolder.~GotoArcHolder();
        m_pPrv->failuretreeHolder.~Bp();

        m_pAllocator->Free(m_pPrv);
    }
    m_pAllocator = NULL;
    m_pPrv = NULL;
}

size_t AhoCorasick::MemSize() const NN_NOEXCEPT
{
    size_t l, g, r, f;
    MemSize(&l, &g, &r, &f);
    return l + g + r + f + sizeof(m_pAllocator);
}

void AhoCorasick::MemSize(size_t* pLenStore, size_t* pGotoArc, size_t* pReportArc,
                          size_t* pFailureArc) const NN_NOEXCEPT
{
    if (m_pPrv)
    {
        if (pLenStore)
        {
            *pLenStore = m_pPrv->lenHolder.MemSize();
        }
        if (pGotoArc)
        {
            *pGotoArc = m_pPrv->gotoarcHolder.MemSize();
        }
        if (pReportArc)
        {
            *pReportArc = m_pPrv->reporttreeHolder.MemSize();
        }
        if (pFailureArc)
        {
            *pFailureArc = m_pPrv->failuretreeHolder.MemSize();
        }
    }
    else
    {
        if (pLenStore)
        {
            *pLenStore = 0;
        }
        if (pGotoArc)
        {
            *pGotoArc = 0;
        }
        if (pReportArc)
        {
            *pReportArc = 0;
        }
        if (pFailureArc)
        {
            *pFailureArc = 0;
        }
    }
}

/**
 * @brief           データを検査して検出対象のパターンを検出します。
 * @param[in]       data        検査対象のデータ
 * @param[in]       n           データサイズ
 * @param[in]       callback    パターンが検出された場合に呼び出されるコールバック関数
 * @param[in, out]  pUserObj     ユーザー指定オブジェクトへのポインタ
 */
void AhoCorasick::Match(const void* pData, size_t n, MatchCallback callback,
                        void* pUserObj) NN_NOEXCEPT
{
    if (!m_pPrv)
    {
        return;
    }
    const char* p = reinterpret_cast<const char*>(pData);
    const char* p_end = p + n;
    int nodeId = 0;
    for (; p != p_end; ++p)
    {
        for (int i = 8 - LabelWidth; i >= 0; i -= LabelWidth)
        {
            if (i != 8 - LabelWidth && nodeId == 0)
            {
                // nodeがrootにあって読み込み文字がバイト境界にないときは
                // 先に読み飛ばしてfailしてしまう
                continue;
            }
            char c = static_cast<char>((*p >> i) & LabelMask);
            for (;;)
            {
                int nextId = m_pPrv->gotoarcHolder.Goto(nodeId, c);
                // goto でたどれなければ failure を見る
                if (nextId < 0)
                {
                    if (nodeId == 0)
                    {
                        break;  // rootノードでストップ
                    }
                    int cpos = m_pPrv->failuretreeHolder.ToPos(nodeId);
                    int ppos = m_pPrv->failuretreeHolder.Parent(cpos);
                    NN_SDK_ASSERT(ppos >= 0);
                    if (ppos < 0)
                    {
                        return;
                    }
                    nodeId = m_pPrv->failuretreeHolder.ToNodeId(ppos);
                }
                else
                {
                    nodeId = nextId;
                    break;
                }
            }
        }
        // lenHolderがデータを持っていればTerminalである
        if (callback)
        {
            Map::FindType lenResult = m_pPrv->lenHolder.Find(nodeId);
            if (lenResult.first)
            {
                size_t len = lenResult.second / NumLabelPerByte;
                if (!(*callback)(p - len + 1, p + 1, nodeId, pUserObj))
                {
                    return;
                }
            }
            Map::FindType report = m_pPrv->reporttreeHolder.Find(nodeId);
            while (report.first)
            {
                size_t len = m_pPrv->lenHolder.Find(report.second).second / NumLabelPerByte;
                if (!(*callback)(p - len + 1, p + 1, report.second, pUserObj))
                {
                    return;
                }
                report = m_pPrv->reporttreeHolder.Find(report.second);
            }
        }
    }
}

void AhoCorasick::Reset() NN_NOEXCEPT
{
    if (m_pPrv)
    {
        m_pPrv->lenHolder.Reset();
        m_pPrv->reporttreeHolder.Reset();
        m_pPrv->gotoarcHolder.Reset();
        m_pPrv->failuretreeHolder.Reset();
        if (m_pAllocator)
        {
            ReleaseAllocator();
        }
    }
}

bool AhoCorasick::Export(BinaryWriter* w) const NN_NOEXCEPT
{
    if (!m_pPrv)
    {
        return false;
    }
    if (!m_pPrv->lenHolder.Export(w))
    {
        return false;
    }
    if (!m_pPrv->reporttreeHolder.Export(w))
    {
        return false;
    }
    if (!m_pPrv->gotoarcHolder.Export(w))
    {
        return false;
    }
    if (!m_pPrv->failuretreeHolder.Export(w))
    {
        return false;
    }
    return true;
}

bool AhoCorasick::Import(BinaryReader* r) NN_NOEXCEPT
{
    if (!m_pPrv)
    {
        // アロケータが指定されているなら動的確保でそれを利用する
        if (m_pAllocator)
        {
            void* pTmp = m_pAllocator->Allocate(sizeof(AhoCorasickPrivate));
            m_pPrv = pTmp ? new (pTmp) AhoCorasickPrivate() : NULL;
            if (m_pPrv)
            {
                // 実装クラスの各インスタンスにもアロケータを設定
                m_pPrv->lenHolder.SetAllocator(m_pAllocator);
                m_pPrv->reporttreeHolder.SetAllocator(m_pAllocator);
                m_pPrv->gotoarcHolder.SetAllocator(m_pAllocator);
                m_pPrv->failuretreeHolder.SetAllocator(m_pAllocator);
            }
        }
        else
        {
            m_pPrv = new (std::nothrow) AhoCorasickPrivate();
        }
        if (!m_pPrv)
        {
            return false;
        }
    }
    if (!m_pPrv->lenHolder.Import(r) ||
        !m_pPrv->reporttreeHolder.Import(r) ||
        !m_pPrv->gotoarcHolder.Import(r) ||
        !m_pPrv->failuretreeHolder.Import(r))
    {
        this->Reset();
        return false;
    }
    return true;
}

}}} // nn::ngc::detail
