﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_CharacterEncoding.h>
#include "ngc_Private.h"

namespace nn
{
namespace ngc
{

namespace {

/*!--------------------------------------------------------------------------*
  Name:         CheckNumber

  @param character [in] 調べる対象の UTF-16 文字。

  @brief        数を表す文字かどうかを返します。
  @return 数を表す文字なら ture を、そうでなければ false を返します。
 *---------------------------------------------------------------------------*/
inline bool CheckNumber(uint16_t character)
{
    if ((0x0030 <= character && character <= 0x0039) ||    // 0～9
        (0xFF10 <= character && character <= 0xFF19) ||    // ０～９
        (0x2070 <= character && character <= 0x2079) ||    // ⁰～⁹
        (0x2080 <= character && character <= 0x2089) ||    // ₀～₉
        (0x2160 <= character && character <= 0x216B) ||    // Ⅰ～Ⅻ
        (0x2170 <= character && character <= 0x217B) ||    // ⅰ～ⅻ
        (0x2460 <= character && character <= 0x2473) ||    // ①～⑳
        (0x2474 <= character && character <= 0x2487) ||    // ⑴～⒇
        (0x2488 <= character && character <= 0x249B) ||    // ⒈～⒛
        (0x2776 <= character && character <= 0x277F) ||    // ❶～❿
        (0x2780 <= character && character <= 0x2789) ||    // ➀～➉
        (0x278A <= character && character <= 0x2793) ||    // ➊～➓
        (character == 0x00B2) ||                                // ² (指数表記)
        (character == 0x00B3) ||                                // ³ (指数表記)
        (character == 0x00B9) ||                                // ¹ (指数表記)
        (character == 0x24EA) ||                                // ⓪
                                                                // ⓿ (※一般的なフォントには入っていません)
        (character == 0x24FF))
    {
        return true;
    }

    return false;
}

}

/*!--------------------------------------------------------------------------*
  Name:         CountNumbers

  @param pString [in] 調べる対象のnullptr終端文字列を指定します。

  @brief        文字列に含まれる、数を表す文字の個数を返します。
  @return 指定された文字列に含まれる数を表す文字の個数が返ります。失敗した場合に負の数が返ります。
 *---------------------------------------------------------------------------*/
int CountNumbers( const char* pStringUtf8 ) NN_NOEXCEPT
{
    // 失敗したので-1を返します。
    if ( !pStringUtf8 )
    {
        return -1;
    }

    // 数字のカウントを行います。
    int numberCount = 0;
    uint16_t characterUtf16[2];
    const char* pStr = pStringUtf8;

    while(NN_STATIC_CONDITION(1))
    {
        char characterUtf8[4] = {};
        auto result = nn::util::PickOutCharacterFromUtf8String(characterUtf8, &pStr);
        if (result != nn::util::CharacterEncodingResult_Success)
        {
             return -1;
        }

        // UTF-8の文字を1文字をUTF-16に変換
        result = nn::util::ConvertCharacterUtf8ToUtf16Native(characterUtf16, characterUtf8);
        if (result != nn::util::CharacterEncodingResult_Success)
        {
            return -1;
        }

        auto character = characterUtf16[0];

        if(character == L'\0')
        {
            break;
        }

        // チェック
        if (CheckNumber(character))
        {
            numberCount++;
        }
    }

    return numberCount;
}

/*!--------------------------------------------------------------------------*
  Name:         CountNumbers

  @param pString [in] 調べる対象のnullptr終端文字列を指定します。

  @brief        文字列に含まれる、数を表す文字の個数を返します。
  @return 指定された文字列に含まれる数を表す文字の個数が返ります。失敗した場合に負の数が返ります。
 *---------------------------------------------------------------------------*/
int CountNumbers(const uint16_t* pString) NN_NOEXCEPT
{
    // 失敗したので-1を返します。
    if (!pString)
    {
        return -1;
    }

    // 数字のカウントを行います。
    int numberCount = 0;
    int i = 0;
    while (NN_STATIC_CONDITION(1))
    {
        auto character = pString[i];

        if (character == L'\0')
        {
            break;
        }

        // チェック
        if (CheckNumber(character))
        {
            numberCount++;
        }
        ++i;
    }

    return numberCount;
}

int CountNumbers(const char16_t* pString) NN_NOEXCEPT
{
    return CountNumbers(reinterpret_cast<const uint16_t*>(pString));
}

}   // namespace ngc
}   // namespace nn

