﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/util/util_ResEndian.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nfc/server/core/nfc_CoreManager.h>
#include <nn/arp/arp_Api.h>
#include <nn/mii/mii_StoreDataAccessor.h>
#include <nn/mii/mii_StoreDataContext.h>
#include <nn/mii/mii_Nfp.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include "nfp_Noft2.h"
#include "nfp_BackupDataStream.h"
#include "nfp_Crypto.h"
#include "nfp_Util.h"
#include "nfp_Date.h"
#include <nn/nfc/server/util/nfc_ScopedMutexLock.h>
#include <nn/nfc/server/util/nfc_UtilUtil.h>

namespace nn { namespace nfp {namespace server {

#define NN_NFP_SERVER_SAVE_BEGIN() {                            \
std::unique_ptr<nn::Bit8[]> saveData(new nn::Bit8[Noft2::DataSize]);    \
std::memcpy(saveData.get(), m_LogicalData, Noft2::DataSize)

#define NN_NFP_SERVER_ROLLBACK()                \
std::memcpy(m_LogicalData, saveData.get(), Noft2::DataSize);

#define NN_NFP_SERVER_SAVE_END() }

#define GET_COMMON_BLOCK_DATA() \
reinterpret_cast<CommonBlockData*>(m_LogicalDataNode[StreamRamSystemId].dataStartAddress)

#define GET_LOCKED_BLOCK_DATA() \
reinterpret_cast<const LockedBlockData*>(m_LogicalDataNode[StreamRomId].dataStartAddress)

#define GET_APPLICATION_BLOCK_DATA() \
reinterpret_cast<ApplicationBlockData*>(m_LogicalDataNode[StreamRamAppId].dataStartAddress)

namespace
{

// RAM 内の共有領域用のストリームです。
static const int       StreamRamSystemId               = 0;
static const bool      StreamRamSystemIsReadOnly       = false;
static const uintptr_t StreamRamSystemAddress          = 40;
static const size_t    StreamRamSystemDataSize         = 180;

// ROM 用のストリームです。
static const int       StreamRomId                     = 1;
static const bool      StreamRomIsReadOnly             = true;
static const uintptr_t StreamRomAddress                = 476;
static const size_t    StreamRomDataSize               = 7;

// RAM 内のアプリケーション領域用のストリームです。
static const int       StreamRamAppId                  = 2;
static const bool      StreamRamAppIsReadOnly          = false;
static const uintptr_t StreamRamAppAddress             = 220;
static const size_t    StreamRamAppDataSize            = 216;

const nn::Bit8 Cc[4] = {0xF1, 0x10, 0xFF, 0xEE};
const nn::Bit8 Pack[4] = {0x80, 0x80, 0x00, 0x00};
const nn::Bit8 Cfg0[4] = { 0x00, 0x00, 0x00, 0x04 };
const nn::Bit8 Cfg1[4] = { 0x5F, 0x00, 0x00, 0x00 };
const nn::Bit8 StaticLock[]  = { 0x0F, 0xE0 };
const nn::Bit8 DynamicLock[] = { 0x01, 0x00, 0x0F, 0xBD };
const size_t PageCount = (Noft2::DataSize + (nn::xcd::Type2TagPageSize - 1)) / nn::xcd::Type2TagPageSize;

const nn::Bit64 NxTitleIdExtentionMask = 0x00000000F0000000;

// NTF のヘッダフォーマット
struct NtfHeaderFormatEx
{
    nn::Bit8 headerVersion;         // 0x00
    nn::Bit8 submissionVersion;     // 0x01
    nn::Bit8 reserve1[3];           // 0x02 - 0x04 (NtfHeaderFormat: reserve[0] ~ tagID[0])
    nn::Bit8 productCode[8];        // 0x05 - 0x0C (NtfHeaderFormat: tagID[1] ~ makerCode[0])
    nn::Bit8 reserve2[3];           // 0x0D - 0x0F (NtfHeaderFormat: makerCode[1] ~ makerCode[4])
    nn::Bit8 identifyCode[4];       // 0x10 - 0x13
    nn::Bit8 reserve3[1];           // 0x14
    nn::Bit8 characterId[3];        // 0x15 - 0x17
    nn::Bit8 numberingId[2];        // 0x18 - 0x19
    nn::Bit8 seriesId;              // 0x1A
    nn::Bit8 nfpType;               // 0x1B
    nn::Bit8 reserve4[68];          // 0x1C - 0x5F
    nn::Bit8 hash[32];              // 0x60 - 0x7F
};

// NOFT version.2 の物理フォーマットです。
struct PhysicalData
{
    nn::Bit8 serial[9];
    nn::Bit8 internal[1];
    nn::Bit8 staticLock[2];
    nn::Bit8 cc[4];
    nn::Bit8 activation[1];
    nn::Bit8 systemWriteCounter[2];
    nn::Bit8 nfpVersion[1];
    nn::Bit8 ramData[32];
    nn::Bit8 hmacRom[32];
    nn::Bit8 romData[7];
    nn::Bit8 noftVersion[1];
    nn::Bit8 manufacture[4];
    nn::Bit8 random[32];
    nn::Bit8 hmacAll[32];
    nn::Bit8 common[144];
    nn::Bit8 application[216];
    nn::Bit8 dynamicLock[3];
    nn::Bit8 reserved1[1];
    nn::Bit8 cfg0[4];
    nn::Bit8 cfg1[4];
    nn::Bit8 pwd[4];
    nn::Bit8 pack[2];
    nn::Bit8 reserved2[2];
};

// NOFT version.2 の論理フォーマットです。
struct LogicalData
{
    nn::Bit8 serial2[1];
    nn::Bit8 internal[1];
    nn::Bit8 staticLock[2];
    nn::Bit8 cc[4];
    nn::Bit8 hmacAll[32];
    nn::Bit8 activation[1];
    nn::Bit8 systemWriteCounter[2];
    nn::Bit8 nfpVersion[1];
    nn::Bit8 ramData[32];
    nn::Bit8 common[144];
    nn::Bit8 application[216];
    nn::Bit8 hmacRom[32];
    nn::Bit8 serial1[8];
    nn::Bit8 romData[7];
    nn::Bit8 noftVersion[1];
    nn::Bit8 manufacture[4];
    nn::Bit8 random[32];
    nn::Bit8 dynamicLock[3];
    nn::Bit8 reserved1[1];
    nn::Bit8 cfg0[4];
    nn::Bit8 cfg1[4];
    nn::Bit8 pwd[4];
    nn::Bit8 pack[2];
    nn::Bit8 reserved2[2];
};

// RAM 内の共有領域に書き込まれるデータの内容です。
struct CommonBlockData
{
    nn::Bit8           writeStatus[1];
    nn::Bit8           systemWriteCounter[2];
    nn::Bit8           nfpVersion[1];
    nn::Bit8           regInfoAndLocale[1];
    nn::Bit8           countryCode[1];
    nn::Bit8           moveCounter[2];
    nn::Bit8           regDate[2];
    nn::Bit8           lastWriteDate[2];
    nn::Bit8           platformId[4];
    nn::Bit8           nickname[sizeof(char16_t) * nn::nfp::NicknameLengthMax];
    nn::Bit8           miiData[96];
    nn::Bit8           titleId[8];
    nn::Bit8           writeCounter[2];
    nn::Bit8           accessId[4];
    nn::Bit8           titleIdEx[1];
    nn::Bit8           miiExtVersion[1];
    nn::Bit8           miiExtension[8];
    nn::Bit8           reserved[20];
    nn::Bit8           extCrc[4];
};

// ROM に書き込まれているデータの内容です。
struct LockedBlockData
{
    nn::Bit8  characterId[nn::nfp::CharacterIdSize];
    nn::Bit8  nfpType[1];
    nn::Bit8  numberingId[2];
    nn::Bit8  seriesId[1];
};

// RAM 内のアプリケーション領域です。
typedef nn::Bit8 ApplicationBlockData[StreamRamAppDataSize];

template <typename T>
T GetValueFromTag(const void* pSrc) NN_NOEXCEPT
{
    T unpacked;
    std::memcpy(&unpacked, pSrc, sizeof(T));
    return nn::util::LoadBigEndian<T>(&unpacked);
}

template <typename T>
void SetValueToTag(void* pDst, T value) NN_NOEXCEPT
{
    T tagByteOrderValue;
    nn::util::StoreBigEndian<T>( &tagByteOrderValue, value);
    std::memcpy(pDst, &tagByteOrderValue, sizeof(T));
}

nn::Bit8 GetRegInfo(const CommonBlockData& commonBlockData) NN_NOEXCEPT
{
    return GetValueFromTag<nn::Bit8>(commonBlockData.regInfoAndLocale) >> 4;
}

bool HasApplicationArea(const CommonBlockData& commonBlockData) NN_NOEXCEPT
{
    const nn::Bit8 regInfo = GetRegInfo(commonBlockData);
    return (nn::nfp::RegisterInfoFlags_ApplicationArea & regInfo) != 0;
}

nn::Bit32 GetHardwareId() NN_NOEXCEPT
{
    //端末固有のIDを取得
    const nn::Bit64 terminalId = nn::nfc::server::core::Manager::GetInstance().GetTerminalId();

    //CRC32
    nn::Bit32 hardwareId = Crc32(&terminalId, sizeof(nn::Bit64));

    return hardwareId;
}

/*!
  @brief         オーバフローしないようにカウンタをインクリメントします。
  @param[in]     counter インクリメント前の値です。
  @return        インクリメント値の値です。
*/
uint16_t IncrementCounter(uint16_t counter) NN_NOEXCEPT
{
    return counter < 65535U ? counter + 1 : counter;
}

nn::Bit64 GetUserTitleId(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::arp::ApplicationLaunchProperty prop;
    auto result = nn::arp::GetApplicationLaunchProperty(&prop, service->GetPid());
    if(result.IsFailure())
    {
        prop.id.value = 0x01; //NX
        prop.id.value = (prop.id.value << 56);
    }
    return prop.id.value;
}

void SetTitleId(CommonBlockData* pOutCommonBlockData, nn::Bit64 titleId) NN_NOEXCEPT
{
    SetValueToTag(pOutCommonBlockData->titleIdEx, static_cast<nn::Bit8>((titleId >> 28) & 0x0F));

    nn::Bit64 compatibleTitleId;
    compatibleTitleId = titleId;
    compatibleTitleId &= ~NxTitleIdExtentionMask;
    compatibleTitleId |= (static_cast<nn::Bit64>(0x3) << 28);

    SetValueToTag(pOutCommonBlockData->titleId, compatibleTitleId);
}

bool IsNewTitleId(nn::Bit64 titleId) NN_NOEXCEPT
{
    nn::Bit8 platform = static_cast<nn::Bit8>((titleId >> 56) & 0xFF);
    return (platform != 0x00);
}

nn::Bit64 GetTitleId(const CommonBlockData& commonBlockData) NN_NOEXCEPT
{
    nn::Bit64 titleId = GetValueFromTag<nn::Bit64>(commonBlockData.titleId);

    if(!IsNewTitleId(titleId))
    {
        return titleId;
    }

    nn::Bit8 titleIdEx = GetValueFromTag<nn::Bit8>(commonBlockData.titleIdEx);
    nn::Bit64 uniqueId = (static_cast<nn::Bit64>(titleIdEx & 0x0F) << 28);

    titleId &= ~NxTitleIdExtentionMask;
    titleId |= uniqueId;

    return titleId;
}

nn::Result GetNickname(char* pOutNickname, size_t nicknameSize, const CommonBlockData& commonBlockData)
{
    uint16_t nickname[nn::nfp::NicknameLengthMax + 1];

    for(int i = 0; i < nn::nfp::NicknameLengthMax; i++)
    {
        nickname[i] = GetValueFromTag<uint16_t>(&commonBlockData.nickname[i * sizeof(uint16_t)]);
    }
    nickname[nn::nfp::NicknameLengthMax] = 0;

    std::memset(pOutNickname, 0, sizeof(nicknameSize));
    auto result = nn::util::ConvertStringUtf16NativeToUtf8(pOutNickname,
                                                           static_cast<int>(nicknameSize),
                                                           nickname);
    if(!(result == nn::util::CharacterEncodingResult_Success))
    {
        return nn::nfc::ResultNeedRegister();
    }

    NN_RESULT_SUCCESS;
}

nn::Result SetNickname(CommonBlockData* pOutCommonBlockData, const char* pNickname)
{
    uint16_t nickname[nn::nfp::NicknameLengthMax + 1];
    std::memset(nickname, 0, sizeof(nickname));

    auto result = nn::util::ConvertStringUtf8ToUtf16Native(nickname,
                                                           nn::nfp::NicknameLengthMax + 1,
                                                           pNickname);
    if(!(result == nn::util::CharacterEncodingResult_Success))
    {
        return nn::nfc::ResultInvalidNickname();
    }
    for(int i = 0; i < nn::nfp::NicknameLengthMax; i++)
    {
        SetValueToTag(&pOutCommonBlockData->nickname[i * sizeof(uint16_t)], nickname[i]);
    }

    NN_RESULT_SUCCESS;
}

nn::Bit32 CalculateOldExtCrc(const CommonBlockData& commonBlockData) NN_NOEXCEPT
{
    // 旧 CRC 適用範囲
    size_t dataSize = static_cast<size_t>(reinterpret_cast<uintptr_t>(commonBlockData.extCrc) - reinterpret_cast<uintptr_t>(commonBlockData.miiData));

    return Crc32(commonBlockData.miiData, dataSize);
}

nn::Bit32 CalculateExtCrc(const CommonBlockData& commonBlockData) NN_NOEXCEPT
{
    // CRC 適用範囲(旧 CRC 適用範囲から titleID( 8 byte ) / WriteCounter ( 2 byte ) / AccessID ( 4 byte ) が除かれた)
    size_t dataSize = static_cast<size_t>(reinterpret_cast<uintptr_t>(commonBlockData.extCrc) - reinterpret_cast<uintptr_t>(commonBlockData.miiData)) - sizeof(commonBlockData.titleId) - sizeof(commonBlockData.writeCounter) - sizeof(commonBlockData.accessId);

    std::unique_ptr<nn::Bit8[]> data(new nn::Bit8[dataSize]);
    std::memcpy(data.get(), commonBlockData.miiData, sizeof(commonBlockData.miiData));
    std::memcpy(data.get() + sizeof(commonBlockData.miiData), commonBlockData.titleIdEx, dataSize - sizeof(commonBlockData.miiData));

    return Crc32(data.get(), dataSize);
}

bool ExistMiiExtention(const CommonBlockData& commonBlockData) NN_NOEXCEPT
{
    nn::Bit32 crc = GetValueFromTag<nn::Bit32>(commonBlockData.extCrc);

    // 互換性のため、必要であれば旧 CRC 適用範囲もチェックされる
    return ((crc == CalculateExtCrc(commonBlockData) || crc == CalculateOldExtCrc(commonBlockData))
            && commonBlockData.miiExtVersion[0] == 0x00);
}

nn::Result GetMii(nn::mii::StoreData* pOutStoreData, const CommonBlockData& commonBlockData)
{
    std::unique_ptr<nn::mii::NfpStoreData> nfpStoreData(new nn::mii::NfpStoreData);

    std::memcpy(&nfpStoreData->core, commonBlockData.miiData, sizeof(nfpStoreData->core));
    std::memcpy(&nfpStoreData->extention, commonBlockData.miiExtension, sizeof(nfpStoreData->extention));

    nn::mii::StoreDataContext context;
    if(nn::mii::BuildFromNfpStoreData(pOutStoreData, &context, *nfpStoreData, ExistMiiExtention(commonBlockData)).IsFailure())
    {
        return nn::nfc::ResultNeedRegister();
    }

    NN_RESULT_SUCCESS;
}

nn::Result SetMii(CommonBlockData* pOutCommonBlockData, const nn::mii::StoreData& storeData) NN_NOEXCEPT
{
    std::unique_ptr<nn::mii::NfpStoreData> nfpStoreData(new nn::mii::NfpStoreData);

    nn::mii::StoreDataContext context;
    NN_RESULT_TRY(nn::mii::BuildToNfpStoreData(nfpStoreData.get(), &context, storeData))
        NN_RESULT_CATCH_ALL
        {
            return nn::nfc::ResultInvalidMii();
        }
    NN_RESULT_END_TRY

    std::memcpy(pOutCommonBlockData->miiData, &nfpStoreData->core, sizeof(nfpStoreData->core));
    std::memcpy(pOutCommonBlockData->miiExtension, &nfpStoreData->extention, sizeof(nfpStoreData->extention));
    const uint8_t miiExtVersion = 0x00;
    SetValueToTag(pOutCommonBlockData->miiExtVersion, miiExtVersion);

    NN_RESULT_SUCCESS;
}

void SetExtCrc(CommonBlockData* pOutCommonBlockData) NN_NOEXCEPT
{
    memset(pOutCommonBlockData->reserved, 0x00, sizeof(pOutCommonBlockData->reserved));
    SetValueToTag(pOutCommonBlockData->extCrc, CalculateExtCrc(*pOutCommonBlockData));
}

bool HasRegisterInfo(const CommonBlockData& commonBlockData) NN_NOEXCEPT
{
    const nn::Bit8 regInfo = GetRegInfo(commonBlockData);
    if((nn::nfp::RegisterInfoFlags_CommonArea & regInfo) == 0)
    {
        return false;
    }

    std::unique_ptr<nn::mii::StoreData> storeData(new nn::mii::StoreData);
    std::unique_ptr<char []> nickname(new char[NicknameLengthMax * 4 + 1]);
    if(GetMii(storeData.get(), commonBlockData).IsFailure()
       || GetNickname(&nickname[0], NicknameLengthMax * 4 + 1, commonBlockData).IsFailure())
    {
        //Mii やニックネームが不正な場合は、未登録とみなす
        return false;
    }

    return true;
}

void DecodeCommonInfo(nn::nfp::CommonInfo* pOutCommonInfo,
                      const CommonBlockData& commonBlockData,
                      bool isDebug) NN_NOEXCEPT
{
    std::memset(pOutCommonInfo, 0, sizeof(nn::nfp::CommonInfo));

    nn::Bit16 tagDate;
    if (isDebug || HasRegisterInfo(commonBlockData) || HasApplicationArea(commonBlockData))
    {
        tagDate = GetValueFromTag<nn::Bit16>(commonBlockData.lastWriteDate);
    }
    else
    {
        tagDate = 0;
    }
    Date date(tagDate, !isDebug);
    pOutCommonInfo->lastWriteDate = date.GetNfpDate();

    pOutCommonInfo->writeCounter = GetValueFromTag<uint16_t>(commonBlockData.writeCounter);
    pOutCommonInfo->nfpVersion = GetValueFromTag<uint8_t>(commonBlockData.nfpVersion);
    pOutCommonInfo->applicationAreaSize = static_cast<uint32_t>(StreamRamAppDataSize);
}

void DecodeModelInfo(nn::nfp::ModelInfo* pOutModelInfo,
                     const LockedBlockData& lockBlockData) NN_NOEXCEPT
{
    std::memset(pOutModelInfo, 0, sizeof(nn::nfp::ModelInfo));
    std::memcpy(pOutModelInfo->characterId, lockBlockData.characterId, sizeof(lockBlockData.characterId));
    pOutModelInfo->seriesId = GetValueFromTag<nn::Bit8>(lockBlockData.seriesId);
    pOutModelInfo->numberingId = GetValueFromTag<uint16_t>(lockBlockData.numberingId);
    pOutModelInfo->nfpType = GetValueFromTag<nn::Bit8>(lockBlockData.nfpType);
}

nn::Result DecodeRegisterInfo(nn::nfp::RegisterInfo* pOutRegisterInfo,
                              const CommonBlockData& commonBlockData) NN_NOEXCEPT
{
    std::memset(pOutRegisterInfo, 0, sizeof(nn::nfp::RegisterInfo));

    if (!HasRegisterInfo(commonBlockData))
    {
        NN_RESULT_SUCCESS;
    }

    std::unique_ptr<nn::mii::StoreData> storeData(new nn::mii::StoreData);
    NN_RESULT_DO(GetMii(storeData.get(), commonBlockData));

    nn::mii::StoreDataAccessor accessor(storeData.get());
    accessor.GetCharInfo(&pOutRegisterInfo->miiData);

    NN_RESULT_DO(GetNickname(pOutRegisterInfo->nickname, sizeof(pOutRegisterInfo->nickname), commonBlockData));
    pOutRegisterInfo->fontRegion   = GetValueFromTag<nn::Bit8>(commonBlockData.regInfoAndLocale) & 0x0F;
#if 0 //TODO 現在国データについてreservedにしているため
    pOutRegisterInfo->country      = GetValueFromTag<nn::Bit8>(commonBlockData.countryCode);
#endif
    Date date(GetValueFromTag<nn::Bit16>(commonBlockData.regDate));
    pOutRegisterInfo->registerDate = date.GetNfpDate();

    NN_RESULT_SUCCESS;
}

nn::Result DecodeRegisterInfo(nn::nfp::RegisterInfoPrivate* pOutRegisterInfo,
                              const CommonBlockData& commonBlockData) NN_NOEXCEPT
{
    std::memset(pOutRegisterInfo, 0, sizeof(nn::nfp::RegisterInfoPrivate));

    if (!HasRegisterInfo(commonBlockData))
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(GetMii(&pOutRegisterInfo->miiData, commonBlockData));

    NN_RESULT_DO(GetNickname(pOutRegisterInfo->nickname, sizeof(pOutRegisterInfo->nickname), commonBlockData));
    pOutRegisterInfo->fontRegion   = GetValueFromTag<nn::Bit8>(commonBlockData.regInfoAndLocale) & 0x0F;
#if 0 //TODO 現在国データについてreservedにしているため
    pOutRegisterInfo->country      = GetValueFromTag<nn::Bit8>(commonBlockData.countryCode);
#endif
    Date date(GetValueFromTag<nn::Bit16>(commonBlockData.regDate));
    pOutRegisterInfo->registerDate = date.GetNfpDate();

    NN_RESULT_SUCCESS;
}

void DecodeRegisterInfo(nn::nfp::RegisterInfoDebug* pOutRegisterInfo,
                        const CommonBlockData& commonBlockData) NN_NOEXCEPT
{
    std::memset(pOutRegisterInfo, 0, sizeof(nn::nfp::RegisterInfoDebug));

    std::memcpy(&pOutRegisterInfo->miiDataCore, commonBlockData.miiData, sizeof(pOutRegisterInfo->miiDataCore));
    std::memcpy(&pOutRegisterInfo->miiDataExtention, commonBlockData.miiExtension, sizeof(pOutRegisterInfo->miiDataExtention));
    Date date(GetValueFromTag<nn::Bit16>(commonBlockData.regDate), false);
    pOutRegisterInfo->registerDate = date.GetNfpDate();
    for(int i = 0; i < nn::nfp::NicknameLengthMax; i++)
    {
        pOutRegisterInfo->nickname[i] = GetValueFromTag<char16_t>(&commonBlockData.nickname[i * sizeof(char16_t)]);
    }
    pOutRegisterInfo->nickname[nn::nfp::NicknameLengthMax] = 0;
    pOutRegisterInfo->fontRegion   = GetValueFromTag<nn::Bit8>(commonBlockData.regInfoAndLocale) & 0x0F;
    pOutRegisterInfo->miiExtentionVersion = GetValueFromTag<uint8_t>(commonBlockData.miiExtVersion);
    std::memcpy(pOutRegisterInfo->reserved, commonBlockData.reserved, sizeof(pOutRegisterInfo->reserved));
    pOutRegisterInfo->extentionCrc = GetValueFromTag<nn::Bit32>(commonBlockData.extCrc);
#if 0 //TODO 現在国データについてreservedにしているため
    pOutRegisterInfo->country      = GetValueFromTag<nn::Bit8>(commonBlockData.countryCode);
#endif
}

nn::nfp::PlatformType GetPlatformType(const CommonBlockData& commonBlockData) NN_NOEXCEPT
{
    nn::Bit64 titleId = GetValueFromTag<nn::Bit64>(commonBlockData.titleId);
    nn::Bit8 platformId = static_cast<nn::Bit8>((titleId >> 28) & 0x0F);
    return static_cast<nn::nfp::PlatformType>(platformId);
}

void DecodeAdminInfo(nn::nfp::AdminInfo* pOutAdminInfo, const CommonBlockData& commonBlockData) NN_NOEXCEPT
{
    std::memset(pOutAdminInfo, 0, sizeof(nn::nfp::AdminInfo));
    pOutAdminInfo->moveCounter   = GetValueFromTag<uint16_t>(commonBlockData.moveCounter);
    pOutAdminInfo->registerInfo  = (GetValueFromTag<nn::Bit8>(commonBlockData.regInfoAndLocale) >> 4) & 0x0F;
    if(!HasRegisterInfo(commonBlockData))
    {
        //Mii やニックネームが不正な場合は、未登録とみなす
        pOutAdminInfo->registerInfo &= ~nn::nfp::RegisterInfoFlags_CommonArea;
    }
    pOutAdminInfo->formatVersion = 2;
    pOutAdminInfo->platform      = 0xFF;
    if (HasApplicationArea(commonBlockData))
    {
        pOutAdminInfo->applicationId.value = GetTitleId(commonBlockData);
        pOutAdminInfo->accessId  = GetValueFromTag<nn::Bit32>(commonBlockData.accessId);
        pOutAdminInfo->platform  = static_cast<nn::Bit8>(GetPlatformType(commonBlockData));
    }
}

void DecodeAdminInfo(nn::nfp::AdminInfoDebug* pOutAdminInfo, const CommonBlockData& commonBlockData) NN_NOEXCEPT
{
    std::memset(pOutAdminInfo, 0, sizeof(nn::nfp::AdminInfoDebug));

    pOutAdminInfo->applicationId.value = GetValueFromTag<nn::Bit64>(commonBlockData.titleId);
    pOutAdminInfo->accessId  = GetValueFromTag<nn::Bit32>(commonBlockData.accessId);
    pOutAdminInfo->moveCounter   = GetValueFromTag<uint16_t>(commonBlockData.moveCounter);
    pOutAdminInfo->registerInfo  = (GetValueFromTag<nn::Bit8>(commonBlockData.regInfoAndLocale) >> 4) & 0x0F;
    pOutAdminInfo->formatVersion = 2;
    pOutAdminInfo->platform  = static_cast<nn::Bit8>(GetPlatformType(commonBlockData));
    pOutAdminInfo->applicationIdExt = GetValueFromTag<nn::Bit8>(commonBlockData.titleIdEx);
}

void DecodeSystemInfo(nn::nfp::SystemInfo* pOutSystemInfo, const CommonBlockData& commonBlockData) NN_NOEXCEPT
{
    pOutSystemInfo->writeStatus        = GetValueFromTag<nn::Bit8>(commonBlockData.writeStatus);
    pOutSystemInfo->systemWriteCounter = GetValueFromTag<uint16_t>(commonBlockData.systemWriteCounter);
    pOutSystemInfo->hardwareId         = GetValueFromTag<nn::Bit32>(commonBlockData.platformId);
}

nn::Result CreateCommonBlockData(CommonBlockData* pOutCommonBlockData,
                           const nn::nfp::SystemInfo& systemInfo,
                           const nn::nfp::CommonInfo& commonInfo,
                           const nn::nfp::RegisterInfoDebug& registerInfo,
                           const nn::nfp::AdminInfoDebug& adminInfo) NN_NOEXCEPT
{
    std::memset(pOutCommonBlockData, 0, sizeof(CommonBlockData));

    SetValueToTag(pOutCommonBlockData->writeStatus, systemInfo.writeStatus);
    SetValueToTag(pOutCommonBlockData->systemWriteCounter, systemInfo.systemWriteCounter);
    SetValueToTag(pOutCommonBlockData->nfpVersion, commonInfo.nfpVersion);
    SetValueToTag(pOutCommonBlockData->regInfoAndLocale, static_cast<nn::Bit8>((adminInfo.registerInfo << 4) | registerInfo.fontRegion));
#if 0 //TODO 現在国データについてreservedにしているため
    SetValueToTag(pOutCommonBlockData->countryCode, registerInfo.country);
#endif
    SetValueToTag(pOutCommonBlockData->moveCounter, adminInfo.moveCounter);
    Date registerDate(registerInfo.registerDate, false);
    SetValueToTag(pOutCommonBlockData->regDate, registerDate.GetTagDate());
    Date lastWriteDate(commonInfo.lastWriteDate, false);
    SetValueToTag(pOutCommonBlockData->lastWriteDate, lastWriteDate.GetTagDate());
    SetValueToTag(pOutCommonBlockData->platformId, systemInfo.hardwareId);
    for(int i = 0; i < nn::nfp::NicknameLengthMax; i++)
    {
        SetValueToTag(&pOutCommonBlockData->nickname[i * sizeof(char16_t)], registerInfo.nickname[i]);
    }
    std::memcpy(pOutCommonBlockData->miiData, &registerInfo.miiDataCore, sizeof(pOutCommonBlockData->miiData));
    SetValueToTag(pOutCommonBlockData->titleId, adminInfo.applicationId.value);
    SetValueToTag(pOutCommonBlockData->writeCounter, commonInfo.writeCounter);
    SetValueToTag(pOutCommonBlockData->accessId, adminInfo.accessId);
    SetValueToTag(pOutCommonBlockData->titleIdEx, adminInfo.applicationIdExt);
    SetValueToTag(pOutCommonBlockData->miiExtVersion, registerInfo.miiExtentionVersion);
    std::memcpy(pOutCommonBlockData->miiExtension, &registerInfo.miiDataExtention, sizeof(pOutCommonBlockData->miiExtension));
    std::memcpy(pOutCommonBlockData->reserved, registerInfo.reserved, sizeof(pOutCommonBlockData->reserved));
    SetValueToTag(pOutCommonBlockData->extCrc, registerInfo.extentionCrc);

    NN_RESULT_SUCCESS;
}

#if 0
void Noft2Dump(const void* pTagData, size_t tagDataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(tagDataSize == Noft2::DataSize);
    NN_SDK_LOG("Noft2 Memory Content\n");
    const nn::Bit8* data = reinterpret_cast<const nn::Bit8*>(pTagData);
    NN_UNUSED(data);
    for (size_t i = 0; i < PageCount; ++i)
    {
        NN_SDK_LOG("0x%02X |", i);
        for (size_t j = 0; j < nn::xcd::Type2TagPageSize; ++j)
        {
            NN_SDK_LOG(" %02X", data[i * nn::xcd::Type2TagPageSize + j]);
        }
        NN_SDK_LOG("\n");
    }
}
#endif

void ConvertToLogical(void* pOutLogicalData, size_t logicalDataSize, const void* pPhysicalData, size_t physicalDataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(logicalDataSize == Noft2::DataSize);
    NN_ABORT_UNLESS(physicalDataSize == Noft2::DataSize);

    LogicalData* logical = reinterpret_cast<LogicalData*>(pOutLogicalData);
    const PhysicalData* physical = reinterpret_cast<const PhysicalData*>(pPhysicalData);

    std::memcpy(logical->serial2,     physical->serial + 8,    8);
    std::memcpy(logical->hmacAll,     physical->hmacAll,       32);
    std::memcpy(logical->activation,  physical->activation,    36);
    std::memcpy(logical->common,      physical->common,        360);
    std::memcpy(logical->hmacRom,     physical->hmacRom,       32);
    std::memcpy(logical->serial1,     physical->serial,        8);
    std::memcpy(logical->romData,     physical->romData,       44);
    std::memcpy(logical->dynamicLock, physical->dynamicLock,   20);
}

void ConvertToPhysical(void* pOutPhysicalData, size_t physicalDataSize, const void* pLogicalData, size_t logicalDataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(logicalDataSize == Noft2::DataSize);
    NN_ABORT_UNLESS(physicalDataSize == Noft2::DataSize);

    PhysicalData* physical = static_cast<PhysicalData*>(pOutPhysicalData);
    const LogicalData* logical = static_cast<const LogicalData*>(pLogicalData);

    std::memcpy(physical->serial,      logical->serial1,      8);
    std::memcpy(physical->serial + 8,  logical->serial2,      8);
    std::memcpy(physical->activation,  logical->activation,   36);
    std::memcpy(physical->hmacRom,     logical->hmacRom,      32);
    std::memcpy(physical->romData,     logical->romData,      44);
    std::memcpy(physical->hmacAll,     logical->hmacAll,      32);
    std::memcpy(physical->common,      logical->common,       360);
    std::memcpy(physical->dynamicLock, logical->dynamicLock,  20);
}

void GenerateSeedFromPhysical(void* pOutSeed, size_t seedSize, const void* pPhysicalData, size_t physicalDataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(seedSize == nn::nfp::server::Crypto::PrfSeedSize);
    NN_ABORT_UNLESS(physicalDataSize == Noft2::DataSize);
    nn::Bit8* seed = reinterpret_cast<nn::Bit8*>(pOutSeed);
    const PhysicalData* physical = reinterpret_cast<const PhysicalData*>(pPhysicalData);

    std::memset(seed,      0,                            seedSize);
    std::memcpy(seed,      physical->systemWriteCounter, 2);
    std::memcpy(seed + 16, physical->serial,             8);
    std::memcpy(seed + 24, physical->serial,             8);
    std::memcpy(seed + 32, physical->random,             32);
}

void GenerateSeedFromLogical(void* pOutSeed, size_t seedSize, const void* pLogicalData, size_t logicalDataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(seedSize == nn::nfp::server::Crypto::PrfSeedSize);
    NN_ABORT_UNLESS(logicalDataSize == Noft2::DataSize);

    nn::Bit8* seed = reinterpret_cast<nn::Bit8*>(pOutSeed);
    const LogicalData* logical = reinterpret_cast<const LogicalData*>(pLogicalData);

    std::memset(seed,      0,                           seedSize);
    std::memcpy(seed,      logical->systemWriteCounter, 2);
    std::memcpy(seed + 16, logical->serial1,            8);
    std::memcpy(seed + 24, logical->serial1,            8);
    std::memcpy(seed + 32, logical->random,             32);
}

bool IsHmacRight(const void* pData, size_t dataSize, const void* pHmac, size_t hmacSize, void* pKey, size_t keySize) NN_NOEXCEPT
{
    // 入力データの HMAC を計算して pMac と比較します。
    std::unique_ptr<nn::Bit8> hmac(new nn::Bit8[hmacSize]);
    nn::nfp::server::Crypto::SignHmacSha256(hmac.get(), hmacSize, pData, dataSize, pKey, keySize);
    return (std::memcmp(hmac.get(), pHmac, hmacSize) == 0);
}

/*!
 * @brief          NOFT2 タグの ROM 領域を検証します。
 * @param[in]      pLogicalData           検証の対象となるデータです。
 * @param[in]      logicalDataSize        検証の対象となるデータのサイズです。
 * @param[in]      pSeed                  検証に使用する Seed 値です。
 * @param[in]      seedSize               検証に使用する Seed のサイズです。
 * @retval         ResultSuccess           NOFT2 ROM 領域は正常です。
 * @retval         ResultInvalidRomArea    NOFT2 タグではありません。
 */
nn::Result VerifyRomArea(const void* pLogicalData, size_t logicalDataSize, const void* pSeed, size_t seedSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(logicalDataSize == Noft2::DataSize);

    std::unique_ptr<Keys> keys(new Keys);

    // ROM 用の鍵を生成します。
    nn::nfp::server::Crypto::CalculateKey(keys.get(), pSeed, seedSize, false);

    // ROM の HMAC 検証を行います。
    const LogicalData* logical = reinterpret_cast<const LogicalData*>(pLogicalData);
    if(!IsHmacRight(logical->serial1, 52, logical->hmacRom, sizeof(logical->hmacRom), keys->macKey, sizeof(keys->macKey)))
    {
        return nn::nfc::ResultInvalidRomArea();
    }
    NN_RESULT_SUCCESS;
}

/*!
 * @brief          NOFT2 タグの RAM 領域を復号します。
 * @param[out]     pOutDecryptedLogicalData 復号化したデータの出力先です。
 * @param[in]      decryptedLogicalDataSize 復号化したデータの出力先サイズです。
 * @param[in]      pEncryptedLogicalData    暗号化されているデータです。
 * @param[in]      encryptedLogicalDataSize 暗号化されているデータのサイズです。
 * @param[in]      pSeed                    復号化に使用する Seed 値です。
 * @param[in]      seedSize                 復号化に使用する Seed のサイズです。
 * @retval         ResultSuccess            復号化に成功しました。
 * @retval         ResultNeedFormat         RAM 領域が壊れています。
 */
Result DecryptRamArea(void* pOutDecryptedLogicalData,
                      size_t decryptedLogicalDataSize,
                      const void* pEncryptedLogicalData,
                      size_t encryptedLogicalDataSize,
                      const void* pSeed, size_t seedSize) NN_NOEXCEPT

{
    NN_ABORT_UNLESS(decryptedLogicalDataSize == Noft2::DataSize);
    NN_ABORT_UNLESS(encryptedLogicalDataSize == Noft2::DataSize);

    Result result;
    std::unique_ptr<Keys> keys(new Keys);

    // 暗号化された領域をコピーします。
    std::memcpy(pOutDecryptedLogicalData, pEncryptedLogicalData, Noft2::DataSize);

    // RAM 用の鍵を生成します。
    nn::nfp::server::Crypto::CalculateKey(keys.get(), pSeed, seedSize, true);

    // RAM を復号化します。
    LogicalData* dec = reinterpret_cast<LogicalData*>(pOutDecryptedLogicalData);
    const LogicalData* enc = reinterpret_cast<const LogicalData*>(pEncryptedLogicalData);
    nn::nfp::server::Crypto::EncryptAndDecryptAesCtr(dec->ramData, 416, enc->ramData, 416, keys->secretKey, sizeof(keys->secretKey), keys->iv, sizeof(keys->iv));
    std::memcpy(dec->hmacRom, enc->hmacRom, sizeof(dec->hmacRom));

    // RAM の HMAC 検証を行います。
    if(!IsHmacRight(dec->systemWriteCounter, 479, dec->hmacAll, sizeof(dec->hmacAll), keys->macKey, sizeof(keys->macKey)))
    {
        return nn::nfc::ResultNeedFormat();
    }

    NN_RESULT_SUCCESS;
}

/*!
 * @brief          NOFT2 タグの RAM 領域を暗号化し署名を付けます。
 * @param[out]     pOutEncryptedLogicalData 暗号化されたデータの出力先です。
 * @param[in]      encryptedLogicalDataSize 暗号化されたデータの出力先のサイズです。
 * @param[out,in]  pDecryptedLogicalData    復号化されているデータです。
 * @param[in]      decryptedLogicalDataSize 復号化されているデータのサイズです。
 * @param[in]      pSeed                    復号化に使用する Seed 値です。
 * @param[in]      seedSize                 復号化に使用する Seed のサイズです。
 */
void EncryptAndSignHmac(void* pOutEncryptedLogicalData,
                        size_t encryptedLogicalDataSize,
                        void* pDecryptedLogicalData,
                        size_t decryptedLogicalDataSize,
                        const void* pSeed, size_t seedSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(encryptedLogicalDataSize == Noft2::DataSize);
    NN_ABORT_UNLESS(decryptedLogicalDataSize == Noft2::DataSize);

    std::unique_ptr<Keys> keys(new Keys);

    // 暗号化前の領域をコピーします。
    std::memcpy(pOutEncryptedLogicalData, pDecryptedLogicalData, Noft2::DataSize);

    // RAM 用の鍵を生成します。
    nn::nfp::server::Crypto::CalculateKey(keys.get(), pSeed, seedSize, true);

    // RAM に HMAC を付加します。
    LogicalData* dec = reinterpret_cast<LogicalData*>(pDecryptedLogicalData);
    LogicalData* enc = reinterpret_cast<LogicalData*>(pOutEncryptedLogicalData);
    nn::Bit8 hmac[nn::nfp::server::Crypto::HmacSize];
    nn::nfp::server::Crypto::SignHmacSha256(hmac, sizeof(hmac), dec->systemWriteCounter, 479, keys->macKey, sizeof(keys->macKey));
    std::memcpy(enc->hmacAll, hmac, sizeof(enc->hmacAll));
    std::memcpy(dec->hmacAll, hmac, sizeof(dec->hmacAll));

    // RAM を暗号化します。
    nn::nfp::server::Crypto::EncryptAndDecryptAesCtr(enc->ramData, 416, dec->ramData, 416, keys->secretKey, sizeof(keys->secretKey), keys->iv, sizeof(keys->iv));
    std::memcpy(enc->hmacRom, dec->hmacRom, sizeof(enc->hmacRom));
}

nn::Result ConvertToEncryptedLogicalData(void* pOutEncryptedLogicalData, size_t encryptedLogicalDataSize, const void* pPhysicalData, size_t physicalDataSize) NN_NOEXCEPT
{
    const PhysicalData* physical = reinterpret_cast<const PhysicalData*>(pPhysicalData);

    bool isEnabled;
    if(nn::settings::fwdbg::IsDebugModeEnabled() == false
       || nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "nfp", "not_locked_tag") != sizeof(isEnabled))
    {
        // 市場環境の場合、ロックビットを必ず検証する
        // Key が見つからなかった場合は安全な方法を選択する(ロックビットを検証する)
        isEnabled = false;
    }

    // デバッグモードフラグが有効な場合は、ロックビットを検証しません。
    if(!isEnabled)
    {
        // ロックビットが正しく設定されていない場合には失敗します。
        if (std::memcmp(physical->staticLock,  StaticLock,  sizeof(physical->staticLock))  != 0 ||
            std::memcmp(physical->dynamicLock, DynamicLock, sizeof(physical->dynamicLock)) != 0)
        {
            return nn::nfc::ResultInvalidTag();
        }
    }

    // CFG が正しく設定されていない場合には失敗します。
    if (std::memcmp(physical->cfg0, Cfg0, sizeof(physical->cfg0)) != 0 ||
        std::memcmp(physical->cfg1, Cfg1, sizeof(physical->cfg1)) != 0)
    {
        return nn::nfc::ResultInvalidTag();
    }

    // タグから読み込んだデータを論理アドレスに変換します。
    ConvertToLogical(pOutEncryptedLogicalData, encryptedLogicalDataSize, pPhysicalData, physicalDataSize);

    // ROM 領域の HMAC 検証を行います。検証に失敗した場合は Noft2 ではありません。
    nn::Bit8 seed[nn::nfp::server::Crypto::PrfSeedSize];
    GenerateSeedFromPhysical(seed, sizeof(seed), pPhysicalData, physicalDataSize);
    return VerifyRomArea(pOutEncryptedLogicalData, encryptedLogicalDataSize, seed, sizeof(seed));
}

nn::Result ConvertToDecryptedLogicalData(void* pOutDecryptedLogicalData, size_t decryptedLogicalDataSize, const void* pPhysicalData, size_t physicalDataSize) NN_NOEXCEPT
{
    std::unique_ptr<nn::Bit8[]> encryptedLogicalData(new nn::Bit8[Noft2::DataSize]);
    NN_RESULT_DO(ConvertToEncryptedLogicalData(encryptedLogicalData.get(), decryptedLogicalDataSize, pPhysicalData, physicalDataSize));

    const PhysicalData* physical = reinterpret_cast<const PhysicalData*>(pPhysicalData);

    nn::Bit8 seed[nn::nfp::server::Crypto::PrfSeedSize];
    GenerateSeedFromPhysical(seed, sizeof(seed), pPhysicalData, physicalDataSize);

    // 前回の書き込みが完全に完了していない場合にはフォーマットが必要です。
    if (physical->activation[0] != 0xA5)
    {
        return nn::nfc::ResultNeedFormat();
    }

    // RAM 領域を復号化して HMAC 検証を行います。検証に失敗した場合はタグが壊れています。
    NN_RESULT_DO(DecryptRamArea(pOutDecryptedLogicalData,
                                decryptedLogicalDataSize,
                                encryptedLogicalData.get(),
                                Noft2::DataSize,
                                seed, sizeof(seed)));

#if 0
    // 復号化したデータをダンプします。
    Noft2Dump(pOutDecryptedLogicalData, Noft2::DataSize);
#endif

    NN_RESULT_SUCCESS;
}

void BreakPhysicalData(void* pPhysicalData, size_t physicalDataSize, nn::nfp::BreakType breakType) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(physicalDataSize == Noft2::DataSize);

    // 指定された方法でタグに書き込むデータを破壊します。
    PhysicalData* physical = reinterpret_cast<PhysicalData*>(pPhysicalData);
    switch (breakType)
    {
    case nn::nfp::BreakType_Activation:
        physical->activation[0] = 0x00;
        break;
    case nn::nfp::BreakType_Hmac:
        physical->hmacAll[0] ^= 0x80;
        break;
    case nn::nfp::BreakType_None:
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

nn::Result CheckNTagId(nn::nfc::TagId& tagId, const void* pPhysicalData, size_t physicalDataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(physicalDataSize == Noft2::DataSize);

    // UID/serial number の一致判定
    nn::Bit8 bcc = 0x88;
    nn::Bit8 serial[9];

    int i = 0;
    for (; i < 3; i++)
    {
        serial[i] = static_cast<nn::Bit8>(tagId.uid[i]);
        bcc ^= serial[i];
    }
    serial[i] = bcc;

    bcc = 0x00;
    for (; i < 7; i++)
    {
        serial[i + 1] = static_cast<nn::Bit8>(tagId.uid[i]);
        bcc ^= serial[i + 1];
    }
    serial[i + 1] = bcc;

    const PhysicalData* physical = reinterpret_cast<const PhysicalData*>(pPhysicalData);
    if (std::memcmp(serial, physical->serial, sizeof(serial)) != 0)
    {
        return nn::nfc::ResultNeedRetry();
    }

    NN_RESULT_SUCCESS;
}

nn::Result CheckNoftVersion(void* pPhysicalData, size_t physicalDataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(physicalDataSize == Noft2::DataSize);

    PhysicalData* physical = reinterpret_cast<PhysicalData*>(pPhysicalData);
    if(physical->noftVersion[0] != 0x02)
    {
        return nn::nfc::ResultInvalidFormatVersion();
    }

    NN_RESULT_SUCCESS;
}

void GetLockPageTable(bool* pOutLockPageTable, size_t lockPageCount, const void* pPhysicalData, size_t physicalDataSize)
{
    NN_ABORT_UNLESS(lockPageCount == PageCount);
    NN_ABORT_UNLESS(physicalDataSize == Noft2::DataSize);

    const PhysicalData* physical = reinterpret_cast<const PhysicalData*>(pPhysicalData);
    nn::Bit8 staticLock[2];
    nn::Bit8 dynamicLock;
    std::memcpy(staticLock, physical->staticLock, sizeof(staticLock));
    std::memcpy(&dynamicLock, physical->dynamicLock, sizeof(dynamicLock));

    //page 0 ~ 3
    for(auto i = 0; i < 4; ++i)
    {
        pOutLockPageTable[i] = 0;
    }
    //page 4 ~ 7
    for(auto i = 0; i < 4; ++i)
    {
        pOutLockPageTable[i + 4] = (staticLock[0] >> (i + 4)) & 0x01;
    }
    //page 8 ~ 15
    for(auto i = 0; i < 8; ++i)
    {
        pOutLockPageTable[i + 8] = (staticLock[1] >> i) & 0x01;
    }

    //page 16 ~ 129
    {
        for(auto i = 0; i < 8; ++i)
        {
            for(auto j = 0; j < 16; ++j)
            {
                auto page = 16 * (i + 1) + j;
                if(page <= 129 )
                {
                    pOutLockPageTable[page] = (dynamicLock >> i) & 0x01;
                }
                else
                {
                    break;
                }
            }
        }
    }
}

nn::Result DecryptNtfAndHashCheck(void* pOutDecryptedNtf, size_t decryptedNtfSize, const void* encryptedNtf, size_t encryptedNtfSize)
{
    NN_ABORT_UNLESS(decryptedNtfSize == Noft2::DataSize);
    NN_ABORT_UNLESS(encryptedNtfSize == 672);

    const nn::Bit8* p = reinterpret_cast<const nn::Bit8*>(encryptedNtf) + sizeof(NtfHeaderFormatEx);
    const NtfHeaderFormatEx* header = reinterpret_cast<const NtfHeaderFormatEx*>(encryptedNtf);

    nn::nfp::server::Crypto::DecryptNtf(pOutDecryptedNtf, 480, p, 480);
    {
        std::unique_ptr<nn::Bit8 []> temp(new nn::Bit8[96]);
        nn::nfp::server::Crypto::DecryptNtf(&temp[0], 96, p + 448, 96);
        std::memcpy(reinterpret_cast<nn::Bit8*>(pOutDecryptedNtf) + 480, &temp[32], 60);
    }
    nn::Bit8 hash[32];
    nn::nfp::server::Crypto::CalculateSha256(&hash, sizeof(hash), pOutDecryptedNtf, decryptedNtfSize);
    if(std::memcmp(header->hash, hash, sizeof(hash)) != 0)
    {
        return nn::nfc::ResultInvalidFormat();
    }

    NN_RESULT_SUCCESS;
}

void CreateUserMemory(void* pOutPhysicalData, size_t physicalDataSize, const void* decryptedNtf, size_t decryptedNtfSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(physicalDataSize == Noft2::DataSize);
    NN_ABORT_UNLESS(decryptedNtfSize == Noft2::DataSize);

    std::unique_ptr<LogicalData> decryptedLogicalData(new LogicalData());
    ConvertToLogical(decryptedLogicalData.get(), Noft2::DataSize, decryptedNtf, decryptedNtfSize);

    decryptedLogicalData->activation[0] = 0xA5;
    decryptedLogicalData->systemWriteCounter[0] = 0x00;
    decryptedLogicalData->systemWriteCounter[1] = 0x00;
    decryptedLogicalData->nfpVersion[0] = 0x00;

    // ランダム値の埋め込み
    FillRandom(decryptedLogicalData->random, sizeof(decryptedLogicalData->random));

    std::unique_ptr<nn::Bit8[]> seed(new nn::Bit8[nn::nfp::server::Crypto::PrfSeedSize]);
    GenerateSeedFromLogical(&seed[0], nn::nfp::server::Crypto::PrfSeedSize, decryptedLogicalData.get(), Noft2::DataSize);

    {
        std::unique_ptr<Keys> keys(new Keys);

        // ROM 用の鍵を生成します。
        nn::nfp::server::Crypto::CalculateKey(keys.get(), seed.get(), nn::nfp::server::Crypto::PrfSeedSize, false);
        // ROM HMAC を付加します。
        nn::nfp::server::Crypto::SignHmacSha256(decryptedLogicalData->hmacRom, sizeof(decryptedLogicalData->hmacRom), decryptedLogicalData->serial1, 52, keys->macKey, sizeof(keys->macKey));
    }

    std::unique_ptr<LogicalData> encryptedLogicalData(new LogicalData);
    EncryptAndSignHmac(encryptedLogicalData.get(), Noft2::DataSize, decryptedLogicalData.get(), Noft2::DataSize, seed.get(), nn::nfp::server::Crypto::PrfSeedSize);

    // 物理アドレスに変換します。
    ConvertToPhysical(pOutPhysicalData, physicalDataSize, encryptedLogicalData.get(), Noft2::DataSize);
}

}

Noft2::Noft2(const nn::nfc::DeviceHandle& deviceHandle, const nn::nfc::TagId& id, nn::os::SystemEventType* accessFinishEvent, nn::os::SystemEventType* accessResetEvent) NN_NOEXCEPT : Ntag(deviceHandle, id, accessFinishEvent, accessResetEvent), m_State(Noft2::State_Init), m_IsUpdatedAppBlock(false), m_IsCreatedRegisterInfo(false), m_IsOpenedApplicationArea(false)
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    std::memset(m_LogicalDataNode, 0, sizeof(m_LogicalDataNode));
}

Noft2::~Noft2() NN_NOEXCEPT
{
};

nn::Result Noft2::ReadPhysicalData(void* pOutPhysicalData, nn::nfc::server::core::Service* service, size_t physicalDataSize) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    //全データを読み出す
    nn::nfc::server::core::NtagReadParameter ntagReadParameter = {};
    ntagReadParameter.timeoutMsec = 2000;
    ntagReadParameter.isPasswordRequired = true;
    ntagReadParameter.tagId.length = 0;
    ntagReadParameter.blockCount = static_cast<int>(nn::util::DivideUp(physicalDataSize, nn::xcd::NtagReadBlockSizeMax));
    auto* addresses = ntagReadParameter.addresses;
    for(auto block = 0; block < ntagReadParameter.blockCount; ++block)
    {
        addresses[block].startPage = 0x00 + static_cast<uint8_t>(nn::xcd::NtagReadBlockPageCountMax * block);
        if(block < ntagReadParameter.blockCount - 1){
            addresses[block].endPage   = addresses[block].startPage + static_cast<uint8_t>(nn::xcd::NtagReadBlockPageCountMax - 1);
        }
        else
        {
            //最終ブロック
            addresses[block].endPage   = static_cast<uint8_t>(nn::util::DivideUp(physicalDataSize, nn::xcd::Type2TagPageSize) - 1);
        }
    }

    std::unique_ptr<nn::nfc::server::core::NtagData> ntagData(new nn::nfc::server::core::NtagData);
    //NTAG210,212,213 は存在しないページを読み込もうとするため失敗(ResultNeedRetry)します。(SIGLO-68098, SIGLO-68454)
    NN_RESULT_DO(Read(ntagData.get(), service, ntagReadParameter));
    nn::Bit8* data = reinterpret_cast<nn::Bit8*>(pOutPhysicalData);
    for(auto i = 0; i < ntagData->blockCount; ++i)
    {
        const auto& block = ntagData->readDataBlocks[i];
        auto dataSize = (block.address.endPage - block.address.startPage + 1) * nn::xcd::Type2TagPageSize;
        std::memcpy(data, block.data, dataSize);
        data += dataSize;
    }

    if(ntagData->type2TagVersion != nn::xcd::NfcType2TagVersion_Ntag215)
    {
        //NTAG215 でない場合は
        //dynamic lock bytes から始まるラスト 5 ページ分は想定通りの値が入っているものとみなして処理を継続する。
        auto physical = reinterpret_cast<PhysicalData*>(pOutPhysicalData);
        std::memcpy(physical->dynamicLock, DynamicLock, sizeof(DynamicLock));
        std::memcpy(physical->cfg0, Cfg0, sizeof(Cfg0));
        std::memcpy(physical->cfg1, Cfg1, sizeof(Cfg1));
        std::memset(physical->pwd, 0x00, sizeof(physical->pwd) + sizeof(physical->pack) + sizeof(physical->reserved2));
    }
    NN_RESULT_DO(CheckNTagId(m_Id, pOutPhysicalData, physicalDataSize));

    NN_RESULT_SUCCESS;
}

nn::Result Noft2::WritePhysicalData(nn::nfc::server::core::Service* service, const void* pPhysicalData, size_t physicalDataSize, bool readyToWrite) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(physicalDataSize == Noft2::DataSize);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(readyToWrite)
    {
        //同一セッション中に一度 Read しておかないと書き込みできないため
        NN_RESULT_DO(ReadyToWrite(service, true));
    }

    const PhysicalData* physical = reinterpret_cast<const PhysicalData*>(pPhysicalData);

    std::unique_ptr<bool[]> lockPageTable(new bool[PageCount]);
    GetLockPageTable(lockPageTable.get(), PageCount, pPhysicalData, physicalDataSize);

    std::unique_ptr<nn::nfc::server::core::NtagWriteParameter> ntagWriteParameter(new nn::nfc::server::core::NtagWriteParameter);

    ntagWriteParameter->timeoutMsec = 2000;
    ntagWriteParameter->isPasswordRequired = true;
    ntagWriteParameter->tagId = m_Id;
    ntagWriteParameter->type2TagVersion = m_Type2TagVersion;
    ntagWriteParameter->ntagWriteData.isActivationNeeded = true;

    //Activation クリア時設定
    std::memset(ntagWriteParameter->ntagWriteData.clearData, 0xFF, nn::xcd::NtagActivationAreaSize);
    //Activation 時設定
    std::memcpy(&ntagWriteParameter->ntagWriteData.activationData, &physical->activation[0], nn::xcd::NtagActivationAreaSize);

    const uint8_t startPage = 0x05; //0x05ページ以降を書き込み
    const uint8_t endPage = PageCount - 5 - 1; //最後の5ページは書き込まない
    uint8_t targetPage = startPage;
    while(targetPage <= endPage)
    {
        auto* dataBlocks = ntagWriteParameter->ntagWriteData.dataBlocks;

        int blockCount = 0;
        while(blockCount < nn::xcd::NtagWriteBlockCountMax)
        {
            int pageCount = 0;
            while(targetPage <= endPage && pageCount < nn::xcd::NtagWriteBlockPageCountMax)
            {
                if(!lockPageTable[targetPage])
                {
                    if(pageCount == 0)
                    {
                        dataBlocks[blockCount].startPageAddress = targetPage;
                    }
                    ++pageCount;
                }
                else
                {
                    if(pageCount != 0)
                    {
                        break;
                    }
                }
                ++targetPage;
            }

            if(pageCount == 0)
            {
                //1ページも書き込むページがない
                break;
            }

            dataBlocks[blockCount].dataSize = static_cast<uint8_t>(nn::xcd::Type2TagPageSize * pageCount);
            const nn::Bit8* data = reinterpret_cast<const nn::Bit8*>(pPhysicalData);
            std::memcpy(dataBlocks[blockCount].data, &data[dataBlocks[blockCount].startPageAddress * nn::xcd::Type2TagPageSize], dataBlocks[blockCount].dataSize);
            ++blockCount;
        }
        if(blockCount == 0)
        {
            //1ブロックも書き込む必要がない
            break;
        }

        ntagWriteParameter->ntagWriteData.blockCount = blockCount;
        NN_RESULT_DO(Write(service, *ntagWriteParameter));
    }

    NN_RESULT_SUCCESS;
}

nn::Result Noft2::Mount(nn::nfc::server::core::Service* service, nn::nfp::MountTarget mountTarget) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES((mountTarget & (nn::nfp::MountTarget_Rom | nn::nfp::MountTarget_Ram)) != 0);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    std::unique_ptr<nn::Bit8[]> physical(new nn::Bit8[Noft2::DataSize]);
    NN_RESULT_DO(ReadPhysicalData(physical.get(), service, Noft2::DataSize));
    NN_RESULT_DO(CheckNoftVersion(physical.get(), Noft2::DataSize));

    if(mountTarget == nn::nfp::MountTarget_Rom)
    {
        NN_RESULT_DO(ConvertToEncryptedLogicalData(m_LogicalData, sizeof(m_LogicalData), physical.get(), Noft2::DataSize));
        // 0 は NFP RAM の共有領域ですが利用できません。
        std::memset(m_LogicalDataNode, 0, sizeof(m_LogicalDataNode));
        uintptr_t base = reinterpret_cast<uintptr_t>(m_LogicalData);
        int  i    = StreamRamSystemId;
        m_LogicalDataNode[i].id               = StreamRamSystemId;

        // 1 は NFP ROM です。
        i = StreamRomId;
        m_LogicalDataNode[i].id               = StreamRomId;
        m_LogicalDataNode[i].isReadOnly       = StreamRomIsReadOnly;
        m_LogicalDataNode[i].isShort          = true;
        m_LogicalDataNode[i].dataSize         = StreamRomDataSize;
        m_LogicalDataNode[i].dataStartAddress = base + StreamRomAddress;

        // 2 は NFP RAM のアプリケーション領域ですが利用できません。
        i = StreamRamAppId;
        m_LogicalDataNode[i].id               = StreamRamAppId;
    }
    else if(mountTarget & nn::nfp::MountTarget_Ram)
    {
        NN_RESULT_DO(BackupDataStream::GetInstance()->Initialize());
        NN_UTIL_SCOPE_EXIT
        {
            BackupDataStream::GetInstance()->Finalize();
        };

        nn::Result result;
        result = ConvertToDecryptedLogicalData(m_LogicalData, sizeof(m_LogicalData), physical.get(), Noft2::DataSize);
        if(nn::nfc::ResultNeedFormat::Includes(result) && BackupDataStream::GetInstance()->HasBackupData(m_Id))
        {
            // バックアップデータが破損していないことを確認します。
            result = BackupDataStream::GetInstance()->Read(physical.get(), Noft2::DataSize, m_Id);
            if(result.IsFailure())
            {
                if(nn::nfc::ResultBackupError::Includes(result))
                {
                    return nn::nfc::ResultNeedFormat();
                }

                // 致命的なエラー
                return result;
            }

            std::unique_ptr<nn::Bit8[]> decryptedLogical(new nn::Bit8[Noft2::DataSize]);
            result = ConvertToDecryptedLogicalData(decryptedLogical.get(), Noft2::DataSize, physical.get(), Noft2::DataSize);
            if (result.IsFailure())
            {
                return nn::nfc::ResultNeedFormat();
            }
            return nn::nfc::ResultNeedRestore();
        }

        if(result.IsFailure())
        {
            return result;
        }

        // タグから読み込んだ内容をシステムセーブデータとして保存します。
        NN_RESULT_DO(BackupDataStream::GetInstance()->Write(physical.get(), Noft2::DataSize, m_Id));

        // 0 は NFP RAM の共有領域です。
        uintptr_t base = reinterpret_cast<uintptr_t>(m_LogicalData);
        int  i    = StreamRamSystemId;
        m_LogicalDataNode[i].id               = StreamRamSystemId;
        m_LogicalDataNode[i].isReadOnly       = StreamRamSystemIsReadOnly;
        m_LogicalDataNode[i].isShort          = true;
        m_LogicalDataNode[i].dataSize         = StreamRamSystemDataSize;
        m_LogicalDataNode[i].dataStartAddress = base + StreamRamSystemAddress;

        // 1 は NFP ROM です。
        i = StreamRomId;
        m_LogicalDataNode[i].id               = StreamRomId;
        m_LogicalDataNode[i].isReadOnly       = StreamRomIsReadOnly;
        m_LogicalDataNode[i].isShort          = true;
        m_LogicalDataNode[i].dataSize         = StreamRomDataSize;
        m_LogicalDataNode[i].dataStartAddress = base + StreamRomAddress;

        // 2 は NFP RAM のアプリケーション領域です。
        i = StreamRamAppId;
        m_LogicalDataNode[i].id               = StreamRamAppId;
        m_LogicalDataNode[i].isReadOnly       = StreamRamAppIsReadOnly;
        m_LogicalDataNode[i].isShort          = true;
        m_LogicalDataNode[i].dataSize         = StreamRamAppDataSize;
        m_LogicalDataNode[i].dataStartAddress = base + StreamRamAppAddress;
    }
    else
    {
        NN_ABORT("Unknown mount target\n");
    }

    m_MountTarget = mountTarget;
    SetState(Noft2::State_Mount);
    NN_RESULT_SUCCESS;
}

nn::Result Noft2::Unmount() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    SetState(Noft2::State_Init);
    std::memset(m_LogicalDataNode, 0, sizeof(m_LogicalDataNode));
    std::memset(&m_MountTarget, 0, sizeof(m_MountTarget)); //TODO MountTarget_None を定義すべき
    NN_RESULT_SUCCESS;
}

void Noft2::SetState(Noft2::State state) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    m_State = state;
}

nn::Result Noft2::OpenApplicationArea(nn::Bit32 accessId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    CommonBlockData* commonBlockData  = GET_COMMON_BLOCK_DATA();

    // アプリケーション領域が存在しない場合には失敗します。
    if (!HasApplicationArea(*commonBlockData))
    {
        return nn::nfc::ResultNeedCreate();
    }

    // アプリケーション専用領域にアクセス可能かを AccessId から判定します。
    nn::Bit32 tagAccessId = GetValueFromTag<nn::Bit32>(commonBlockData->accessId);
    if (accessId != tagAccessId)
    {
        return nn::nfc::ResultAccessIdMisMatch();
    }

    m_IsOpenedApplicationArea = true;
    NN_RESULT_SUCCESS;
}

nn::Result Noft2::GetApplicationArea(void* pOutBuffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutBuffer);
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutSize);
    NN_NFC_SERVER_UTIL_REQUIRES(0 < bufferSize);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    // アプリケーション領域がオープンされていない場合には失敗します。
    if (!m_IsOpenedApplicationArea)
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    ApplicationBlockData* applicationBlockData  = GET_APPLICATION_BLOCK_DATA();

    // アプリケーション領域の内容をコピーします。
    *pOutSize = bufferSize < StreamRamAppDataSize ? bufferSize : StreamRamAppDataSize;
    std::memcpy(pOutBuffer, applicationBlockData, *pOutSize);
    NN_RESULT_SUCCESS;
}

nn::Result Noft2::SetApplicationArea(const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pData);
    NN_NFC_SERVER_UTIL_REQUIRES(0 < dataSize);
    if(StreamRamAppDataSize < dataSize)
    {
        return nn::nfc::ResultInvalidSize();
    }

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    // アプリケーション領域がオープンされていない場合には失敗します。
    if (!m_IsOpenedApplicationArea)
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    ApplicationBlockData* applicationBlockData  = GET_APPLICATION_BLOCK_DATA();

    // アプリケーション領域のキャッシュを書き換えます。
    size_t copySize = dataSize < StreamRamAppDataSize ? dataSize : StreamRamAppDataSize;
    std::memcpy(applicationBlockData, pData, copySize);
    FillRandom(reinterpret_cast<nn::Bit8*>(applicationBlockData) + copySize, StreamRamAppDataSize - copySize);
    m_IsUpdatedAppBlock = true;
    NN_RESULT_SUCCESS;
}

nn::Result Noft2::RecreateApplicationArea(const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    NN_RESULT_DO(SetApplicationArea(createInfo.pInitialData, createInfo.initialDataSize));

    CommonBlockData* commonBlockData = GET_COMMON_BLOCK_DATA();
    SetValueToTag(commonBlockData->accessId, createInfo.accessId);

    NN_RESULT_SUCCESS;
}

nn::Result Noft2::Flush(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    NN_NFP_SERVER_SAVE_BEGIN();

    // 共有領域を書き換えます。
    CommonBlockData* commonBlockData  = GET_COMMON_BLOCK_DATA();

    bool hasMiiExtention = ExistMiiExtention(*commonBlockData);

    UpdateWriteDate(commonBlockData);
    UpdateMoveCounter(commonBlockData);

    if (m_IsUpdatedAppBlock)
    {
        nn::Bit64 titleId = GetUserTitleId(service);
        SetTitleId(commonBlockData, titleId);

        if(hasMiiExtention)
        {
            //拡張 Mii が既に登録されている場合のみ
            //titleIdEx が更新されているため、CRC計算しなおす
            SetExtCrc(commonBlockData);
        }
    }

    // タグに書き込みます。
    NN_RESULT_TRY(FlushNormal(service))
        NN_RESULT_CATCH_ALL
        {
            NN_NFP_SERVER_ROLLBACK();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_NFP_SERVER_SAVE_END();

    NN_RESULT_SUCCESS;
}

nn::Result Noft2::Restore(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    nn::Result result;
    std::unique_ptr<nn::Bit8[]> physical(new nn::Bit8[Noft2::DataSize]);
    std::unique_ptr<nn::Bit8[]> decryptedLogical(new nn::Bit8[Noft2::DataSize]);

    // 修復する必要があるか確認する
    NN_RESULT_DO(ReadPhysicalData(physical.get(), service, Noft2::DataSize));

    NN_RESULT_DO(CheckNoftVersion(physical.get(), Noft2::DataSize));

    result = ConvertToDecryptedLogicalData(decryptedLogical.get(), Noft2::DataSize, physical.get(), Noft2::DataSize);
    if(result.IsSuccess())
    {
        return nn::nfc::ResultNotBroken();
    }

    if (!nn::nfc::ResultNeedFormat::Includes(result))
    {
        return result;
    }

    // バックアップデータを取得します。
    {
        NN_RESULT_DO(BackupDataStream::GetInstance()->Initialize());
        NN_UTIL_SCOPE_EXIT
        {
            BackupDataStream::GetInstance()->Finalize();
        };

        result = BackupDataStream::GetInstance()->Read(physical.get(), Noft2::DataSize, m_Id);
        if(result.IsFailure())
        {
            if(nn::nfc::ResultBackupError::Includes(result))
            {
                // Restore するべきか再度 Mount で判定しなおすように
                return nn::nfc::ResultNeedRestart();
            }

            // 致命的なエラー
            return result;
        }
    }

    // バックアップデータが壊れていないことを確認します。
    result = ConvertToDecryptedLogicalData(decryptedLogical.get(), Noft2::DataSize, physical.get(), Noft2::DataSize);
    if (result.IsFailure())
    {
        // Restore するべきか再度 Mount で判定しなおすように
        return nn::nfc::ResultNeedRestart();
    }

    // タグを復元します。
    return WritePhysicalData(service, physical.get(), Noft2::DataSize, false);
}

nn::Result Noft2::CreateApplicationArea(nn::nfc::server::core::Service* service, const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(createInfo.pInitialData);
    NN_NFC_SERVER_UTIL_REQUIRES(0 < createInfo.initialDataSize);
    if(StreamRamAppDataSize < createInfo.initialDataSize)
    {
        return nn::nfc::ResultInvalidSize();
    }

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    CommonBlockData* commonBlockData = GET_COMMON_BLOCK_DATA();
    // 既にアプリケーション領域が存在する場合には失敗します。
    if (HasApplicationArea(*commonBlockData))
    {
        return nn::nfc::ResultAlreadyCreated();
    }

    NN_NFP_SERVER_SAVE_BEGIN();

    bool hasMiiExtention = ExistMiiExtention(*commonBlockData);

    // 共有領域を書き換えます。
    UpdateWriteDate(commonBlockData);

    nn::Bit64 titleId = GetUserTitleId(service);
    SetTitleId(commonBlockData, titleId);
    SetValueToTag(commonBlockData->accessId, createInfo.accessId);
    nn::Bit8 regInfoAndLocale = GetValueFromTag<nn::Bit8>(commonBlockData->regInfoAndLocale);
    regInfoAndLocale |= (nn::nfp::RegisterInfoFlags_ApplicationArea << 4);
    SetValueToTag(commonBlockData->regInfoAndLocale, regInfoAndLocale);

    UpdateMoveCounter(commonBlockData);

    // アプリケーション専用領域を書き換えます。
    ApplicationBlockData* applicationBlockData  = GET_APPLICATION_BLOCK_DATA();
    size_t copySize = createInfo.initialDataSize < StreamRamAppDataSize ? createInfo.initialDataSize : StreamRamAppDataSize;

    std::memcpy(applicationBlockData, createInfo.pInitialData, copySize);
    FillRandom(reinterpret_cast<nn::Bit8*>(applicationBlockData) + createInfo.initialDataSize, StreamRamAppDataSize - createInfo.initialDataSize);
    m_IsUpdatedAppBlock = true;

    if(hasMiiExtention)
    {
        //拡張 Mii が既に登録されている場合のみ
        //titleIdEx が更新されているため、CRC計算しなおす
        SetExtCrc(commonBlockData);
    }

    // タグに書き込みます。
    NN_RESULT_TRY(FlushNormal(service))
        NN_RESULT_CATCH_ALL
        {
            NN_NFP_SERVER_ROLLBACK();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_NFP_SERVER_SAVE_END();

    NN_RESULT_SUCCESS;
}


nn::Result Noft2::GetRegisterInfo(nn::nfp::RegisterInfo* pOutRegisterInfo) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutRegisterInfo);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    const CommonBlockData* commonBlockData = GET_COMMON_BLOCK_DATA();
    // 初期登録情報が登録されていない場合には失敗します。
    if (!HasRegisterInfo(*commonBlockData))
    {
        return nn::nfc::ResultNeedRegister();
    }

    // 初期登録情報を出力します。
    NN_RESULT_DO(DecodeRegisterInfo(pOutRegisterInfo, *commonBlockData));
    NN_RESULT_SUCCESS;
}

nn::Result Noft2::GetCommonInfo(nn::nfp::CommonInfo* pOutCommonInfo) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutCommonInfo);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    const CommonBlockData* commonBlockData = GET_COMMON_BLOCK_DATA();

    // 共有領域を出力します。
    DecodeCommonInfo(pOutCommonInfo, *commonBlockData, false);
    NN_RESULT_SUCCESS;
}

nn::Result Noft2::GetModelInfo(nn::nfp::ModelInfo* pOutModelInfo) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutModelInfo);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRom())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    // 共有領域を出力します。
    const LockedBlockData* lockedBlockData = GET_LOCKED_BLOCK_DATA();
    DecodeModelInfo(pOutModelInfo, *lockedBlockData);
    NN_RESULT_SUCCESS;
}

nn::Result Noft2::Format(nn::nfp::ModelInfo* pOutModelInfo, nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    std::unique_ptr<nn::Bit8[]> physicalData(new nn::Bit8[Noft2::DataSize]);

    NN_RESULT_DO(CreatePhysicalDataForFormat(physicalData.get(), service, Noft2::DataSize));
    NN_RESULT_DO(WritePhysicalData(service, physicalData.get(), Noft2::DataSize, false));

    std::unique_ptr<nn::Bit8[]> logicalData(new nn::Bit8[Noft2::DataSize]);
    ConvertToLogical(logicalData.get(), Noft2::DataSize, physicalData.get(), Noft2::DataSize);
    DecodeModelInfo(pOutModelInfo, *reinterpret_cast<LockedBlockData*>(logicalData.get() + StreamRomAddress));
    NN_RESULT_SUCCESS;
}

nn::Result Noft2::GetAdminInfo(nn::nfp::AdminInfo* pOutAdminInfo) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutAdminInfo);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    const CommonBlockData* commonBlockData = GET_COMMON_BLOCK_DATA();

    // 管理領域を出力します。
    DecodeAdminInfo(pOutAdminInfo, *commonBlockData);
    NN_RESULT_SUCCESS;
}

nn::Result Noft2::GetRegisterInfo(nn::nfp::RegisterInfoPrivate* pOutRegisterInfo) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutRegisterInfo);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    const CommonBlockData* commonBlockData = GET_COMMON_BLOCK_DATA();
    // 初期登録情報が登録されていない場合には失敗します。
    if (!HasRegisterInfo(*commonBlockData))
    {
        return nn::nfc::ResultNeedRegister();
    }

    // 初期登録情報を出力します。
    NN_RESULT_DO(DecodeRegisterInfo(pOutRegisterInfo, *commonBlockData));
    NN_RESULT_SUCCESS;
}

nn::Result Noft2::SetRegisterInfo(const nn::nfp::RegisterInfoPrivate& regInfo) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    NN_NFP_SERVER_SAVE_BEGIN();

    // 未登録時だけ writeCounter と regDate を書き換えます。
    CommonBlockData* commonBlockData = GET_COMMON_BLOCK_DATA();
    if (!HasRegisterInfo(*commonBlockData))
    {
        m_IsCreatedRegisterInfo = true;
        SetValueToTag(commonBlockData->regDate, Date::GetNow().GetTagDate());

        // Flush までの間にも取得される可能性があるので適当な値を入れておきます。
        const nn::Bit16 dummyDate = 0;
        SetValueToTag(commonBlockData->lastWriteDate, dummyDate);
    }

    // キャッシュ上の共用領域を更新します。
    SetValueToTag(commonBlockData->countryCode, static_cast<nn::Bit8>(0));//TODO 国コード取得が必要 static_cast<bit8>(nn::cfg::GetCountry());
    nn::Bit8 regInfoAndLocale = GetValueFromTag<nn::Bit8>(commonBlockData->regInfoAndLocale);
    regInfoAndLocale |= (nn::nfp::RegisterInfoFlags_CommonArea  << 4);
    regInfoAndLocale &= 0xF0;
    regInfoAndLocale |= regInfo.fontRegion;
    SetValueToTag(commonBlockData->regInfoAndLocale, regInfoAndLocale);

    NN_RESULT_TRY(SetNickname(commonBlockData, regInfo.nickname))
        NN_RESULT_CATCH_ALL
        {
            NN_NFP_SERVER_ROLLBACK();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_TRY(SetMii(commonBlockData, regInfo.miiData))
        NN_RESULT_CATCH_ALL
        {
            NN_NFP_SERVER_ROLLBACK();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    //CRC計算のため、最後にする必要がある
    SetExtCrc(commonBlockData);

    NN_NFP_SERVER_SAVE_END();

    NN_RESULT_SUCCESS;
}

nn::Result Noft2::DeleteRegisterInfo(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    CommonBlockData* commonBlockData = GET_COMMON_BLOCK_DATA();

    // 登録情報が存在しない場合には失敗します。
    if (!HasRegisterInfo(*commonBlockData))
    {
        return nn::nfc::ResultNeedRegister();
    }

    // 書き込み失敗時の復帰処理のため現在の状態を記憶しておきます。
    NN_NFP_SERVER_SAVE_BEGIN();

    // 共用領域の登録情報を削除します。
    nn::Bit8 regInfoAndLocale = GetValueFromTag<nn::Bit8>(commonBlockData->regInfoAndLocale);
    regInfoAndLocale &= ~ (nn::nfp::RegisterInfoFlags_CommonArea << 4);
    regInfoAndLocale ^= (0x0F & Random8());
    SetValueToTag(commonBlockData->regInfoAndLocale, regInfoAndLocale);
    SetValueToTag(commonBlockData->countryCode, static_cast<nn::Bit8>(0));
    SetValueToTag(commonBlockData->regDate, Random16());
    std::memset(commonBlockData->nickname, Random8(), sizeof(commonBlockData->nickname));
    FillRandom(commonBlockData->miiData, sizeof(commonBlockData->miiData));
    FillRandom(commonBlockData->miiExtension, sizeof(commonBlockData->miiExtension));
    SetValueToTag(commonBlockData->miiExtVersion, Random8());
    SetValueToTag(commonBlockData->extCrc, Random32());

    m_IsCreatedRegisterInfo = false;

    // タグに書き込みます。
    NN_RESULT_TRY(FlushNormal(service))
        NN_RESULT_CATCH_ALL
        {
            NN_NFP_SERVER_ROLLBACK();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_NFP_SERVER_SAVE_END();

    NN_RESULT_SUCCESS;
}

nn::Result Noft2::DeleteApplicationArea(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    CommonBlockData* commonBlockData = GET_COMMON_BLOCK_DATA();

    // アプリケーション領域が存在しない場合には失敗します。
    if (!HasApplicationArea(*commonBlockData))
    {
        return nn::nfc::ResultNeedCreate();
    }

    NN_NFP_SERVER_SAVE_BEGIN();

    bool hasMiiExtention = ExistMiiExtention(*commonBlockData);

    // 共用領域の登録情報を削除します。
    nn::Bit8 regInfoAndLocale = GetValueFromTag<nn::Bit8>(commonBlockData->regInfoAndLocale);
    regInfoAndLocale &= ~ (nn::nfp::RegisterInfoFlags_ApplicationArea << 4);
    SetValueToTag(commonBlockData->regInfoAndLocale, regInfoAndLocale);
    SetValueToTag(commonBlockData->accessId, Random32());
    SetValueToTag(commonBlockData->titleId, Random64());
    SetValueToTag(commonBlockData->titleIdEx, Random8());

    // アプリケーション専用領域を乱数で埋めます。
    ApplicationBlockData* applicationBlockData  = GET_APPLICATION_BLOCK_DATA();
    FillRandom(applicationBlockData, StreamRamAppDataSize);
    m_IsUpdatedAppBlock = false;
    m_IsOpenedApplicationArea = false;

    if(hasMiiExtention)
    {
        //拡張 Mii が既に登録されている場合のみ
        //titleIdEx が更新されているため、CRC計算しなおす
        SetExtCrc(commonBlockData);
    }

    // タグに書き込みます。
    NN_RESULT_TRY(FlushNormal(service))
        NN_RESULT_CATCH_ALL
        {
            NN_NFP_SERVER_ROLLBACK();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_NFP_SERVER_SAVE_END();

    NN_RESULT_SUCCESS;
}

nn::Result Noft2::ExistsApplicationArea(bool* outValue) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(outValue);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    CommonBlockData* commonBlockData = GET_COMMON_BLOCK_DATA();

    *outValue = HasApplicationArea(*commonBlockData);

    NN_RESULT_SUCCESS;
}

nn::Result Noft2::GetAll(nn::nfp::NfpData* pOutNfpData) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutNfpData);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    // 出力を初期化します。
    std::memset(pOutNfpData, 0, sizeof(nn::nfp::NfpData));

    // 共有領域を出力します。デバッグ用の API なので情報の隠ぺいは行いません。
    CommonBlockData* commonBlockData = GET_COMMON_BLOCK_DATA();
    const bool isDebugMode = true;
    DecodeSystemInfo(&pOutNfpData->systemInfo, *commonBlockData);
    DecodeCommonInfo(&pOutNfpData->commonInfo, *commonBlockData, isDebugMode);
    DecodeAdminInfo(&pOutNfpData->adminInfo, *commonBlockData);
    DecodeRegisterInfo(&pOutNfpData->registerInfo, *commonBlockData);

    // アプリケーション領域を出力します。
    ApplicationBlockData* applicationBlockData  = GET_APPLICATION_BLOCK_DATA();
    std::memcpy(pOutNfpData->applicationArea, applicationBlockData, StreamRamAppDataSize);

    NN_RESULT_SUCCESS;
}

nn::Result Noft2::SetAll(const nn::nfp::NfpData& nfpData) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    // 引数で受け取ったタグデータを反映します。
    CommonBlockData* commonBlockData = GET_COMMON_BLOCK_DATA();
    NN_RESULT_DO(CreateCommonBlockData(commonBlockData, nfpData.systemInfo, nfpData.commonInfo, nfpData.registerInfo, nfpData.adminInfo));

    ApplicationBlockData* applicationBlockData  = GET_APPLICATION_BLOCK_DATA();
    std::memcpy(applicationBlockData, nfpData.applicationArea, StreamRamAppDataSize);

    NN_RESULT_SUCCESS;
}

nn::Result Noft2::FlushDebug(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    // タグに書き込みます。
    return FlushCommon(service, nn::nfp::BreakType_None);
}

nn::Result Noft2::BreakTag(nn::nfc::server::core::Service* service, nn::nfp::BreakType breakType) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES(breakType == nn::nfp::BreakType_Activation
                           || breakType == nn::nfp::BreakType_Hmac
                           || breakType == nn::nfp::BreakType_None);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    // タグに書き込みます。
    return FlushCommon(service, breakType);
}

size_t Noft2::GetApplicationAreaSize() NN_NOEXCEPT
{
    return StreamRamAppDataSize;
}

nn::Result Noft2::FlushNormal(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    nn::Result result;

    // 共有領域を更新します。
    CommonBlockData* commonBlockData = GET_COMMON_BLOCK_DATA();

    uint16_t systemWriteCounter = GetValueFromTag<uint16_t>(commonBlockData->systemWriteCounter);
    ++systemWriteCounter;
    SetValueToTag(commonBlockData->systemWriteCounter, systemWriteCounter);

    if (m_IsUpdatedAppBlock || m_IsCreatedRegisterInfo)
    {
        uint16_t writeCounter = GetValueFromTag<uint16_t>(commonBlockData->writeCounter);
        writeCounter = IncrementCounter(writeCounter);
        SetValueToTag(commonBlockData->writeCounter, writeCounter);
    }

    // タグに書き込みます。
    NN_RESULT_DO(FlushCommon(service, nn::nfp::BreakType_None));

    m_IsUpdatedAppBlock     = false;
    m_IsCreatedRegisterInfo = false;

    NN_RESULT_SUCCESS;
}

nn::Result Noft2::FlushCommon(nn::nfc::server::core::Service* service, nn::nfp::BreakType breakType) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    std::unique_ptr<nn::Bit8[]> physicalData(new nn::Bit8[Noft2::DataSize]);

    NN_RESULT_DO(CreatePhysicalDataForFlush(physicalData.get(), Noft2::DataSize));

    //タグから読み込んだ内容をシステムセーブデータとして保存します。
    {
        NN_RESULT_DO(BackupDataStream::GetInstance()->Initialize());
        NN_UTIL_SCOPE_EXIT
        {
            BackupDataStream::GetInstance()->Finalize();
        };

        NN_RESULT_DO(BackupDataStream::GetInstance()->Write(physicalData.get(), Noft2::DataSize, m_Id));
    }

    // デバッグ用にタグに書き込むデータを破壊する処理です。
    BreakPhysicalData(physicalData.get(), Noft2::DataSize, breakType);

    return WritePhysicalData(service, physicalData.get(), Noft2::DataSize, true);
}

nn::Result Noft2::CreatePhysicalDataForFlush(void* pOutPhysicalData, size_t physicalDataSize) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    // タグデータを暗号化します。
    nn::Bit8 seed[nn::nfp::server::Crypto::PrfSeedSize];
    GenerateSeedFromLogical(seed, sizeof(seed), m_LogicalData, sizeof(m_LogicalData));

    std::unique_ptr<nn::Bit8[]> encryptedLogicalData(new nn::Bit8[Noft2::DataSize]);
    EncryptAndSignHmac(encryptedLogicalData.get(), Noft2::DataSize, m_LogicalData, sizeof(m_LogicalData), seed, sizeof(seed));

    // 物理アドレスに変換します。
    ConvertToPhysical(pOutPhysicalData, physicalDataSize, encryptedLogicalData.get(), Noft2::DataSize);
    NN_RESULT_SUCCESS;
}

nn::Result Noft2::CreatePhysicalDataForFormat(void* pOutPhysicalData, nn::nfc::server::core::Service* service, size_t physicalDataSize) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    std::unique_ptr<LogicalData> logicalData(new LogicalData);

    {
        std::unique_ptr<nn::Bit8[]> physicalData(new nn::Bit8[Noft2::DataSize]);
        NN_RESULT_DO(ReadPhysicalData(physicalData.get(), service, Noft2::DataSize));

        NN_RESULT_DO(CheckNoftVersion(physicalData.get(), Noft2::DataSize));

        // RAM 全体を適当な乱数で初期化します。

        ConvertToLogical(logicalData.get(), Noft2::DataSize, physicalData.get(), Noft2::DataSize);
        FillRandom(logicalData->activation, 396);//activation ~ application
    }

    // 初期値が必要な領域について値を設定します。
    logicalData->activation[0]          = 0xA5;
    logicalData->systemWriteCounter[0] |= 0x80;
    std::memset(logicalData->nfpVersion, 0, 5);
    std::memset(&logicalData->common[104], 0, 2); //WriteCounter

    {
        std::unique_ptr<nn::Bit8[]> encryptedLogicalData(new nn::Bit8[Noft2::DataSize]);

        // 暗号化と署名を行います。
        nn::Bit8 seed[nn::nfp::server::Crypto::PrfSeedSize];
        GenerateSeedFromLogical(seed, sizeof(seed), logicalData.get(), Noft2::DataSize);
        EncryptAndSignHmac(encryptedLogicalData.get(), Noft2::DataSize, logicalData.get(), sizeof(m_LogicalData), seed, sizeof(seed));

        // 暗号化された領域を物理アドレスに変換します。
        ConvertToPhysical(pOutPhysicalData, physicalDataSize, encryptedLogicalData.get(), Noft2::DataSize);
    }

    NN_RESULT_SUCCESS;
}

bool Noft2::IsMounted() NN_NOEXCEPT
{
    return (m_State == State_Mount);
}

bool Noft2::IsMountedRam() NN_NOEXCEPT
{
    return (IsMounted() && ((m_MountTarget & nn::nfp::MountTarget_Ram) != 0));
}

bool Noft2::IsMountedRom() NN_NOEXCEPT
{
    return (IsMounted() && ((m_MountTarget & nn::nfp::MountTarget_Rom) != 0));
}

void Noft2::UpdateMoveCounter(CommonBlockData* commonBlockData) NN_NOEXCEPT
{
    nn::Bit32 hardwareId = GetHardwareId();
    nn::Bit32 platformId = GetValueFromTag<nn::Bit32>(commonBlockData->platformId);

    if (platformId != hardwareId)
    {
        uint16_t moveCounter = GetValueFromTag<uint16_t>(commonBlockData->moveCounter);
        moveCounter = IncrementCounter(moveCounter);
        SetValueToTag(commonBlockData->moveCounter, moveCounter);
        SetValueToTag(commonBlockData->platformId, hardwareId);
    }
}

void Noft2::UpdateWriteDate(CommonBlockData* commonBlockData) NN_NOEXCEPT
{
    SetValueToTag(commonBlockData->lastWriteDate, Date::GetNow().GetTagDate());
}

nn::Result Noft2::ReadyToWrite(nn::nfc::server::core::Service* service, bool isPasswordRequired) NN_NOEXCEPT
{
    NN_RESULT_DO(Ntag::ReadyToWrite(service, isPasswordRequired));
    if(m_Type2TagVersion == nn::xcd::NfcType2TagVersion_Ntag210
       || m_Type2TagVersion == nn::xcd::NfcType2TagVersion_Ntag212
       || m_Type2TagVersion == nn::xcd::NfcType2TagVersion_Ntag213)
    {
        return nn::nfc::ResultInvalidTag();
    }

    NN_RESULT_SUCCESS;
}

nn::Result Noft2::CreateUserMemory(void* pOutPhysicalData, nn::nfc::server::core::Service* service, size_t physicalDataSize, const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pData);
    NN_NFC_SERVER_UTIL_REQUIRES(dataSize == 672);
    NN_ABORT_UNLESS(physicalDataSize == Noft2::DataSize);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    const NtfHeaderFormatEx* header = reinterpret_cast<const NtfHeaderFormatEx*>(pData);

#if 0
    NN_SDK_LOG("Product Code : %c%c%c%c%c%c%c%c\n",
               header->productCode[0],
               header->productCode[1],
               header->productCode[2],
               header->productCode[3],
               header->productCode[4],
               header->productCode[5],
               header->productCode[6],
               header->productCode[7]);
#endif

    if(header->headerVersion != 2)
    {
        return nn::nfc::ResultInvalidFormat();
    }

    std::unique_ptr<PhysicalData> decryptedNtf(new PhysicalData);
    NN_RESULT_DO(DecryptNtfAndHashCheck(decryptedNtf.get(), Noft2::DataSize, pData, dataSize));

    //NTAG215 チェック
    {
        nn::Bit8 data[1] = {0x60}; // GET_VERSION コマンド
        nn::Bit8 buffer[8];
        size_t responseSize;

        NN_RESULT_DO(SendCommand(buffer, &responseSize, service, data, sizeof(data), sizeof(buffer), nn::TimeSpan::FromMilliSeconds(2000)));
        if(responseSize != sizeof(buffer) // NACK
           || buffer[6] != 0x11 // !NTAG215
            )
        {
            return nn::nfc::ResultInvalidTag();
        }
    }

    // UID, Statick Lock, CC をコピー
    NN_RESULT_DO(SendFastReadCommand(decryptedNtf->serial, service, 0x00, 0x03, (0x03 + 1) * 4));

    // Static Lock をチェック
    if(decryptedNtf->staticLock[0] != 0
       || decryptedNtf->staticLock[1] != 0)
    {
        return nn::nfc::ResultInvalidTag();
    }

    {
        nn::Bit8 dynamicLock[4];
        NN_RESULT_DO(SendFastReadCommand(dynamicLock, service, 0x82, 0x82, 1 * 4));

        // Dynamic Lock をチェック
        if(dynamicLock[0] != 0
           || dynamicLock[1] != 0
           || dynamicLock[2] != 0)
        {
            return nn::nfc::ResultInvalidTag();
        }
    }

    nn::nfp::server::CreateUserMemory(pOutPhysicalData, physicalDataSize, decryptedNtf.get(), Noft2::DataSize);

    NN_RESULT_SUCCESS;
}

void Noft2::GetPwd(void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(bufferSize == 4);

    const nn::Bit8 seedForPwd[2] = {0xAA, 0x55};
    nn::Bit8* data = reinterpret_cast<nn::Bit8*>(pOutBuffer);

    data[0] = (m_Id.uid[1] ^ m_Id.uid[3] ^ seedForPwd[0]);
    data[1] = (m_Id.uid[2] ^ m_Id.uid[4] ^ seedForPwd[1]);
    data[2] = (m_Id.uid[3] ^ m_Id.uid[5] ^ seedForPwd[0]);
    data[3] = (m_Id.uid[4] ^ m_Id.uid[6] ^ seedForPwd[1]);
}

nn::Result Noft2::WriteUserMemory(nn::nfc::server::core::Service* service, const void* encryptedPhysicalData, size_t encryptedPhysicalDataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(encryptedPhysicalDataSize == Noft2::DataSize);

    const PhysicalData* p = reinterpret_cast<const PhysicalData*>(encryptedPhysicalData);
    bool isPasswordRequired;

    //既に一度書き込みを行ったタグかCCの値でチェック
    if(std::memcmp(Cc, p->cc, sizeof(p->cc)) == 0)
    {
        isPasswordRequired = true;
    }
    else
    {
        isPasswordRequired = false;
    }

    //一度 Read しておかないと書き込みできないため、代表して CC を読み込む
    //また、必要であれば認証も行う
    NN_RESULT_DO(ReadyToWrite(service, isPasswordRequired));

    return Write(service,
                 2000,
                 false,
                 false, nullptr, nullptr, 0,
                 0x04, 0x81,
                 reinterpret_cast<const nn::Bit8*>(encryptedPhysicalData) + 0x04 * nn::xcd::Type2TagPageSize,
                 (0x81 - 0x04 + 1) * nn::xcd::Type2TagPageSize);
}

nn::Result Noft2::WriteConfigureAndLock(nn::nfc::server::core::Service* service, const void* encryptedPhysicalData, size_t encryptedPhysicalDataSize, NtfWriteType ntfWriteType) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(encryptedPhysicalDataSize == Noft2::DataSize);

    const PhysicalData* p = reinterpret_cast<const PhysicalData*>(encryptedPhysicalData);

    //既に一度書き込みを行ったタグかCCの値でチェック
    if(std::memcmp(Cc, p->cc, sizeof(p->cc)) == 0)
    {
        // PWD_AUTH
        nn::Bit8 data[5];
        data[0] = 0x1B;
        GetPwd(&data[1], 4);
        nn::Bit8 pack[2];
        size_t responseSize;
        NN_RESULT_DO(SendCommand(pack, &responseSize, service, data, sizeof(data), sizeof(pack), nn::TimeSpan::FromMilliSeconds(2000)));
        if(responseSize != sizeof(pack)
           || std::memcmp(pack, Pack, sizeof(pack)) != 0)
        {
            return nn::nfc::ResultAuthenticationError();
        }
    }
    else
    {
        NN_RESULT_DO(SendWriteCommandAndVerify(service, 0x03, Cc, sizeof(Cc)));
        NN_RESULT_DO(SendWriteCommandAndVerify(service, 0x83, Cfg0, sizeof(Cfg0)));
        NN_RESULT_DO(SendWriteCommandAndVerify(service, 0x84, Cfg1, sizeof(Cfg1)));
    }

    NN_RESULT_DO(SendWriteCommand(service, 0x86, Pack, sizeof(Pack)));

    if(ntfWriteType == NtfWriteType_Lock)
    {
        nn::Bit8 lock[4];

        lock[0] = p->serial[8];
        lock[1] = p->internal[0];
        std::memcpy(&lock[2], StaticLock, 2);
        NN_RESULT_DO(SendWriteCommandAndVerify(service, 0x02, lock, sizeof(lock)));

        std::memcpy(&lock[0], DynamicLock, 4);
        NN_RESULT_DO(SendWriteCommandAndVerify(service, 0x82, lock, sizeof(lock)));
    }

    nn::Bit8 pwd[4];
    GetPwd(pwd, sizeof(pwd));
    return SendWriteCommand(service, 0x85, pwd, sizeof(pwd));
}

}}}  // namespace nn::nfp::server
