﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nfc/server/core/nfc_CoreManager.h>
#include <nn/nfp/server/nfp_Device.h>
#include "nfp_Noft2.h"
#include <nn/nfc/server/util/nfc_UtilUtil.h>
#include <nn/nfc/server/util/nfc_ScopedMutexLock.h>
#include <nn/nfc/server/core/nfc_ScopedSession.h>
#include <nn/nfc/server/nfc_Device.h>

namespace nn { namespace nfp {namespace server {

namespace
{
}

Device::Device(nn::nfc::server::Device* device) NN_NOEXCEPT
: m_Device(device)
{
}

Device::~Device() NN_NOEXCEPT
{
}

nn::Result Device::Mount(nn::nfc::server::core::Service* service, ModelType modelType, MountTarget mountTarget) NN_NOEXCEPT
{
    NN_UNUSED(modelType);
    NN_NFC_SERVER_UTIL_REQUIRES_EQUAL(ModelType_Amiibo, modelType);

    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(m_Device->CheckActive());

    nn::nfc::TagInfo tagInfo;
    NN_RESULT_DO(m_Device->GetTagInfo(&tagInfo, service));

    nn::nfc::server::core::ScopedSession session(service, m_Device->m_Handle);
    NN_RESULT_DO(session.Keep());

    bool isNtag;
    isNtag = Noft2::IsNtag(static_cast<nn::nfc::NfcProtocol>(tagInfo.protocol), static_cast<nn::nfc::TagType>(tagInfo.type));
    if(!isNtag)
    {
        //NTAG 以外のタグ
        // - Type 1
        // - Type 3
        // - Type 4A
        // - Type 4B
        // - ISO15693
        return nn::nfc::ResultInvalidTag();
    }

    std::unique_ptr<Noft2> targetTag(new Noft2(m_Device->m_Handle, m_Device->m_SelectedTagId, &m_Device->m_AccessFinishEvent, &m_Device->m_AccessResetEvent));

    NN_RESULT_DO(m_Device->AfterAccessTag(service, &session, targetTag->Mount(service, mountTarget)));

    m_MountedTag = std::move(targetTag);//マウントされるのは Noft2 である
    m_Device->SetState(nn::nfc::server::Device::State_Mount);
    m_LastAccessInfo = nn::nfp::server::PlayReport::AccessInfo_Read;
    m_LastCall = Call_Mount;
    NN_RESULT_SUCCESS;
}

nn::Result Device::Unmount() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMount());

    std::unique_ptr<nn::nfp::ModelInfo> modelInfo(new nn::nfp::ModelInfo);
    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提

    if(noft2->GetModelInfo(modelInfo.get()).IsSuccess())
    {
        SavePlayReport(*modelInfo.get());
    }
    m_MountedTag.reset(nullptr);
    m_Device->SetState(nn::nfc::server::Device::State_Active);

    NN_RESULT_SUCCESS;
}

nn::Result Device::OpenApplicationArea(nn::Bit32 accessId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    return noft2->OpenApplicationArea(accessId);
}

nn::Result Device::GetApplicationArea(void* pOutBuffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    return noft2->GetApplicationArea(pOutBuffer, pOutSize, bufferSize);
}

nn::Result Device::SetApplicationArea(const void* pData, size_t dataSize) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    NN_RESULT_DO(noft2->SetApplicationArea(pData, dataSize));
    m_LastCall = Call_SetApplicationArea;
    NN_RESULT_SUCCESS;
}

nn::Result Device::RecreateApplicationArea(const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    NN_RESULT_DO(noft2->RecreateApplicationArea(createInfo));
    m_LastCall = Call_RecreateApplicationArea;
    NN_RESULT_SUCCESS;
}

nn::Result Device::Flush(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    nn::nfc::server::core::ScopedSession session(service, m_Device->m_Handle);
    NN_RESULT_DO(session.Keep());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    NN_RESULT_DO(m_Device->AfterAccessTag(service, &session, noft2->Flush(service)));

    if(m_LastCall == Call_SetApplicationArea || m_LastCall == Call_RecreateApplicationArea)
    {
        m_LastAccessInfo = nn::nfp::server::PlayReport::AccessInfo_WriteApplicationArea;
    }
    else if(m_LastCall == Call_SetRegisterInfo)
    {
        m_LastAccessInfo = nn::nfp::server::PlayReport::AccessInfo_WriteRegisterInfo;
    }

    NN_RESULT_SUCCESS;
}

nn::Result Device::Restore(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(m_Device->CheckActive());

    nn::nfc::TagInfo tagInfo;
    NN_RESULT_DO(m_Device->GetTagInfo(&tagInfo, service));

    nn::nfc::server::core::ScopedSession session(service, m_Device->m_Handle);
    NN_RESULT_DO(session.Keep());

    bool isNtag;
    isNtag = Noft2::IsNtag(static_cast<nn::nfc::NfcProtocol>(tagInfo.protocol), static_cast<nn::nfc::TagType>(tagInfo.type));
    if(!isNtag)
    {
        //NTAG 以外のタグ
        // - Type 1
        // - Type 3
        // - Type 4A
        // - Type 4B
        // - ISO15693
        return nn::nfc::ResultInvalidTag();
    }

    std::unique_ptr<Noft2> targetTag(new Noft2(m_Device->m_Handle, m_Device->m_SelectedTagId, &m_Device->m_AccessFinishEvent, &m_Device->m_AccessResetEvent));

    return m_Device->AfterAccessTag(service, &session, targetTag->Restore(service));
}

nn::Result Device::CreateApplicationArea(nn::nfc::server::core::Service* service, const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    nn::nfc::server::core::ScopedSession session(service, m_Device->m_Handle);
    NN_RESULT_DO(session.Keep());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    NN_RESULT_DO(m_Device->AfterAccessTag(service, &session, noft2->CreateApplicationArea(service, createInfo)));

    m_LastAccessInfo = nn::nfp::server::PlayReport::AccessInfo_WriteApplicationArea;

    NN_RESULT_SUCCESS;
}

nn::Result Device::GetRegisterInfo(nn::nfp::RegisterInfo* pOutRegisterInfo) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    return noft2->GetRegisterInfo(pOutRegisterInfo);
}

nn::Result Device::GetCommonInfo(nn::nfp::CommonInfo* pOutCommonInfo) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    return noft2->GetCommonInfo(pOutCommonInfo);
}

nn::Result Device::GetModelInfo(nn::nfp::ModelInfo* pOutModelInfo) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRom());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    return noft2->GetModelInfo(pOutModelInfo);
}

nn::Result Device::Format(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(m_Device->CheckActive());

    nn::nfc::TagInfo tagInfo;
    NN_RESULT_DO(m_Device->GetTagInfo(&tagInfo, service));

    nn::nfc::server::core::ScopedSession session(service, m_Device->m_Handle);
    NN_RESULT_DO(session.Keep());

    bool isNtag;
    isNtag = Noft2::IsNtag(static_cast<nn::nfc::NfcProtocol>(tagInfo.protocol), static_cast<nn::nfc::TagType>(tagInfo.type));
    if(!isNtag)
    {
        //NTAG 以外のタグ
        // - Type 1
        // - Type 3
        // - Type 4A
        // - Type 4B
        // - ISO15693
        return nn::nfc::ResultInvalidTag();
    }

    std::unique_ptr<Noft2> targetTag(new Noft2(m_Device->m_Handle, m_Device->m_SelectedTagId, &m_Device->m_AccessFinishEvent, &m_Device->m_AccessResetEvent));
    std::unique_ptr<nn::nfp::ModelInfo> modelInfo(new nn::nfp::ModelInfo);
    NN_RESULT_DO(m_Device->AfterAccessTag(service, &session, targetTag->Format(modelInfo.get(), service)));

    m_LastAccessInfo = nn::nfp::server::PlayReport::AccessInfo_Format;
    SavePlayReport(*modelInfo.get());

    NN_RESULT_SUCCESS;
}

nn::Result Device::GetAdminInfo(nn::nfp::AdminInfo* pOutAdminInfo) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    return noft2->GetAdminInfo(pOutAdminInfo);
}

nn::Result Device::GetRegisterInfo(nn::nfp::RegisterInfoPrivate* pOutRegisterInfo) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    return noft2->GetRegisterInfo(pOutRegisterInfo);
}

nn::Result Device::SetRegisterInfo(const nn::nfp::RegisterInfoPrivate& regInfo) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    NN_RESULT_DO(noft2->SetRegisterInfo(regInfo));
    m_LastCall = Call_SetRegisterInfo;
    NN_RESULT_SUCCESS;
}

nn::Result Device::DeleteRegisterInfo(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    nn::nfc::server::core::ScopedSession session(service, m_Device->m_Handle);
    NN_RESULT_DO(session.Keep());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    return m_Device->AfterAccessTag(service, &session, noft2->DeleteRegisterInfo(service));
}

nn::Result Device::DeleteApplicationArea(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    nn::nfc::server::core::ScopedSession session(service, m_Device->m_Handle);
    NN_RESULT_DO(session.Keep());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    NN_RESULT_DO(m_Device->AfterAccessTag(service, &session, noft2->DeleteApplicationArea(service)));

    m_LastAccessInfo = nn::nfp::server::PlayReport::AccessInfo_DeleteApplicationArea;

    NN_RESULT_SUCCESS;
}

nn::Result Device::ExistsApplicationArea(bool* outValue) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    return noft2->ExistsApplicationArea(outValue);
}

nn::Result Device::GetAll(nn::nfp::NfpData* pOutNfpData) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    return noft2->GetAll(pOutNfpData);
}

nn::Result Device::SetAll(const nn::nfp::NfpData& nfpData) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    return noft2->SetAll(nfpData);
}

nn::Result Device::FlushDebug(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    nn::nfc::server::core::ScopedSession session(service, m_Device->m_Handle);
    NN_RESULT_DO(session.Keep());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    return m_Device->AfterAccessTag(service, &session, noft2->FlushDebug(service));
}

nn::Result Device::BreakTag(nn::nfc::server::core::Service* service, BreakType breakType) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMountRam());

    nn::nfc::server::core::ScopedSession session(service, m_Device->m_Handle);
    NN_RESULT_DO(session.Keep());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提
    return m_Device->AfterAccessTag(service, &session, noft2->BreakTag(service, breakType));
}

nn::Result Device::WriteNtf(nn::nfc::server::core::Service* service, const void* pData, size_t dataSize, NtfWriteType ntfWriteType) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(m_Device->CheckActive());

    nn::nfc::TagInfo tagInfo;
    NN_RESULT_DO(m_Device->GetTagInfo(&tagInfo, service));

    nn::nfc::server::core::ScopedSession session(service, m_Device->m_Handle);
    NN_RESULT_DO(session.Keep());

    bool isNtag;
    isNtag = Noft2::IsNtag(static_cast<nn::nfc::NfcProtocol>(tagInfo.protocol), static_cast<nn::nfc::TagType>(tagInfo.type));
    if(!isNtag)
    {
        //NTAG 以外のタグ
        // - Type 1
        // - Type 3
        // - Type 4A
        // - Type 4B
        // - ISO15693
        return nn::nfc::ResultInvalidTag();
    }

    std::unique_ptr<Noft2> targetTag(new Noft2(m_Device->m_Handle, m_Device->m_SelectedTagId, &m_Device->m_AccessFinishEvent, &m_Device->m_AccessResetEvent));
    std::unique_ptr<nn::Bit8> encryptedPhysicalData(new nn::Bit8[Noft2::DataSize]);

    NN_RESULT_DO(m_Device->AfterAccessTag(service, &session, targetTag->CreateUserMemory(encryptedPhysicalData.get(), service, Noft2::DataSize, pData, dataSize)));

    // nn::xcd::SendNfcRawData を使用するため一旦セッション切断
    session.Release();
    NN_RESULT_DO(session.Keep());

    NN_RESULT_DO(m_Device->AfterAccessTag(service, &session, targetTag->WriteUserMemory(service, encryptedPhysicalData.get(), Noft2::DataSize)));

    // 再び nn::xcd::StartNtagWrite / nn::xcd::StartNtagRead を使用するため一旦セッション切断
    session.Release();
    NN_RESULT_DO(session.Keep());

    return m_Device->AfterAccessTag(service, &session, targetTag->WriteConfigureAndLock(service, encryptedPhysicalData.get(), Noft2::DataSize, ntfWriteType));
}

size_t Device::GetApplicationAreaSize() NN_NOEXCEPT
{
    return Noft2::GetApplicationAreaSize();
}

nn::Result Device::CheckMount() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    if(m_Device->m_State != nn::nfc::server::Device::State_Mount)
    {
        if(m_Device->m_State == nn::nfc::server::Device::State_Deactive)
        {
            return nn::nfc::ResultTagNotFound();
        }
        return nn::nfc::ResultInvalidDeviceState();
    }

    NN_RESULT_SUCCESS;
}

nn::Result Device::CheckMountRam() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMount());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提

    if(!noft2->IsMountedRam())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    NN_RESULT_SUCCESS;
}

nn::Result Device::CheckMountRom() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Device->m_Mutex);

    NN_RESULT_DO(CheckMount());

    Noft2* noft2 = static_cast<Noft2*>(m_MountedTag.get());//マウントされるのが Noft2 である前提

    if(!noft2->IsMountedRom())
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    NN_RESULT_SUCCESS;
}

void Device::SavePlayReport(const nn::nfp::ModelInfo& modelInfo) NN_NOEXCEPT
{
    nn::hid::NpadIdType npadId;

    if(m_Device->m_ActiveService == nullptr
       || m_Device->GetNpadId(&npadId, m_Device->m_ActiveService).IsFailure())
    {
        return;
    }

    NN_FUNCTION_LOCAL_STATIC(nn::nfp::server::PlayReport, playReport);

    playReport.Lock();
    playReport.SetTagId(m_Device->m_SelectedTagId);
    playReport.SetModelInfo(modelInfo);
    playReport.SetProcessId(m_Device->m_ActiveService->GetPid());
    playReport.SetAccessInfo(m_LastAccessInfo);
    playReport.SetNpadId(npadId);
    playReport.Save();
    playReport.Unlock();
}


}}}  // namespace nn::nfp::server
