﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/news_NewsService.h>
#include <nn/news/detail/service/news_NewlyArrivedEventHolder.h>
#include <nn/news/detail/service/news_NewsDatabaseService.h>
#include <nn/news/detail/service/news_NewsDataService.h>
#include <nn/news/detail/service/news_Common.h>
#include <nn/news/detail/service/core/news_ForegroundStorageLockManager.h>
#include <nn/news/detail/service/core/news_NewlyArrivedEventListener.h>
#include <nn/news/detail/service/core/news_NewsDatabase.h>
#include <nn/news/detail/service/core/news_NewsImporter.h>
#include <nn/news/detail/service/core/news_NewsSubscriber.h>
#include <nn/news/detail/service/core/news_NewsTaskManager.h>
#include <nn/news/detail/service/core/news_PassphraseManager.h>
#include <nn/news/detail/service/core/news_ReceivedHistoryManager.h>
#include <nn/news/detail/service/core/news_Version.h>
#include <nn/bcat/service/bcat_ApiService.h>
#include <nn/bcat/bcat_Result.h>

#define CHECK_CAPABILITY(flag) \
    NN_RESULT_THROW_UNLESS(m_Capability.IsPermitted(flag), ResultNotPermitted())

#define CHECK_IPC_STRING(value, maxSize) \
    do                                                              \
    {                                                               \
        size_t size = value.GetLength();                            \
        if (size == 0 || size > maxSize || value[size - 1] != '\0') \
        {                                                           \
            NN_RESULT_THROW(ResultInvalidArgument());               \
        }                                                           \
    }                                                               \
    while (NN_STATIC_CONDITION(0))

namespace nn { namespace news { namespace detail { namespace service {

NewsService::NewsService(const Capability& capability) NN_NOEXCEPT :
    m_Capability(capability)
{
}

NewsService::~NewsService() NN_NOEXCEPT
{
}

nn::Result NewsService::PostLocalNews(const nn::sf::InBuffer& data) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Post);

    NN_RESULT_THROW_UNLESS(detail::service::core::ForegroundStorageLockManager::GetInstance().TryAcquireImporterLock(), ResultLocked());

    NN_UTIL_SCOPE_EXIT
    {
        detail::service::core::ForegroundStorageLockManager::GetInstance().ReleaseImporterLock();
    };

    NN_DETAIL_NEWS_INFO("[news] NewsImporter::ImportFromLocal ...\n");

    NN_RESULT_DO(detail::service::core::NewsImporter::ImportFromLocal(data.GetPointerUnsafe(), data.GetSize()));

    NN_DETAIL_NEWS_INFO("[news] NewsImporter::ImportFromLocal done!\n");

    NN_RESULT_SUCCESS;
}

nn::Result NewsService::SetPassphrase(nn::ApplicationId appId, const nn::sf::InArray<char>& passphrase) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Config);

    CHECK_IPC_STRING(passphrase, PassphraseLengthMax + 1);

    NN_RESULT_THROW_UNLESS(appId != nn::ApplicationId::GetInvalidId(), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(detail::service::core::StringVerifier::VerifyPassphrase(passphrase.GetData()), ResultInvalidArgument());

    NN_RESULT_DO(detail::service::core::PassphraseManager::GetInstance().Save(appId, passphrase.GetData()));

    NN_RESULT_SUCCESS;
}

nn::Result NewsService::GetSubscriptionStatus(nn::sf::Out<std::int32_t> outStatus,
    const nn::sf::InArray<char>& topicIdString) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View | Capability::Flag_Config);

    CHECK_IPC_STRING(topicIdString, TopicIdLengthMax + 1);

    TopicId topicId = {};
    nn::util::Strlcpy(topicId.value, topicIdString.GetData(), sizeof (topicId.value));

    NN_RESULT_THROW_UNLESS(detail::service::core::StringVerifier::VerifyTopicId(topicId.value), ResultInvalidArgument());

    *outStatus = detail::service::core::NewsTaskManager::GetInstance().GetSubscriptionStatus(topicId);

    NN_RESULT_SUCCESS;
}

nn::Result NewsService::GetTopicList(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::news::TopicId>& outTopicIds, std::int32_t filter) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    int actualCount = 0;

    detail::service::core::NewsTaskManager::GetInstance().GetTopicList(&actualCount,
        outTopicIds.GetData(), static_cast<int>(outTopicIds.GetLength()), static_cast<SubscriptionStatusFilter>(filter));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result NewsService::IsSystemUpdateRequired(nn::sf::Out<bool> outIsRequired) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    bool isRequired;
    NN_RESULT_DO(detail::service::core::Version::GetInstance().IsSystemUpdateRequired(&isRequired));

    *outIsRequired = isRequired;

    NN_RESULT_SUCCESS;
}

nn::Result NewsService::RequestImmediateReception(const nn::sf::InArray<char>& topicIdString) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    CHECK_IPC_STRING(topicIdString, TopicIdLengthMax + 1);

    TopicId topicId = {};
    nn::util::Strlcpy(topicId.value, topicIdString.GetData(), sizeof (topicId.value));

    NN_RESULT_THROW_UNLESS(detail::service::core::StringVerifier::VerifyTopicId(topicId.value), ResultInvalidArgument());

    NN_RESULT_DO(detail::service::core::NewsTaskManager::GetInstance().RunImmediately(topicId));

    NN_RESULT_SUCCESS;
}

nn::Result NewsService::DecodeArchiveFile(nn::sf::Out<std::uint64_t> outSize, const nn::sf::OutBuffer& decoded, const nn::sf::InBuffer& encoded) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    nn::ApplicationId appId = {};
    char passphrase[PassphraseLengthMax + 1] = {};

    NN_RESULT_DO(detail::service::core::PassphraseManager::GetInstance().Get(&appId, passphrase, NN_ARRAY_SIZE(passphrase)));

    size_t actualSize = 0;

    NN_RESULT_TRY(nn::bcat::service::DecodeArchiveFile(&actualSize,
        decoded.GetPointerUnsafe(), decoded.GetSize(), encoded.GetPointerUnsafe(), encoded.GetSize(), appId, passphrase))
        NN_RESULT_CATCH(nn::bcat::ResultDataVerificationFailed)
        {
            NN_RESULT_THROW(ResultDataVerificationFailed());
        }
        NN_RESULT_CATCH(nn::bcat::ResultUnsupportedFormatDetected)
        {
            NN_RESULT_THROW(ResultUnsupportedFormatDetected());
        }
    NN_RESULT_END_TRY;

    *outSize = actualSize;

    NN_RESULT_SUCCESS;
}

nn::Result NewsService::SetSubscriptionStatus(const nn::sf::InArray<char>& topicIdString, std::int32_t status) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management | Capability::Flag_Config);

    CHECK_IPC_STRING(topicIdString, TopicIdLengthMax + 1);

    TopicId topicId = {};
    nn::util::Strlcpy(topicId.value, topicIdString.GetData(), sizeof (topicId.value));

    NN_RESULT_THROW_UNLESS(detail::service::core::StringVerifier::VerifyTopicId(topicId.value), ResultInvalidArgument());

    NN_RESULT_DO(detail::service::core::NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId,
        static_cast<SubscriptionStatus>(status)));

    NN_RESULT_SUCCESS;
}

nn::Result NewsService::RequestAutoSubscription(nn::ApplicationId appId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);

    NN_RESULT_THROW_UNLESS(appId != nn::ApplicationId::GetInvalidId(), ResultInvalidArgument());

    detail::service::core::NewsSubscriber::Register(appId);

    NN_RESULT_SUCCESS;
}

nn::Result NewsService::ClearStorage() NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);

    NN_RESULT_THROW_UNLESS(detail::service::core::ForegroundStorageLockManager::GetInstance().TryAcquireImporterLock(), ResultLocked());

    NN_UTIL_SCOPE_EXIT
    {
        detail::service::core::ForegroundStorageLockManager::GetInstance().ReleaseImporterLock();
    };

    NN_RESULT_DO(detail::service::core::ReceivedHistoryManager::GetInstance().Clear());

    NN_RESULT_TRY(service::core::FileSystem::DeleteDirectoryRecursively("news:/data"))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    NN_RESULT_TRY(detail::service::core::NewsDatabase::GetInstance().DeleteAll())
        NN_RESULT_CATCH_ALL
        {
            detail::service::core::FileSystem::Rollback(NN_DETAIL_NEWS_DATA_MOUNT_NAME);
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    // DB の削除とデータの削除は同時に成功する必要がある。
    detail::service::core::FileSystem::Commit(NN_DETAIL_NEWS_DATA_MOUNT_NAME);
    detail::service::core::FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

    NN_RESULT_SUCCESS;
}

nn::Result NewsService::ClearSubscriptionStatusAll() NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management | Capability::Flag_Config);

    NN_RESULT_DO(detail::service::core::NewsTaskManager::GetInstance().Clear());

    NN_RESULT_SUCCESS;
}

nn::Result NewsService::GetNewsDatabaseDump(nn::sf::Out<std::uint64_t> outSize, const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Debug);

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    size_t actualSize;

    NN_RESULT_DO(detail::service::core::NewsDatabase::GetInstance().GetDump(&actualSize, buffer.GetPointerUnsafe(), buffer.GetSize()));

    *outSize = static_cast<uint64_t>(actualSize);

    NN_RESULT_SUCCESS;
}

}}}}
