﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_PassphraseManager.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

namespace
{
    const char* g_FilePath = "news-sys:/passphrase.bin";
}

PassphraseManager::PassphraseManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_AppId(nn::ApplicationId::GetInvalidId())
{
}

nn::Result PassphraseManager::Get(nn::ApplicationId* outAppId, char* passphrase, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAppId);
    NN_SDK_REQUIRES_NOT_NULL(passphrase);
    NN_SDK_REQUIRES(size >= PassphraseLengthMax + 1);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (m_Passphrase[0] == '\0')
    {
        NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

        nn::fs::FileHandle handle = {};

        NN_RESULT_TRY(nn::fs::OpenFile(&handle, g_FilePath, nn::fs::OpenMode_Read))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_THROW(ResultPassphraseNotSaved());
            }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        char buffer[17 + PassphraseLengthMax + 1] = {};

        size_t fileSize;
        NN_RESULT_DO(nn::fs::ReadFile(&fileSize, handle, 0, buffer, sizeof (buffer)));

        NN_RESULT_THROW_UNLESS(fileSize > 17 && fileSize < sizeof (buffer), ResultPassphraseNotSaved());
        NN_RESULT_THROW_UNLESS(buffer[16] == ':', ResultPassphraseNotSaved());

        buffer[fileSize] = '\0';

        char* end = nullptr;
        nn::ApplicationId appId = {std::strtoull(buffer, &end, 16)};

        NN_RESULT_THROW_UNLESS(end && *end == ':', ResultPassphraseNotSaved());

        m_AppId = appId;
        nn::util::Strlcpy(m_Passphrase, &buffer[17], PassphraseLengthMax + 1);
    }

    *outAppId = m_AppId;
    nn::util::Strlcpy(passphrase, m_Passphrase, static_cast<int>(size));

    NN_RESULT_SUCCESS;
}

nn::Result PassphraseManager::Save(nn::ApplicationId appId, const char* passphrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(passphrase);

    int length = nn::util::Strnlen(passphrase, PassphraseLengthMax + 1);

    NN_RESULT_THROW_UNLESS(length > 0 && length <= PassphraseLengthMax, ResultInvalidArgument());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (appId == m_AppId && nn::util::Strncmp(passphrase, m_Passphrase, PassphraseLengthMax + 1) == 0)
    {
        NN_RESULT_SUCCESS;
    }

    char buffer[17 + PassphraseLengthMax + 1] = {};
    size_t size = static_cast<size_t>(nn::util::SNPrintf(buffer, sizeof (buffer), "%016llx:%s", appId.value, passphrase));

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();
    {
        NN_RESULT_TRY(nn::fs::DeleteFile(g_FilePath))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;

        NN_RESULT_DO(FileSystem::CreateFile(g_FilePath, size));

        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, g_FilePath, nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, buffer, size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }
    FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

    m_AppId = appId;
    nn::util::Strlcpy(m_Passphrase, passphrase, PassphraseLengthMax + 1);

    NN_RESULT_SUCCESS;
}

nn::Result PassphraseManager::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    NN_RESULT_TRY(nn::fs::DeleteFile(g_FilePath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

    m_AppId = nn::ApplicationId::GetInvalidId();
    m_Passphrase[0] = '\0';

    NN_RESULT_SUCCESS;
}

}}}}}
