﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_NewsSubscriber.h>
#include <nn/news/detail/service/core/news_TopicListDownloader.h>
#include <nn/news/detail/service/core/news_PassphraseManager.h>
#include <nn/news/detail/service/core/news_NewsTaskManager.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

namespace
{
    const int ApplicationListCountMax = 32;

    const int TopicCountMaxPerApplication = 64;

    struct Record
    {
        nn::ApplicationId appId;
    };
}

namespace
{
    nn::os::SdkMutexType g_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

    Record g_ReserveRecords[ApplicationListCountMax] = {};
    int g_ReserveCount = 0;

    Record g_ProcessedRecords[ApplicationListCountMax] = {};
    int g_ProcessedCount = 0;

    nn::os::TimerEvent g_Event(nn::os::EventClearMode_ManualClear);

    std::atomic<bool> g_IsReconnectionRequired(false);
}

namespace
{
    void RemoveApplication(int* outCount, nn::ApplicationId appId, Record* pRecords, int count) NN_NOEXCEPT
    {
        for (int i = 0; i < count; i++)
        {
            if (appId == pRecords[i].appId)
            {
                int moveCount = count - 1 - i;

                if (moveCount > 0)
                {
                    std::memmove(&pRecords[i], &pRecords[i + 1], sizeof (Record) * moveCount);
                }

                std::memset(&pRecords[--count], 0, sizeof (Record));
                break;
            }
        }

        *outCount = count;
    }

    void AddApplication(int* outCount, nn::ApplicationId appId, Record* pRecords, int count) NN_NOEXCEPT
    {
        // 重複分を削除する。
        RemoveApplication(&count, appId, pRecords, count);

        if (count == ApplicationListCountMax)
        {
            // 終端の要素を破棄する。
            count--;
        }

        if (count > 0)
        {
            std::memmove(&pRecords[1], &pRecords[0], sizeof (Record) * count);
        }

        pRecords[0].appId = appId;
        count++;

        *outCount = count;
    }

    nn::Result Subscribe(nn::ApplicationId appId, nn::nifm::NetworkConnection* pConnection) NN_NOEXCEPT
    {
        nn::ApplicationId newsAppId = {};
        char newsPassphrase[PassphraseLengthMax + 1] = {};

        NN_RESULT_DO(PassphraseManager::GetInstance().Get(&newsAppId, newsPassphrase, sizeof (newsPassphrase)));

        TopicId topicIds[TopicCountMaxPerApplication] = {};
        int count;

        TopicListDownloader downloader;

        NN_RESULT_DO(downloader.Download(&count, topicIds, NN_ARRAY_SIZE(topicIds),
            appId, newsAppId, newsPassphrase, pConnection));

        for (int i = 0; i < count; i++)
        {
            if (NewsTaskManager::GetInstance().GetSubscriptionStatus(topicIds[i]) == SubscriptionStatus_Unconfigured)
            {
                NN_RESULT_DO(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicIds[i],
                    SubscriptionStatus_AutoSubscribed));

                NN_DETAIL_NEWS_INFO("[news] '%s' was auto subscribed.\n", topicIds[i].value);
            }
            else
            {
                NN_DETAIL_NEWS_INFO("[news] '%s' has already been configured.\n", topicIds[i].value);
            }
        }

        NN_DETAIL_NEWS_INFO("[news] Done.\n");

        NN_RESULT_SUCCESS;
    }
}

void NewsSubscriber::Register(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    AddApplication(&g_ReserveCount, appId, g_ReserveRecords, g_ReserveCount);

    g_Event.Signal();
}

nn::Result NewsSubscriber::Process(nn::os::Event* pCancelEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCancelEvent);

    Record records[ApplicationListCountMax] = {};
    int count;

    // この時点で登録されているアプリケーションについて処理を行う。
    {
        std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

        if (g_ReserveCount == 0)
        {
            NN_RESULT_SUCCESS;
        }

        std::memcpy(records, g_ReserveRecords, sizeof (Record) * g_ReserveCount);
        count = g_ReserveCount;

        g_ReserveCount = 0;
    }

    nn::nifm::NetworkConnection connection;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(connection.GetRequestHandle(),
        nn::nifm::RequirementPreset_InternetForSystemProcessSharable));

    connection.SubmitRequest();

    nn::os::WaitAny(pCancelEvent->GetBase(), connection.GetSystemEvent().GetBase());

    if (pCancelEvent->TryWait())
    {
        NN_RESULT_THROW(ResultSystemTerminateRequired());
    }
    if (!connection.IsAvailable())
    {
        std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

        for (int i = 0; i < count; i++)
        {
            AddApplication(&g_ReserveCount, records[i].appId, g_ReserveRecords, g_ReserveCount);
        }
        g_IsReconnectionRequired = true;

        NN_RESULT_THROW(ResultInternetRequestNotAccepted());
    }

    g_IsReconnectionRequired = false;

    bool retry = false;

    for (int i = 0; i < count; i++)
    {
        nn::Result result = Subscribe(records[i].appId, &connection);

        if (result.IsSuccess())
        {
            std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

            AddApplication(&g_ProcessedCount, records[i].appId, g_ProcessedRecords, g_ProcessedCount);
        }
        else if (ResultServerError404::Includes(result))
        {
            NN_DETAIL_NEWS_INFO("[news] The topic list is not found. (status 404)\n");

            std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

            // 居住国を一切指定しない場合、404 エラーになる。
            AddApplication(&g_ProcessedCount, records[i].appId, g_ProcessedRecords, g_ProcessedCount);
        }
        else
        {
            std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

            AddApplication(&g_ReserveCount, records[i].appId, g_ReserveRecords, g_ReserveCount);
            retry = true;
        }
    }

    if (retry)
    {
        // 1 時間後に再試行する。
        g_Event.StartOneShot(nn::TimeSpan::FromHours(1));
    }

    NN_RESULT_SUCCESS;
}

nn::os::TimerEvent& NewsSubscriber::GetEvent() NN_NOEXCEPT
{
    return g_Event;
}

void NewsSubscriber::NotifyAccountCountryListUpdated() NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    if (g_ProcessedCount == 0)
    {
        return;
    }

    for (int i = 0; i < g_ProcessedCount; i++)
    {
        AddApplication(&g_ReserveCount, g_ProcessedRecords[i].appId, g_ReserveRecords, g_ReserveCount);
    }

    g_ProcessedCount = 0;

    g_Event.Signal();
}

void NewsSubscriber::NotifyNetworkConnected() NN_NOEXCEPT
{
    if (g_IsReconnectionRequired)
    {
        g_Event.Signal();
    }
}

}}}}}
