﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_DownloaderThread.h>
#include <nn/news/detail/service/core/news_AccountCountryChecker.h>
#include <nn/news/detail/service/core/news_NewsTaskManager.h>
#include <nn/news/detail/service/core/news_NewsDownloader.h>
#include <nn/news/detail/service/core/news_NewsSubscriber.h>
#include <nn/news/detail/service/core/news_One2OneNotificationManager.h>
#include <nn/npns/npns_Api.h>
#include <nn/npns/npns_ApiSystem.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

namespace
{
    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK Bit8 g_ThreadStack[64 * 1024];

    nn::os::Event g_StopEvent(nn::os::EventClearMode_ManualClear);
}

namespace
{
    nn::os::Event g_NpnsConnectedEvent(nn::os::EventClearMode_ManualClear);

    nn::os::Event g_SystemRunnableEvent(nn::os::EventClearMode_ManualClear);
    std::atomic<bool> g_IsSystemRunnable(true);
}

namespace
{
    void RecoverStorage() NN_NOEXCEPT
    {
        StorageManager::GetInstance().Lock();

        NN_UTIL_SCOPE_EXIT
        {
            StorageManager::GetInstance().Unlock();
        };

        nn::Result result = StorageManager::Recover();

        if (result.IsFailure())
        {
            NN_DETAIL_NEWS_INFO("[news] StorageManager::Recover() failed. code = %03d-%04d\n",
                result.GetModule(), result.GetDescription());
        }
    }

    void ProcessTopicNewsDownload(nn::bgtc::Task& bgtcTask) NN_NOEXCEPT
    {
        NewsTaskManager::Record record;

        while (NewsTaskManager::GetInstance().GetCurrentTask(&record).IsSuccess())
        {
            NN_DETAIL_NEWS_INFO("[news] Task(%s) ...\n", record.topicId.value);

            nn::nifm::NetworkConnection connection;

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(connection.GetRequestHandle(),
                nn::nifm::RequirementPreset_InternetForSystemProcess));

            connection.SubmitRequest();

            nn::os::WaitAny(g_StopEvent.GetBase(), connection.GetSystemEvent().GetBase());

            if (g_StopEvent.TryWait())
            {
                return;
            }

            nn::Result result = NewsDownloader::Download(record.topicId, record.eTag, connection, bgtcTask, &g_StopEvent);

            if (result.IsSuccess())
            {
                NN_DETAIL_NEWS_INFO("[news] Task(%s) done!\n", record.topicId.value);
            }
            else
            {
                NN_DETAIL_NEWS_INFO("[news] Task(%s) done! code = %03d-%04d\n",
                    record.topicId.value, result.GetModule(), result.GetDescription());
            }

            NewsTaskManager::GetInstance().NotifyDone(record, result);
        }
    }

    void ProcessOne2OneNewsDownload(nn::bgtc::Task& bgtcTask) NN_NOEXCEPT
    {
        One2OneNotificationManager::Record record;

        while (One2OneNotificationManager::GetInstance().GetCurrentNotification(&record).IsSuccess())
        {
            NN_DETAIL_NEWS_INFO("[news] Task(One2One) ...\n");

            nn::nifm::NetworkConnection connection;

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(connection.GetRequestHandle(),
                nn::nifm::RequirementPreset_InternetForSystemProcess));

            connection.SubmitRequest();

            nn::os::WaitAny(g_StopEvent.GetBase(), connection.GetSystemEvent().GetBase());

            if (g_StopEvent.TryWait())
            {
                return;
            }

            nn::Result result = NewsDownloader::Download(record.url, connection, bgtcTask, &g_StopEvent);

            if (result.IsSuccess())
            {
                NN_DETAIL_NEWS_INFO("[news] Task(One2One) done!\n");
            }
            else
            {
                NN_DETAIL_NEWS_INFO("[news] Task(One2One) done! code = %03d-%04d\n", result.GetModule(), result.GetDescription());
            }

            One2OneNotificationManager::GetInstance().NotifyDone(record, result);
        }
    }

    void ProcessNpnsSubscription() NN_NOEXCEPT
    {
        TopicId topicId = {};

        while (NewsTaskManager::GetInstance().GetNpnsTopicToBeSubscribed(&topicId))
        {
            nn::npns::State state = nn::npns::GetState();

            if (!(state == nn::npns::State_Connected || state == nn::npns::State_ConnectedOnHalfAwake))
            {
                break;
            }

            NN_DETAIL_NEWS_INFO("[news] nn::npns::SubscribeTopic(%s)\n", topicId.value);

            nn::Result result = nn::npns::SubscribeTopic(topicId.value);

            if (result.IsSuccess())
            {
                NN_DETAIL_NEWS_INFO("[news] nn::npns::SubscribeTopic(%s) done!\n", topicId.value);
            }
            else
            {
                NN_DETAIL_NEWS_INFO("[news] nn::npns::SubscribeTopic(%s) done! code = %03d-%04d\n",
                    topicId.value, result.GetModule(), result.GetDescription());
            }

            NewsTaskManager::GetInstance().NotifyNpnsTopicSubscriptionProcessed(topicId, result);

            if (nn::npns::ResultCommandTimeOut::Includes(result))
            {
                // 以降の購読処理がタイムアウト待ちしないよう、速やかに終了する。
                return;
            }
        }

        while (NewsTaskManager::GetInstance().GetNpnsTopicToBeUnsubscribed(&topicId))
        {
            nn::npns::State state = nn::npns::GetState();

            if (!(state == nn::npns::State_Connected || state == nn::npns::State_ConnectedOnHalfAwake))
            {
                break;
            }

            NN_DETAIL_NEWS_INFO("[news] nn::npns::UnsubscribeTopic(%s)\n", topicId.value);

            nn::Result result = nn::npns::UnsubscribeTopic(topicId.value);

            if (result.IsSuccess())
            {
                NN_DETAIL_NEWS_INFO("[news] nn::npns::UnsubscribeTopic(%s) done!\n", topicId.value);
            }
            else
            {
                NN_DETAIL_NEWS_INFO("[news] nn::npns::UnsubscribeTopic(%s) done! code = %03d-%04d\n",
                    topicId.value, result.GetModule(), result.GetDescription());

                // すでに未購読の状態だったら成功扱いとする。
                if (nn::npns::ResultTopicNotSubscribed::Includes(result) || nn::npns::ResultTopicNotFound::Includes(result))
                {
                    result = nn::ResultSuccess();
                }
            }

            NewsTaskManager::GetInstance().NotifyNpnsTopicUnsubscriptionProcessed(topicId, result);

            if (nn::npns::ResultCommandTimeOut::Includes(result))
            {
                // 以降の購読解除処理がタイムアウト待ちしないよう、速やかに終了する。
                return;
            }
        }
    }

    void WorkerThread(void*) NN_NOEXCEPT
    {
        RecoverStorage();

        NN_ABORT_UNLESS_RESULT_SUCCESS(AccountCountryChecker::GetInstance().Initialize());

        if (Account::IsAvailableNintendoAccountLinked())
        {
            NewsTaskManager::GetInstance().NotifyNintendoAccountLinked();
        }

        {
            bool isSystemUpdateRequired;
            NN_ABORT_UNLESS_RESULT_SUCCESS(Version::GetInstance().IsSystemUpdateRequired(&isSystemUpdateRequired));

            if (!isSystemUpdateRequired)
            {
                NewsTaskManager::GetInstance().NotifySystemUpdated();
            }
        }

        nn::bgtc::Task bgtcTask;
        NN_ABORT_UNLESS_RESULT_SUCCESS(bgtcTask.Initialize());

        NN_UTIL_SCOPE_EXIT
        {
            bgtcTask.Finalize();
        };

        while (NN_STATIC_CONDITION(true))
        {
            while (!g_IsSystemRunnable)
            {
                nn::os::WaitAny(g_StopEvent.GetBase(), g_SystemRunnableEvent.GetBase());

                if (g_StopEvent.TryWait())
                {
                    return;
                }

                g_SystemRunnableEvent.Clear();
            }

            nn::os::WaitAny(g_StopEvent.GetBase(),
                NewsTaskManager::GetInstance().GetScheduleEvent().GetBase(),
                NewsTaskManager::GetInstance().GetSubscriptionEvent().GetBase(),
                One2OneNotificationManager::GetInstance().GetEvent().GetBase(),
                NewsSubscriber::GetEvent().GetBase(),
                g_NpnsConnectedEvent.GetBase());

            if (g_StopEvent.TryWait())
            {
                return;
            }
            if (!g_IsSystemRunnable)
            {
                continue;
            }

            nn::Result bgtcResult = bgtcTask.NotifyStarting();

            if (bgtcResult.IsFailure())
            {
                NN_DETAIL_NEWS_INFO("[news] nn::bgtc::Task::NotifyStarting() failed. code = %03d-%04d\n",
                    bgtcResult.GetModule(), bgtcResult.GetDescription());
            }

            NN_UTIL_SCOPE_EXIT
            {
                if (bgtcResult.IsSuccess())
                {
                    bgtcTask.NotifyFinished();
                }
            };

            if (NewsTaskManager::GetInstance().GetScheduleEvent().TryWait())
            {
                NN_DETAIL_NEWS_INFO("[news] (!) NewsTaskManager::ScheduleEvent was signaled.\n");
                NewsTaskManager::GetInstance().GetScheduleEvent().Clear();

                NewsTaskManager::GetInstance().Schedule();
                ProcessTopicNewsDownload(bgtcTask);
            }
            if (NewsTaskManager::GetInstance().GetSubscriptionEvent().TryWait() || g_NpnsConnectedEvent.TryWait())
            {
                if (NewsTaskManager::GetInstance().GetSubscriptionEvent().TryWait())
                {
                    NN_DETAIL_NEWS_INFO("[news] (!) NewsTaskManager::SubscriptionEvent was signaled.\n");
                    NewsTaskManager::GetInstance().GetSubscriptionEvent().Clear();
                }
                if (g_NpnsConnectedEvent.TryWait())
                {
                    NN_DETAIL_NEWS_INFO("[news] (!) NpnsConnectedEvent was signaled.\n");
                    g_NpnsConnectedEvent.Clear();
                }

                ProcessNpnsSubscription();
            }
            if (One2OneNotificationManager::GetInstance().GetEvent().TryWait())
            {
                NN_DETAIL_NEWS_INFO("[news] (!) One2OneNotificationManager::Event was signaled.\n");
                One2OneNotificationManager::GetInstance().GetEvent().Clear();

                ProcessOne2OneNewsDownload(bgtcTask);
            }
            if (NewsSubscriber::GetEvent().TryWait())
            {
                NN_DETAIL_NEWS_INFO("[news] (!) NewsSubscriber::Event was signaled.\n");
                NewsSubscriber::GetEvent().Clear();

                NewsSubscriber::Process(&g_StopEvent);
            }
        }
    }
}

void DownloaderThread::Start() NN_NOEXCEPT
{
    g_StopEvent.Clear();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
        g_ThreadStack, sizeof (g_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(news, Downloader)));

    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(news, Downloader));
    nn::os::StartThread(&g_Thread);
}

void DownloaderThread::Stop() NN_NOEXCEPT
{
    g_StopEvent.Signal();

    nn::os::DestroyThread(&g_Thread);
}

bool DownloaderThread::GetNextScheduleInterval(nn::TimeSpan* outInterval) NN_NOEXCEPT
{
    // 実行可能なタスクが存在すれば、即座に起きるようにする。
    {
        NewsTaskManager::Record record = {};

        if (NewsTaskManager::GetInstance().GetCurrentTask(&record).IsSuccess())
        {
            *outInterval = nn::TimeSpan(0);
            return true;
        }
    }
    {
        One2OneNotificationManager::Record record = {};

        if (One2OneNotificationManager::GetInstance().GetCurrentNotification(&record).IsSuccess())
        {
            *outInterval = nn::TimeSpan(0);
            return true;
        }
    }

    nn::TimeSpan interval1;
    nn::TimeSpan interval2;

    bool result1 = NewsTaskManager::GetInstance().GetNextScheduleInterval(&interval1);
    bool result2 = One2OneNotificationManager::GetInstance().GetNextScheduleInterval(&interval2);

    if (!(result1 || result2))
    {
        return false;
    }

    *outInterval = (result1 && result2) ? std::min(interval1, interval2) : (result1 ? interval1 : interval2);

    return true;
}

void DownloaderThread::NotifyNpnsConnected() NN_NOEXCEPT
{
    g_NpnsConnectedEvent.Signal();
}

void DownloaderThread::NotifyNetworkConnected() NN_NOEXCEPT
{
    NewsTaskManager::GetInstance().NotifyNetworkConnected();
    One2OneNotificationManager::GetInstance().NotifyNetworkConnected();
}

void DownloaderThread::NotifySystemRunnableStateChanged(bool isRunnable) NN_NOEXCEPT
{
    g_IsSystemRunnable = isRunnable;

    g_SystemRunnableEvent.Signal();
}

}}}}}
