﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/news_ShimLibraryGlobal.h>
#include <nn/news/news_Result.h>
#include <mutex>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace news { namespace detail {

ShimLibraryGlobal::ShimLibraryGlobal() NN_NOEXCEPT :
    m_Session(nullptr),
    m_NewlyArrivedEventHolder(nullptr),
    m_OverwriteEventHolder(nullptr)
{
}

void ShimLibraryGlobal::Initialize() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_Session)
    {
        CreateIpcSession();
    }
}

detail::ipc::INewsService* ShimLibraryGlobal::GetSession() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_Session)
    {
        CreateIpcSession();
    }

    return m_Session;
}

nn::Result ShimLibraryGlobal::RegisterNewlyArrivedEvent(nn::os::SystemEvent* outEvent) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_Session)
    {
        CreateIpcSession();
    }

    NN_RESULT_THROW_UNLESS(!m_NewlyArrivedEventHolder, ResultAlreadyRegistered());

    detail::ipc::INewlyArrivedEventHolder* holder = nullptr;

    NN_RESULT_DO(detail::ipc::CreateNewlyArrivedEventHolder(&holder));

    NN_UTIL_SCOPE_EXIT
    {
        if (holder)
        {
            nn::sf::ReleaseSharedObject(holder);
        }
    };

    nn::sf::NativeHandle handle;

    NN_RESULT_DO(holder->Get(&handle));

    outEvent->AttachReadableHandle(handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_ManualClear);
    handle.Detach();

    m_NewlyArrivedEventHolder = holder;
    holder = nullptr;

    NN_RESULT_SUCCESS;
}

nn::Result ShimLibraryGlobal::RegisterOverwriteEvent(nn::os::SystemEvent* outEvent) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_Session)
    {
        CreateIpcSession();
    }

    NN_RESULT_THROW_UNLESS(!m_OverwriteEventHolder, ResultAlreadyRegistered());

    detail::ipc::IOverwriteEventHolder* holder = nullptr;

    NN_RESULT_DO(detail::ipc::CreateOverwriteEventHolder(&holder));

    NN_UTIL_SCOPE_EXIT
    {
        if (holder)
        {
            nn::sf::ReleaseSharedObject(holder);
        }
    };

    nn::sf::NativeHandle handle;

    NN_RESULT_DO(holder->Get(&handle));

    outEvent->AttachReadableHandle(handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_ManualClear);
    handle.Detach();

    m_OverwriteEventHolder = holder;
    holder = nullptr;

    NN_RESULT_SUCCESS;
}

nn::Result ShimLibraryGlobal::CreateNewsDataService(detail::ipc::INewsDataService** outService) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_Session)
    {
        CreateIpcSession();
    }

    NN_RESULT_DO(detail::ipc::CreateNewsDataService(outService));

    NN_RESULT_SUCCESS;
}

nn::Result ShimLibraryGlobal::CreateNewsDatabaseService(detail::ipc::INewsDatabaseService** outService) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_Session)
    {
        CreateIpcSession();
    }

    NN_RESULT_DO(detail::ipc::CreateNewsDatabaseService(outService));

    NN_RESULT_SUCCESS;
}

void ShimLibraryGlobal::CreateIpcSession() NN_NOEXCEPT
{
    detail::ipc::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::CreateNewsService(&m_Session));
}

}}}
