﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/mii/mii_ImageDatabase.h>
#include <nn/mii/mii_Result.h>
#include <nn/mii/detail/service/mii_ImageServiceProviderClient.h>
#include <nn/mii/detail/service/mii_IImageDatabaseService.sfdl.h>

#include <nn/nn_Abort.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <new>
#include <mutex>

namespace nn { namespace mii {

NN_DEFINE_STATIC_CONSTANT(const int ImageDatabase::ImageWidth);
NN_DEFINE_STATIC_CONSTANT(const int ImageDatabase::ImageHeight);
NN_DEFINE_STATIC_CONSTANT(const int ImageDatabase::ImageSize);
NN_DEFINE_STATIC_CONSTANT(const int ImageDatabase::MaxCount);
NN_DEFINE_STATIC_CONSTANT(const int ImageDatabase::MaxFileSize);

namespace
{
    // 各種サイズチェック
    NN_STATIC_ASSERT(sizeof(ImageAttribute) == 68);
    NN_STATIC_ASSERT(ImageDatabase::ImageWidth == detail::ImageDatabaseImpl::ImageWidth);
    NN_STATIC_ASSERT(ImageDatabase::ImageHeight == detail::ImageDatabaseImpl::ImageHeight);
    NN_STATIC_ASSERT(ImageDatabase::ImageSize == detail::ImageDatabaseImpl::ImageSize);
    NN_STATIC_ASSERT(ImageDatabase::MaxCount == detail::ImageDatabaseImpl::MaxCount);
    NN_STATIC_ASSERT(ImageDatabase::MaxFileSize == detail::ImageDatabaseImpl::MaxFileSize);

    nn::sf::SharedPointer<detail::IImageDatabaseService> GetServiceObject() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(nn::os::SdkMutex, s_Mutex);
        NN_FUNCTION_LOCAL_STATIC(detail::ImageServiceProviderClient, s_ImageServiceProviderClient);
        NN_FUNCTION_LOCAL_STATIC(nn::sf::SharedPointer<detail::IImageDatabaseService>, s_pImageDatabaseService, = nullptr);

        std::lock_guard<nn::os::SdkMutex> lock(s_Mutex);

        if(s_pImageDatabaseService)
        {
            return s_pImageDatabaseService;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            s_ImageServiceProviderClient.GetImageDatabaseServiceSharedPointer(&s_pImageDatabaseService));

        return s_pImageDatabaseService;
    }
}

ImageDatabase::ImageDatabase() NN_NOEXCEPT
    : m_IsInitialized(false)
{
}

ImageDatabase::~ImageDatabase() NN_NOEXCEPT
{
}

nn::Result ImageDatabase::Initialize(DatabaseResult* pResult, bool enableFormat) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pResult);

    int8_t outResult;
    NN_RESULT_DO( GetServiceObject()->Initialize(&outResult, enableFormat) );

    m_IsInitialized = true;
    *pResult = static_cast<DatabaseResult>(outResult);
    NN_RESULT_SUCCESS;
}

bool ImageDatabase::IsInitialized() const NN_NOEXCEPT
{
    return m_IsInitialized;
}

nn::Result ImageDatabase::Reload() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    return GetServiceObject()->Reload();
}

int ImageDatabase::GetCount() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    int32_t outValue;
    NN_ABORT_UNLESS_RESULT_SUCCESS( GetServiceObject()->GetCount(&outValue) );
    return outValue;
}

bool ImageDatabase::IsEmpty() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    bool outValue;
    NN_ABORT_UNLESS_RESULT_SUCCESS( GetServiceObject()->IsEmpty(&outValue) );
    return outValue;
}

bool ImageDatabase::IsFull() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    bool outValue;
    NN_ABORT_UNLESS_RESULT_SUCCESS( GetServiceObject()->IsFull(&outValue) );
    return outValue;
}

void ImageDatabase::GetAttribute(ImageAttribute* pAttribute, int index) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES_NOT_NULL(pAttribute);
    NN_SDK_REQUIRES_RANGE(index, 0, GetCount());

    nn::mii::detail::ImageAttributeImpl outValue;
    NN_ABORT_UNLESS_RESULT_SUCCESS( GetServiceObject()->GetAttribute(&outValue, static_cast<int32_t>(index)) );

    pAttribute->id = outValue.id;
    pAttribute->createId = outValue.createId;
    pAttribute->expression = static_cast<nn::mii::Expression>(outValue.expression);
    pAttribute->poseType = static_cast<nn::mii::ImagePoseType>(outValue.poseType);
    pAttribute->fontRegion = static_cast<nn::mii::FontRegion>(outValue.fontRegion);
    pAttribute->nickname = outValue.nickname;
}

nn::Result ImageDatabase::LoadImage(void* pOut, size_t size, const nn::util::Uuid& id) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES_EQUAL(size,size_t(ImageSize));
    NN_SDK_REQUIRES(id != nn::util::InvalidUuid);

    nn::sf::OutBuffer outBuffer(static_cast<char*>(pOut), size);
    return GetServiceObject()->LoadImage(outBuffer, id);
}

nn::Result ImageDatabase::AddOrUpdateImage(int* pOutIndex
                    , Expression expression
                    , ImagePoseType pose
                    , FontRegion fontRegion
                    , const Nickname& nickname
                    , const nn::mii::CreateId& createId
                    , const nn::util::Uuid* pUsedUuid
                    , const nn::util::Uuid* pValidUuidArray
                    , int validUuidArrayCount
                    , const void* pSrc
                    , int size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES_NOT_NULL(pOutIndex);
    NN_SDK_REQUIRES_RANGE(expression, 0, nn::mii::Expression_End);
    NN_SDK_REQUIRES_RANGE(pose, 0, nn::mii::ImagePoseType_End);
    NN_SDK_REQUIRES_RANGE(fontRegion, 0, FontRegion_End);
    NN_SDK_REQUIRES(nickname.IsValid());
    NN_SDK_REQUIRES(createId.IsValid());
    NN_SDK_REQUIRES_RANGE(validUuidArrayCount, 0, MaxCount);
    NN_SDK_REQUIRES_NOT_NULL(pSrc);
    NN_SDK_REQUIRES_EQUAL(size, int(ImageSize));

    nn::util::Uuid usedUuid = pUsedUuid == nullptr ? nn::util::InvalidUuid : *pUsedUuid; // nullptrなら InvalidUuid を指定
    nn::sf::InArray<nn::util::Uuid> validUuidArray(pValidUuidArray, validUuidArrayCount);
    nn::sf::InBuffer inBuffer(static_cast<const char*>(pSrc), static_cast<size_t>(size));

    int32_t outIndex;

    NN_RESULT_DO(GetServiceObject()->AddOrUpdateImage(
        &outIndex,
        static_cast<int8_t>(expression),
        static_cast<int8_t>(pose),
        static_cast<int8_t>(fontRegion),
        nickname,
        createId,
        usedUuid,
        validUuidArray,
        inBuffer));

    *pOutIndex = outIndex;
    NN_RESULT_SUCCESS;
}

nn::Result ImageDatabase::DeleteImages(const nn::util::Uuid* pIds, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES_NOT_NULL(pIds);
    NN_SDK_REQUIRES_MINMAX(count, 1, MaxCount);

    nn::sf::InArray<nn::util::Uuid> inArray(pIds, static_cast<size_t>(count));
    return GetServiceObject()->DeleteImages(inArray);
}

// --------------------------------------------
// static 関数
// --------------------------------------------

void ImageDatabase::DeleteFile() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( GetServiceObject()->DeleteFile());
}

void ImageDatabase::DestroyFile() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( GetServiceObject()->DestroyFile());
}

void ImageDatabase::ImportFile(const void* pSrc,size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);

    nn::sf::InBuffer inBuffer(static_cast<const char*>(pSrc), size);
    NN_ABORT_UNLESS_RESULT_SUCCESS( GetServiceObject()->ImportFile(inBuffer) );
}

size_t ImageDatabase::ExportFile(void* pOut,size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    int64_t outSize;
    nn::sf::OutBuffer outBuffer(static_cast<char*>(pOut), size);
    NN_ABORT_UNLESS_RESULT_SUCCESS( GetServiceObject()->ExportFile(&outSize, outBuffer) );
    return static_cast<size_t>(outSize);
}

// --------------------------------------------
// テストでのみ使用する関数
// --------------------------------------------
nn::Result ForceInitializeImageDatabase(DatabaseResult* pResult, bool enableFormat) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pResult);

    int8_t outResult;
    NN_RESULT_DO( GetServiceObject()->ForceInitialize(&outResult, enableFormat) );

    *pResult = static_cast<DatabaseResult>(outResult);
    NN_RESULT_SUCCESS;
};

}}
