﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/mii/detail/service/mii_DatabaseServiceInterface.h>

#include <nn/mii/detail/mii_DatabaseImpl.h>
#include <nn/mii/mii_Result.h>
#include <nn/mii/mii_PrivateResult.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>


namespace nn
{
namespace mii
{
namespace detail
{

nn::Result DatabaseServiceInterface::CheckDatabaseTestModeEnabled() NN_NOEXCEPT
{
#if defined (NN_BUILD_CONFIG_OS_HORIZON)
    bool isEnabled = false;
    auto outBytes = nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "mii", "is_db_test_mode_enabled");
    NN_SDK_ASSERT_EQUAL(outBytes, sizeof(isEnabled));
    NN_UNUSED(outBytes);
    NN_RESULT_THROW_UNLESS(isEnabled, ResultDevelopmentOnly());
#else
    // HORIZON 以外では常にテストモード有効
#endif

    NN_RESULT_SUCCESS;
}

DatabaseServiceInterface::DatabaseServiceInterface(
    DatabaseImpl* pDatabaseImpl,
    int specialMiiKeyCode,
    bool isEditor
) NN_NOEXCEPT
    : m_pDatabaseImpl(pDatabaseImpl),
      m_IsEditor(isEditor)
{
    NN_SDK_ASSERT_NOT_NULL(m_pDatabaseImpl);

    m_pDatabaseImpl->SetupContext(&m_DatabaseContext, specialMiiKeyCode);
}

DatabaseServiceInterface::~DatabaseServiceInterface() NN_NOEXCEPT
{
}

nn::Result DatabaseServiceInterface::IsUpdated(nn::sf::Out<bool>outIsUpdated, int sourceFlags) NN_NOEXCEPT
{
    outIsUpdated.Set(m_pDatabaseImpl->IsUpdated(&m_DatabaseContext, sourceFlags));

    NN_RESULT_SUCCESS;
}

nn::Result DatabaseServiceInterface::IsFullDatabase(nn::sf::Out<bool>outIsFullDatabase) NN_NOEXCEPT
{
    outIsFullDatabase.Set(m_pDatabaseImpl->IsFullDatabase());

    NN_RESULT_SUCCESS;
}

nn::Result DatabaseServiceInterface::GetCount(nn::sf::Out<int>outCount, int sourceFlags) NN_NOEXCEPT
{
    outCount.Set(m_pDatabaseImpl->GetCount(sourceFlags, m_DatabaseContext));

    NN_RESULT_SUCCESS;
}

nn::Result DatabaseServiceInterface::Get(nn::sf::Out<int> outCount, nn::sf::OutArray<CharInfoElement> outElements, int sourceFlags) NN_NOEXCEPT
{
    return m_pDatabaseImpl->Get(outCount.GetPointer(), outElements.GetData(), static_cast<int>(outElements.GetLength()), sourceFlags, m_DatabaseContext);
}

nn::Result DatabaseServiceInterface::Get1(nn::sf::Out<int> outCount, nn::sf::OutArray<CharInfo> outElements, int sourceFlags) NN_NOEXCEPT
{
    return m_pDatabaseImpl->Get(outCount.GetPointer(), outElements.GetData(), static_cast<int>(outElements.GetLength()), sourceFlags, m_DatabaseContext);
}

nn::Result DatabaseServiceInterface::UpdateLatest(nn::sf::Out<CharInfo> outInfo, const CharInfo& info, int sourceFlags) NN_NOEXCEPT
{
    *outInfo.GetPointer() = info;

    return m_pDatabaseImpl->UpdateLatest(outInfo.GetPointer(), sourceFlags, m_DatabaseContext);
}

nn::Result DatabaseServiceInterface::BuildRandom(nn::sf::Out<CharInfo> outInfo, int age, int gender, int race) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(Age_Min    <= age &&     age    <= Age_End,      nn::mii::ResultInvalid());
    NN_RESULT_THROW_UNLESS(Gender_Min <= gender &&  gender <= Gender_End,   nn::mii::ResultInvalid());
    NN_RESULT_THROW_UNLESS(Race_Min   <= race &&    race   <= Race_End,     nn::mii::ResultInvalid());

    RandomDesc desc;
    desc.SetAge(static_cast<Age>(age));
    desc.SetGender(static_cast<Gender>(gender));
    desc.SetRace(static_cast<Race>(race));
    m_pDatabaseImpl->BuildRandom(outInfo.GetPointer(), desc);

    NN_RESULT_SUCCESS;
}

nn::Result DatabaseServiceInterface::BuildDefault(nn::sf::Out<CharInfo> outInfo, int index) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(0 <= index && index < nn::mii::DefaultMiiCount, nn::mii::ResultInvalid());

    m_pDatabaseImpl->BuildDefault(outInfo.GetPointer(), index);

    NN_RESULT_SUCCESS;
}

nn::Result DatabaseServiceInterface::Get2(nn::sf::Out<int> outCount, nn::sf::OutArray<StoreDataElement> outElements, int sourceFlags) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsEditor, ResultPermissionDenied());

    return m_pDatabaseImpl->Get(outCount.GetPointer(), outElements.GetData(), static_cast<int>(outElements.GetLength()), sourceFlags, m_DatabaseContext);
}

nn::Result DatabaseServiceInterface::Get3(nn::sf::Out<int> outCount, nn::sf::OutArray<StoreData> outElements, int sourceFlags) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsEditor, ResultPermissionDenied());

    return m_pDatabaseImpl->Get(outCount.GetPointer(), outElements.GetData(), static_cast<int>(outElements.GetLength()), sourceFlags, m_DatabaseContext);
}

nn::Result DatabaseServiceInterface::UpdateLatest1(nn::sf::Out<StoreData> outStoreData, const StoreData& storeData, int sourceFlags) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsEditor, ResultPermissionDenied());

    *outStoreData.GetPointer() = storeData;

    return m_pDatabaseImpl->UpdateLatest(outStoreData.GetPointer(), sourceFlags, m_DatabaseContext);
}

nn::Result DatabaseServiceInterface::FindIndex(nn::sf::Out<int>outIndex, const CreateId& id, bool enabledSpecialMii) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsEditor, ResultPermissionDenied());

    outIndex.Set(m_pDatabaseImpl->FindIndex(id, enabledSpecialMii));

    NN_RESULT_SUCCESS;
}

nn::Result DatabaseServiceInterface::Move(int newIndex, const CreateId& id) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsEditor, ResultPermissionDenied());

    int count = m_pDatabaseImpl->GetCount(SourceFlag_Database, m_DatabaseContext);
    NN_RESULT_THROW_UNLESS(0 <= newIndex && newIndex < count, nn::mii::ResultInvalid());

    return m_pDatabaseImpl->Move(&m_DatabaseContext, newIndex, id);
}

nn::Result DatabaseServiceInterface::AddOrReplace(const StoreData& storeData) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsEditor, ResultPermissionDenied());

    return m_pDatabaseImpl->AddOrReplace(&m_DatabaseContext, storeData);
}

nn::Result DatabaseServiceInterface::Delete(const CreateId& id) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsEditor, ResultPermissionDenied());

    return m_pDatabaseImpl->Delete(&m_DatabaseContext, id);
}

nn::Result DatabaseServiceInterface::DestroyFile() NN_NOEXCEPT
{
    NN_RESULT_DO(CheckDatabaseTestModeEnabled());

    return m_pDatabaseImpl->DestroyFile(&m_DatabaseContext);
}

nn::Result DatabaseServiceInterface::DeleteFile() NN_NOEXCEPT
{
    NN_RESULT_DO(CheckDatabaseTestModeEnabled());

    return m_pDatabaseImpl->DeleteFile();
}

nn::Result DatabaseServiceInterface::Format() NN_NOEXCEPT
{
    NN_RESULT_DO(CheckDatabaseTestModeEnabled());

    m_pDatabaseImpl->Format(&m_DatabaseContext);

    NN_RESULT_SUCCESS;
}

nn::Result DatabaseServiceInterface::Import(const nn::sf::InBuffer& inBuffer) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckDatabaseTestModeEnabled());

    return m_pDatabaseImpl->Import(&m_DatabaseContext, inBuffer.GetPointerUnsafe(), inBuffer.GetSize());
}

nn::Result DatabaseServiceInterface::Export(const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckDatabaseTestModeEnabled());

    m_pDatabaseImpl->Export(outBuffer.GetPointerUnsafe(), outBuffer.GetSize());

    NN_RESULT_SUCCESS;
}

nn::Result DatabaseServiceInterface::IsBrokenDatabaseWithClearFlag(nn::sf::Out<bool> outIsBrokenDatabase) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsEditor, ResultPermissionDenied());

    outIsBrokenDatabase.Set(m_pDatabaseImpl->IsBrokenDatabaseWithClearFlag());

    NN_RESULT_SUCCESS;
}

nn::Result DatabaseServiceInterface::GetIndex(nn::sf::Out<int> outIndex, const CharInfo& info) const NN_NOEXCEPT
{
    return m_pDatabaseImpl->GetIndex(outIndex.GetPointer(), info, m_DatabaseContext);
}

nn::Result DatabaseServiceInterface::SetInterfaceVersion(int version) NN_NOEXCEPT
{
    m_pDatabaseImpl->SetInterfaceVersion(&m_DatabaseContext, version);
    NN_RESULT_SUCCESS;
}

nn::Result DatabaseServiceInterface::Convert(nn::sf::Out<CharInfo> outInfo, const Ver3StoreData& ver3StoreData) NN_NOEXCEPT
{
    return m_pDatabaseImpl->Convert(outInfo.GetPointer(), ver3StoreData);
}

}
}
}
