﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <nn/nn_Common.h>
#include "../detail/lm_LogDataChunk.h"
#include "../detail/lm_LogPacketHeader.h"

namespace nn { namespace lm { namespace impl {

class LogPacketParser
{
public:
    /*!
        @brief  ParsePacket 関数のコールバックです。

        @param[in]  header      取り出したパケットのヘッダ。
        @param[in]  pPayload    取り出したパケットのペイロードを指すポインタ。
        @param[in]  payloadSize 取り出したパケットのペイロードのサイズ。
        @param[in]  argument    引数。

        @return     パケットの取り出しを継続するときは ture、中断するときは false を返してください。

        @details
                    ParsePacket 関数が、バッファからパケットを取り出すたびに呼び出されます。
    */
    using ParsePacketCallback = bool (*)(const detail::LogPacketHeader& header, const void* pPayload, size_t payloadSize, void* argument);

    /*!
        @brief  ParseDataChunk 関数のコールバックです。

        @param[in]  key         取り出したデータチャンクのキー。
        @param[in]  pChunk      取り出したデータチャンクを指すポインタ。
        @param[in]  chunkSize   取り出したデータチャンクのサイズ。
        @param[in]  argument    引数。

        @return       データチャンクの取り出しを継続するときは ture、中断するときは false を返してください。

        @details
                    ParseDataChunk 関数が、パケットからデータチャンクを取り出すたびに呼び出されます。
    */
    using ParseDataChunkCallback = bool (*)(detail::LogDataChunkKey key, const void* pChunk, size_t chunkSize, void* argument);

    /*!
        @brief  ParseTextLog 関数のコールバックです。

        @param[in]  pText       取り出したテキストログを指すポインタ。
        @param[in]  TextSize    取り出したテキストログのサイズ。
        @param[in]  argument    引数。

        @details
                    ParseTextLog 関数が、パケットからデータチャンクを取り出すたびに呼び出されます。
    */
    using ParseTextLogCallback = void (*)(const char* pText, size_t textSize, void* argument);

public:
    /*!
        @brief  ログパケットから、指定したキーのデータチャンクを取り出します。

        @param[out] out         データチャンクのポインタを格納するポインタ。
        @param[out] outSize     データチャンクのサイズ。
        @param[in]  key         取り出すデータチャンクのキー。
        @param[in]  buffer      ログパケットのバッファ。
        @param[in]  bufferSize  ログパケットのバッファのサイズ。

        @return     取り出せたときは true。

        @details
                    指定したキーのデータチャンクを含む場合でも、データチャンクが分割されているときは、
                    データチャンクを取り出せないことに注意してください。
    */
    static bool FindDataChunk(const void** out, size_t* outSize, detail::LogDataChunkKey key, const void* buffer, size_t bufferSize) NN_NOEXCEPT;

    /*!
        @brief  バッファから、ログパケットを取り出します。

        @param[in]  buffer      パケットを取り出すバッファを指すポインタ。
        @param[in]  bufferSize  パケットを取り出すバッファのサイズ。
        @param[in]  callbck     パケットを取り出すたびに呼ばれるコールバック。
        @param[in]  argument    コールバックで受け取る引数。

        @return     バッファ全体の解析が完了したときは true、中断されたときは false。
    */
    static bool ParsePacket(const void* buffer, size_t bufferSize, ParsePacketCallback callback, void* argument) NN_NOEXCEPT;

    /*!
        @brief  ログパケットのペイロードから、データチャンクを取り出します。

        @param[in]  pPayload    データチャンクを取り出すペイロードを指すポインタ。
        @param[in]  payloadSize データチャンクを取り出すペイロードのサイズ。
        @param[in]  callbck     データチャンクを取り出すたびに呼ばれるコールバック。
        @param[in]  argument    コールバックで受け取る引数。

        @return     ペイロード全体の解析が完了したときは true、中断されたときは false。
    */
    static bool ParseDataChunk(const void* pPayload, size_t payloadSize, ParseDataChunkCallback callback, void* argument) NN_NOEXCEPT;

    /*!
        @brief  ログパケットから、テキストログを取り出します。

        @param[in]  pPacket     テキストログを取り出すパケットを指すポインタ。
        @param[in]  packetSize  テキストログを取り出すパケットのサイズ。
        @param[in]  callbck     テキストログを取り出すたびに呼ばれるコールバック。
        @param[in]  argument    コールバックで受け取る引数。

        @details
                    分割されたテキストログを含むパケットを連続して入力したときは、可能な限り分割されたログも取り出します。
                    テキストログが含まれない、または取り出せないときは、コールバックが呼び出されずに返ります。

        @attention
                    この関数はマルチスレッドで利用できません。
    */
    static void ParseTextLogWithContext(const void* pPacket, size_t packetSize, ParseTextLogCallback callback, void* argument) NN_NOEXCEPT;

    /*!
        @brief  ログパケットから、モジュール名を取り出します。モジュール名はヌル終端されます。

        @param[out] buffer      取り出したモジュール名を格納するバッファを指すポインタ。
        @param[in]  bufferSize  @a buffer の指す領域のバイトサイズ。
        @param[in]  pPacket     モジュール名を取り出すパケットを指すポインタ。
        @param[in]  packetSize  モジュール名を取り出すパケットのサイズ。

        @return     取り出したモジュール名のバイトサイズ。終端文字のサイズを含みません。

        @pre
            - buffer != nullptr
            - bufferSize > 0
            - pPacket != nullptr
            - packetSize > 0

        @details
                    パケットにモジュール名が含まれないときは空文字列を格納します。
    */
    static size_t ParseModuleName(char* buffer, size_t bufferSize, const void* pPacket, size_t packetSize) NN_NOEXCEPT;

private:
    static bool ParseDataChunkWithoutSizeCheck(const void* pPayload, size_t payloadSize, ParseDataChunkCallback callback, void* argument) NN_NOEXCEPT;
};

}}} // nn::lm::detail
