﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_ConditionVariable.h>

namespace nn { namespace lm { namespace impl {

// @brief   ログを一時的に格納するためのバッファです。
// @detail  引数に与えられた配列を、ログ格納用のバッファとして使用します。
//          バッファはダブルバッファリングで実装されています。
//
//          マルチスレッドでの書き込みとフラッシュに対応しています。
//          フラッシュはマルチスレッドで利用できますが、他のスレッドのフラッシュを排他します。
class LogBuffer
{
    NN_DISALLOW_COPY( LogBuffer );
    NN_DISALLOW_MOVE( LogBuffer );

public:
    typedef bool (*FlushFunction)(const uint8_t* buffer, size_t bufferSize);

    explicit LogBuffer(void* buffer, size_t bufferSize, FlushFunction function) NN_NOEXCEPT;

    static LogBuffer& GetDefaultInstance() NN_NOEXCEPT;

    bool Push(const void* pSrc, size_t srcSize) NN_NOEXCEPT;
    bool TryPush(const void* pSrc, size_t srcSize) NN_NOEXCEPT;
    void CancelPush() NN_NOEXCEPT;
    bool Flush() NN_NOEXCEPT;
    bool TryFlush() NN_NOEXCEPT;

private:
    struct BufferInfo
    {
        uint8_t*    m_pHead;
        size_t      m_StoredSize;
        size_t      m_ReferenceCount;

        BufferInfo() NN_NOEXCEPT :
            m_pHead(nullptr),
            m_StoredSize(0u),
            m_ReferenceCount(0u)
        {}
    };

    BufferInfo  m_Buffers[2];
    BufferInfo* m_pPushBuffer;
    BufferInfo* m_pFlushBuffer;

    size_t      m_BufferSize;

    FlushFunction m_FlushFunction;

    os::Mutex m_PushBufferMutex;
    os::Mutex m_FlushBufferMutex;

    os::ConditionVariable   m_ConditionPushReady;
    os::ConditionVariable   m_ConditionFlushReady;

    bool    m_PushCanceled;
    size_t  m_PushReadyWaitCount;

    bool PushImpl(const void* pSrc, size_t srcSize, bool blocking) NN_NOEXCEPT;
    bool FlushImpl(bool blocking) NN_NOEXCEPT;
};

}}} // nn::lm::impl
