﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include "lm_EventLogTransmitter.h"
#include "lm_LogBuffer.h"
#include "../detail/lm_Log.h"
#include "../detail/lm_LogPacketHeader.h"
#include "../detail/lm_LogPacketTransmitter.h"

namespace nn { namespace lm { namespace impl {

namespace
{
    bool DefaultFlushFunction(const uint8_t* message, size_t messageSize) NN_NOEXCEPT
    {
        return LogBuffer::GetDefaultInstance().TryPush(message, messageSize);
    }

    const size_t TransmitterBufferAlign = NN_ALIGNOF(detail::LogPacketHeader);

    // SessionBegin/End フラグの送信に必要な最小のペイロードサイズは 3Byte = key(1Byte) + size(1Byte) + 値(1Byte)
    const size_t TransmitterBufferSizeForSessionInfo = detail::LogPacketHeaderSize + 3;

    // LogPacketDropCount の送信に必要な最小のペイロードサイズは 6Byte = key(1Byte) + size(1Byte) + 値(8Byte)
    const size_t TransmitterBufferSizeForDropCount = detail::LogPacketHeaderSize + 10;
}

EventLogTransmitter::EventLogTransmitter(FlushFunction function) NN_NOEXCEPT
    : m_FlushFunction(function)
    , m_LogPacketDropCount(0u)
    , m_LogPacketDropCountMutex(false)
{
    NN_SDK_ASSERT_NOT_NULL(function);
}

EventLogTransmitter& EventLogTransmitter::GetDefaultInstance() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(EventLogTransmitter, instance, (DefaultFlushFunction));
    return instance;
}

bool EventLogTransmitter::PushLogSessionBegin(uint64_t processId) NN_NOEXCEPT
{
    std::aligned_storage<TransmitterBufferSizeForSessionInfo, TransmitterBufferAlign>::type buffer;

    detail::LogPacketTransmitter transmitter(
        &buffer, sizeof(buffer), m_FlushFunction, static_cast<uint8_t>(diag::LogSeverity_Info), 0, processId, true, true);

    transmitter.PushLogSessionBegin();

    auto isSuccess = transmitter.Flush(true);

    if (!isSuccess)
    {
        IncreaseLogPacketDropCount();
    }

    return isSuccess;
}

bool EventLogTransmitter::PushLogSessionEnd(uint64_t processId) NN_NOEXCEPT
{
    std::aligned_storage<TransmitterBufferSizeForSessionInfo, TransmitterBufferAlign>::type buffer;

    detail::LogPacketTransmitter transmitter(
        &buffer, sizeof(buffer), m_FlushFunction, static_cast<uint8_t>(diag::LogSeverity_Info), 0, processId, true, true);

    transmitter.PushLogSessionEnd();

    auto isSuccess = transmitter.Flush(true);
    if (!isSuccess)
    {
        IncreaseLogPacketDropCount();
    }

    return isSuccess;
}

bool EventLogTransmitter::PushLogPacketDropCountIfExists() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LogPacketDropCountMutex);

    if (m_LogPacketDropCount == 0u)
    {
        return true;
    }

    std::aligned_storage<TransmitterBufferSizeForDropCount, TransmitterBufferAlign>::type buffer;

    detail::LogPacketTransmitter transmitter(
        &buffer, sizeof(buffer), m_FlushFunction, static_cast<uint8_t>(diag::LogSeverity_Info), 0, 0, true, true);

    transmitter.PushLogPacketDropCount(m_LogPacketDropCount);

    auto isSuccess = transmitter.Flush(true);

    if (isSuccess)
    {
        NN_DETAIL_LM_WARN("Dropped %lld log packets.\n", m_LogPacketDropCount);
        m_LogPacketDropCount = 0;
    }
    else
    {
        m_LogPacketDropCount++;
    }

    return isSuccess;
}

void EventLogTransmitter::IncreaseLogPacketDropCount() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LogPacketDropCountMutex);

    m_LogPacketDropCount++;
}

}}} // nn::lm::impl
