﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <curl/curl.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_ApplicationId.private.h>

#include <nn/crypto/crypto_Aes128CbcDecryptor.h>
#include <nn/crypto/crypto_Sha256Generator.h>

#include <nn/ssl/ssl_Types.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/util/util_Uuid.h>
#include <nn/util/util_UuidApi.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/lidbe/lidbe_Api.h>
#include <nn/lidbe/lidbe_Result.h>
#include <nn/lidbe/lidbe_Result.private.h>

#ifdef _DEBUG

#define IDBE_PRINTF(...) NN_SDK_LOG(__VA_ARGS__)


#else

#define IDBE_PRINTF(...) (void)0

#endif

// URL のホスト名まで
#define IDBE_URL_HOST_WUP "https://idbe-wup.cdn.nintendo.net/"

#define IDBE_URL_HOST_CTR "https://idbe-ctr.cdn.nintendo.net/"

#define IDBE_URL_HOST_HAC "https://idbe-hac.cdn.nintendo.net/"

// URL パス
#define IDBE_URL_PATH        "icondata/%02X/%016llX-%d.idbe"
// URL パス(Latest)
#define IDBE_URL_PATH_LATEST "icondata/%02X/%016llX.idbe"
// URL 用バッファのサイズ
const int IdbeUrlSize = 128;

// AES の鍵長
const int IdbeAesKeyLength = 16;
// AES のブロック長
const int IdbeAesBlockLength = 16;
// AES の鍵数
const int IdbeNumAesKeys = 4;
// DEV 用 AES 鍵
const int IdbeAesKeyForDev = 0;

const int IdbeDownloadLowSpeedTimeOutSec = 120;


#define CURL_EASY_DO(exp) \
    do                                                                                   \
    {                                                                                    \
        errorCode = (exp);                                                               \
                                                                                         \
        if (errorCode != CURLE_OK)                                                       \
        {                                                                                \
           detail::SetLastResult(detail::HandleCurlError(errorCode));                    \
        }                                                                                \
    }                                                                                    \
    while (NN_STATIC_CONDITION(0))



namespace
{
    // Curl オブジェクトに設定するプラットフォーム ID 判定用のデータ
    // 先頭ポインタがこのデータと一致するかどうかでプラットフォームを判定するため、#define に置き換えてはならない。
    const char* IDBE_CURL_PRIVATE_DATA_WUP = "W";
    const char* IDBE_CURL_PRIVATE_DATA_CTR = "C";

    // 最後に発生したエラー
    nn::Result lastResult = nn::ResultSuccess();

    // SSL コンテキスト
    nn::ssl::Context sslContext;
    nn::Result sslResult;
    CURLcode errorCode;

    // キャンセルフラグ
    volatile bool isCanceled = false;

    //// IV と AES 鍵は WUP/CTR 共通

    // IV
    const uint8_t iv[IdbeAesBlockLength] =
    {
        0xa4, 0x69, 0x87, 0xae, 0x47, 0xd8, 0x2b, 0xb4, 0xfa, 0x8a, 0xbc, 0x04, 0x50, 0x28, 0x5f, 0xa4
    };

    // AES 鍵
    const uint8_t keys[IdbeNumAesKeys][IdbeAesKeyLength] =
    {
        // 開発用
        {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
        // 製品用 1
        {0x90, 0xa0, 0xbb, 0x1e, 0x0e, 0x86, 0x4a, 0xe8, 0x7d, 0x13, 0xa6, 0xa0, 0x3d, 0x28, 0xc9, 0xb8},
        // 製品用 2
        {0xff, 0xbb, 0x57, 0xc1, 0x4e, 0x98, 0xec, 0x69, 0x75, 0xb3, 0x84, 0xfc, 0xf4, 0x07, 0x86, 0xb5},
        // 製品用 3
        {0x80, 0x92, 0x37, 0x99, 0xb4, 0x1f, 0x36, 0xa6, 0xa7, 0x5f, 0xb8, 0xb4, 0x8c, 0x95, 0xf6, 0x6f}
    };

}


namespace nn { namespace lidbe {

namespace detail
{
    // プラットフォーム ID
    enum PlatformId
    {
        PLATFORM_ID_UNKNOWN = 0,
        PLATFORM_ID_WUP     = 1,
        PLATFORM_ID_CTR     = 2,
        PLATFORM_ID_NX      = 3,

        PLATFORM_ID_MAX
    };

    // プログラム ID からプラットフォーム ID を取得します。
    PlatformId GetPlatformId(uint64_t programId);
    // ダウンロードコンテキストからプラットフォーム ID を取得します。
    PlatformId GetPlatformId(DownloadContext* pContext);

    // アイコンファイルの URL を取得します。
    void GetIconUrl(char* pIconUrl, uint64_t programId, uint16_t version);
    // 最新バージョンのアイコンファイルの URL を取得します。
    void GetIconUrl(char* pIconUrl, uint64_t programId);

    // Curl の書き込み関数です。
    size_t WriteFunction(char* pBuffer, size_t size, size_t nmemb, void* pUserData);
    // Curl の進捗関数です。
    int ProgressFunction(void* pUserData, double totalDownloaded, double nowDownloaded, double totalUploaded, double nowUploaded);

    // ダウンロードコンテキストを作成します。
    bool CreateDownloadContext(DownloadContext* pContext, void* pFileBuffer, const char* pUrl, bool isKeepAlive, uint64_t programId);
    // アイコンファイルをダウンロードします。
    bool DownloadIconFile(DownloadContext* pContext);

    // ファイルヘッダをチェックします。
    bool CheckFileHeader(const uint8_t* pBuffer, PlatformId platformId);
    // アイコンファイルの復号化処理を行います。
    bool DecryptIconFile(void* pIconBuffer, const void* pFileBuffer, PlatformId platformId);

    // 最後に発生したエラーを初期化します。
    void ResetLastResult();
    // 最後に発生したエラーを設定します。
    void SetLastResult(nn::Result result);
    // CURLOPT_SSL_CTX_FUNCTIONに対するCallback関数。
    CURLcode SslCtxFunction(CURL* curl, void* ssl, void* param) NN_NOEXCEPT;
    //CURLのエラーコードを nn::Resultへ変換します。
    nn::Result HandleCurlError(CURLcode code) NN_NOEXCEPT;

}

detail::PlatformId detail::GetPlatformId(uint64_t programId)
{

    if (nn::IsCafe(programId))
    {
        return PLATFORM_ID_WUP;
    }
    if (nn::IsCtr(programId))
    {
        return PLATFORM_ID_CTR;
    }
    if (nn::IsNx(programId))
    {
        return PLATFORM_ID_NX;
    }

    return PLATFORM_ID_UNKNOWN;
}

detail::PlatformId detail::GetPlatformId(DownloadContext* pContext)
{
    char* pPrivateData = NULL;
    CURL_EASY_DO(curl_easy_getinfo(pContext->pCurl, CURLINFO_PRIVATE, &pPrivateData));

    // ポインタの一致判定で十分
    if (pPrivateData == IDBE_CURL_PRIVATE_DATA_WUP)
    {
        return PLATFORM_ID_WUP;
    }
    else if (pPrivateData == IDBE_CURL_PRIVATE_DATA_CTR)
    {
        return PLATFORM_ID_CTR;
    }

    return PLATFORM_ID_UNKNOWN;
}

void detail::GetIconUrl(char* pIconUrl, uint64_t programId, uint16_t version)
{
    uint8_t dirNumber = ((programId & 0xFF00) >> 8);

    switch (GetPlatformId(programId))
    {
    case PLATFORM_ID_WUP:
        {
            util::SNPrintf(pIconUrl, IdbeUrlSize, IDBE_URL_HOST_HAC IDBE_URL_PATH, dirNumber, programId, version);
        }
        break;
    case PLATFORM_ID_CTR:
        {
            util::SNPrintf(pIconUrl, IdbeUrlSize, IDBE_URL_HOST_HAC IDBE_URL_PATH, dirNumber, programId, version);
        }
        break;
    default:
        {
            pIconUrl[0] = '\0';
        }
    }
}

void detail::GetIconUrl(char* pIconUrl, uint64_t programId)
{
    uint8_t dirNumber = ((programId & 0xFF00) >> 8);

    switch (GetPlatformId(programId))
    {
    case PLATFORM_ID_WUP:
        {
            util::SNPrintf(pIconUrl, IdbeUrlSize, IDBE_URL_HOST_HAC IDBE_URL_PATH_LATEST, dirNumber, programId);
        }
        break;
    case PLATFORM_ID_CTR:
        {
            util::SNPrintf(pIconUrl, IdbeUrlSize, IDBE_URL_HOST_HAC IDBE_URL_PATH_LATEST, dirNumber, programId);
        }
        break;
    default:
        {
            pIconUrl[0] = '\0';
        }
    }
}


size_t detail::WriteFunction(char* pBuffer, size_t size, size_t nmemb, void* pUserData)
{
    size_t readSize = size * nmemb;

    DownloadContext* pContext = static_cast<DownloadContext*>(pUserData);

    size_t maxSize = IDBE_FILE_SIZE;

    switch (GetPlatformId(pContext))
    {
    case PLATFORM_ID_WUP:
        {
            maxSize = IDBE_FILE_SIZE_WUP;
        }
        break;
    case PLATFORM_ID_CTR:
        {
            maxSize = IDBE_FILE_SIZE_CTR;
        }
        break;
    default:
        break;
    }

    if (pContext->readSize + readSize > maxSize)
    {
        return 0;
    }

    memcpy(static_cast<uint8_t*>(pContext->pBuffer) + pContext->readSize, pBuffer, readSize);
    pContext->readSize += readSize;

    return readSize;
}

int detail::ProgressFunction(void* pUserData, double totalDownloaded, double nowDownloaded, double totalUploaded, double nowUploaded)
{
    if (isCanceled)
    {
        return 1;
    }

    return 0;
}

bool detail::CreateDownloadContext(DownloadContext* pContext, void* pFileBuffer, const char* pUrl, bool isKeepAlive, uint64_t programId)
{
    detail::ResetLastResult();

    if (!pContext)
    {
        IDBE_PRINTF("[IDBE] CreateDownloadContext() failed. (pContext == NULL)\n");
        return false;
    }

    pContext->pCurl = curl_easy_init();
    if (!pContext->pCurl)
    {
        IDBE_PRINTF("[IDBE] CreateDownloadContext() failed. (pCurl == NULL)\n");
        detail::SetLastResult(detail::HandleCurlError(errorCode));                    \
        return false;
    }

    pContext->pBuffer  = pFileBuffer;
    pContext->readSize = 0;

    IDBE_PRINTF("[IDBE] URL = %s\n", pUrl);

    CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_URL, pUrl));


    switch (GetPlatformId(programId))
    {
    case PLATFORM_ID_WUP:
        {
            CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_PRIVATE, IDBE_CURL_PRIVATE_DATA_WUP));
            CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_MAXFILESIZE, IDBE_FILE_SIZE_WUP));
        }
        break;
    case PLATFORM_ID_CTR:
        {
            CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_PRIVATE, IDBE_CURL_PRIVATE_DATA_CTR));
            CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_MAXFILESIZE, IDBE_FILE_SIZE_CTR));
        }
        break;
    default:
        break;
    }


    CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_WRITEDATA, pContext));
    CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_WRITEFUNCTION, detail::WriteFunction));

    // 通信の無い期間が IDBE_DOWNLOAD_LOW_SPEED_TIME_OUT_SEC 秒続いたらタイムアウトとする。
    // ただし CURLOPT_LOW_SPEED_LIMIT は 0 には設定できない(無効となってしまう)ため、最小値の 1 を設定する。
    // これにより「通信速度が 1B/s の期間が IDBE_DOWNLOAD_LOW_SPEED_TIME_OUT_SEC 秒続いたらタイムアウト」となる。
    CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_LOW_SPEED_LIMIT, 1));
    CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_LOW_SPEED_TIME, IdbeDownloadLowSpeedTimeOutSec));

    if (isKeepAlive)
    {
        pContext->pHeader = curl_slist_append(NULL, "Connection: Keep-Alive");

        CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_HTTPHEADER, pContext->pHeader));
    }
    else
    {
        pContext->pHeader = NULL;
    }

    CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_SSL_VERIFYDATE, 0));

    if (errorCode != CURLE_OK)
    {
        IDBE_PRINTF("[IDBE] curl_easy_setopt(pContext->pCurl, CURLOPT_SSL_VERIFYDATE, 0) failed. (errorCode = %d)\n", errorCode);
    }

    CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_SSL_CTX_FUNCTION, SslCtxFunction));

    if (errorCode != CURLE_OK)
    {
        IDBE_PRINTF("[IDBE] curl_easy_setopt(CURLOPT_SSL_CTX_DATA) failed. (errorCode = %d)\n", errorCode);
        DestroyDownloadContext(pContext);
    }

    return (errorCode == CURLE_OK);
}

bool detail::DownloadIconFile(DownloadContext* pContext)
{
    detail::ResetLastResult();

    isCanceled = false;

    // DownloadIconFile を呼び出す時のみ進捗コールバックを設定する
    CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_NOPROGRESS, 0));
    CURL_EASY_DO(curl_easy_setopt(pContext->pCurl, CURLOPT_PROGRESSFUNCTION, detail::ProgressFunction));

    bool isSucceeded = true;

    CURL_EASY_DO(curl_easy_perform(pContext->pCurl));

    if (errorCode != CURLE_OK)
    {
        IDBE_PRINTF("[IDBE] DownloadIconFile() failed. (errorCode = %d)\n", errorCode);
        isSucceeded = false;
    }

    if (isSucceeded)
    {
        isSucceeded = CheckDownloadResult(pContext);
    }

    DestroyDownloadContext(pContext);

    return isSucceeded;
}

bool detail::CheckFileHeader(const uint8_t* pBuffer, PlatformId platformId)
{
    switch (platformId)
    {
    case PLATFORM_ID_WUP:
        {
            // フォーマットバージョンのチェック
            if (pBuffer[0] > IdbeSupportFormatVersionWup)
            {
                IDBE_PRINTF("[IDBE] CheckFileHeader() failed. (formatVersion = %d)\n", pBuffer[0]);
                return false;
            }

            // 鍵番号ののチェック
            if (pBuffer[1] >= IdbeNumAesKeys)
            {
                IDBE_PRINTF("[IDBE] CheckFileHeader() failed. (keyNumber = %d)\n", pBuffer[1]);
                return false;
            }

            // 量産実機で開発用鍵は使えない
            if (pBuffer[1] == IdbeAesKeyForDev)
            {
                IDBE_PRINTF("[IDBE] CheckFileHeader() failed. (don't use dev key)\n");
                return false;
            }
        }
        break;
    case PLATFORM_ID_CTR:
        {
            // フォーマットバージョンのチェック
            if (pBuffer[0] > IdbeSupportFormatVersionCtr)
            {
                IDBE_PRINTF("[IDBE] CheckFileHeader() failed. (formatVersion = %d)\n", pBuffer[0]);
                return false;
            }

            // 鍵番号ののチェック
            if (pBuffer[1] >= IdbeNumAesKeys)
            {
                IDBE_PRINTF("[IDBE] CheckFileHeader() failed. (keyNumber = %d)\n", pBuffer[1]);
                return false;
            }

            // 量産実機で開発用鍵は使えない
            if (pBuffer[1] == IdbeAesKeyForDev)
            {
                IDBE_PRINTF("[IDBE] CheckFileHeader() failed. (don't use dev key)\n");
                return false;
            }
        }
        break;
    default:
        return false;
    }

    return true;
}

bool detail::DecryptIconFile(void* pIconBuffer, const void* pFileBuffer, PlatformId platformId)
{
    if (!pIconBuffer || !pFileBuffer)
    {
        return false;
    }

    const uint8_t* pFileBufferU8 = static_cast<const uint8_t*>(pFileBuffer);

    if (!detail::CheckFileHeader(pFileBufferU8, platformId))
    {
        return false;
    }

    switch (platformId)
    {
    case PLATFORM_ID_WUP:
        {
            nn::Bit8 hash[nn::crypto::Sha256Generator::HashSize + sizeof(nn::Bit64)] = {0};
            {
                nn::crypto::Aes128CbcDecryptor context;
                context.Initialize(keys[pFileBufferU8[1]], IdbeAesKeyLength, iv, IdbeAesBlockLength);

                pFileBufferU8 += IdbeFileHeaderSizeWup;
                context.Update(hash, IdbeHashSizeWup, pFileBufferU8, IdbeHashSizeWup);

                pFileBufferU8 += IdbeHashSizeWup;
                context.Update(pIconBuffer, IdbeSizeWup, pFileBufferU8, IdbeSizeWup);
            }

            nn::Bit8 dataHash[nn::crypto::Sha256Generator::HashSize + sizeof(nn::Bit64)] = {0};
            {
                nn::crypto::Sha256Generator generator;

                generator.Initialize();
                generator.Update(pIconBuffer, IdbeSizeWup);
                generator.GetHash(dataHash, nn::crypto::Sha256Generator::HashSize);
            }

            if (memcmp(hash, dataHash, IdbeHashSizeWup) != 0)
            {
                IDBE_PRINTF("[IDBE] Hash mismatch.\n");
                return false;
            }
        }
        break;
    case PLATFORM_ID_CTR:
        {
            nn::Bit8 hash[nn::crypto::Sha256Generator::HashSize + sizeof(nn::Bit64)] = {0};
            {
                nn::crypto::Aes128CbcDecryptor context;
                context.Initialize(keys[pFileBufferU8[1]], IdbeAesKeyLength, iv, IdbeAesBlockLength);

                pFileBufferU8 += IdbeFileHeaderSizeCtr;
                context.Update(hash, IdbeHashSizeCtr, pFileBufferU8, IdbeHashSizeCtr);

                pFileBufferU8 += IdbeHashSizeCtr;
                context.Update(pIconBuffer, IdbeSizeCtr, pFileBufferU8, IdbeSizeCtr);
            }

            nn::Bit8 dataHash[nn::crypto::Sha256Generator::HashSize + sizeof(nn::Bit64)] = {0};
            {
                nn::crypto::Sha256Generator generator;

                generator.Initialize();
                generator.Update(pIconBuffer, IdbeSizeCtr);
                generator.GetHash(dataHash, nn::crypto::Sha256Generator::HashSize);
            }

            if (memcmp(hash, dataHash, IdbeHashSizeCtr) != 0)
            {
                IDBE_PRINTF("[IDBE] Hash mismatch.\n");
                return false;
            }
        }
        break;
    default:
        return false;
    }

    return true;
}

void detail::ResetLastResult()
{
    lastResult = nn::ResultSuccess();
}

void detail::SetLastResult(nn::Result result)
{
    lastResult = result;
}

bool Initialize()
{
    detail::ResetLastResult();

    sslResult = sslContext.Create(nn::ssl::Context::SslVersion_Auto);
    if (sslResult.IsFailure())
    {
        IDBE_PRINTF("[IDBE] NSSLCreateContext() failed.\n");
        return false;
    }

    return true;
}

CURLcode detail::SslCtxFunction(CURL* curl, void* ssl, void* param) NN_NOEXCEPT
{
    NN_UNUSED(curl);
    NN_UNUSED(ssl);
    NN_UNUSED(param);

    nn::ssl::Context* context = reinterpret_cast<nn::ssl::Context*>(ssl);

    if (context->Create(nn::ssl::Context::SslVersion_Auto).IsFailure())
    {
        return CURLE_ABORTED_BY_CALLBACK;
    }

    return CURLE_OK;

}

void Finalize()
{
    if (sslResult.IsSuccess())
    {
        sslContext.Destroy();
    }
}

bool DownloadIconFile(void* pFileBuffer, uint64_t programId, uint16_t version, bool isKeepAlive)
{
    DownloadContext context;

    if (CreateIconDownloadContext(&context, pFileBuffer, programId, version, isKeepAlive))
    {
        return detail::DownloadIconFile(&context);
    }

    return false;
}

bool DownloadLatestIconFile(void* pFileBuffer, uint64_t programId, bool isKeepAlive)
{
    DownloadContext context;

    if (CreateLatestIconDownloadContext(&context, pFileBuffer, programId, isKeepAlive))
    {
        return detail::DownloadIconFile(&context);
    }

    return false;
}

void Cancel()
{
    isCanceled = true;
}

bool CreateIconDownloadContext(DownloadContext* pContext, void* pFileBuffer, uint64_t programId, uint16_t version, bool isKeepAlive)
{
    char iconUrl[IdbeUrlSize] = {0};

    detail::GetIconUrl(iconUrl, programId, version);

    return detail::CreateDownloadContext(pContext, pFileBuffer, iconUrl, isKeepAlive, programId);
}

bool CreateLatestIconDownloadContext(DownloadContext* pContext, void* pFileBuffer, uint64_t programId, bool isKeepAlive)
{
    char iconUrl[IdbeUrlSize] = {0};

    detail::GetIconUrl(iconUrl, programId);

    return detail::CreateDownloadContext(pContext, pFileBuffer, iconUrl, isKeepAlive, programId);
}

void DestroyDownloadContext(DownloadContext* pContext)
{
    if (!pContext)
    {
        return;
    }

    if (pContext->pHeader)
    {
        curl_slist_free_all(pContext->pHeader);
        pContext->pHeader = NULL;
    }

    if (pContext->pCurl)
    {
        curl_easy_cleanup(pContext->pCurl);
        pContext->pCurl = NULL;
    }
}

bool CheckDownloadResult(DownloadContext* pContext)
{
    detail::ResetLastResult();

    if (!pContext)
    {
        return false;
    }

    int64_t statusCode = 0;

    CURL_EASY_DO(curl_easy_getinfo(pContext->pCurl, CURLINFO_RESPONSE_CODE, &statusCode));

    if (errorCode != CURLE_OK)
    {
        IDBE_PRINTF("[IDBE] CheckDownloadResult() returned false. (errorCode = %d)\n", errorCode);
        return false;
    }

    if (statusCode != 200)
    {
        IDBE_PRINTF("[IDBE] CheckDownloadResult() returned false. (statusCode = %d)\n", statusCode);
        return false;
    }

    size_t expectedSize = IDBE_FILE_SIZE;

    switch (detail::GetPlatformId(pContext))
    {
    case detail::PLATFORM_ID_WUP:
        {
            expectedSize = IDBE_FILE_SIZE_WUP;
        }
        break;
    case detail::PLATFORM_ID_CTR:
        {
            expectedSize = IDBE_FILE_SIZE_CTR;
        }
        break;
    default:
        {
            IDBE_PRINTF("[IDBE] CheckDownloadResult() returned false. (invalid private data)\n");
            return false;
        }
    }

    if (pContext->readSize != expectedSize)
    {
        IDBE_PRINTF("[IDBE] CheckDownloadResult() returned false. (totalSize = %d, expectedSize = %d)\n", pContext->readSize, expectedSize);
        return false;
    }

    return true;
}

bool DecryptIconFile(void* pIconBuffer, const void* pFileBuffer, uint64_t programId)
{
    return detail::DecryptIconFile(pIconBuffer, pFileBuffer, detail::GetPlatformId(programId));
}

namespace detail
{
    nn::Result GetLastResult()
    {
        return lastResult;
    }
}
}}
