﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include "detail/lbl_BrightnessController.h"
#include "detail/lbl_DeviceAccessorAls.h"
#include "detail/lbl_SettingsHolder.h"
#include <nn/os.h>
#include <nn/lbl/detail/lbl_Log.h>
#include <nn/lbl/impl/lbl.h>
#include <nn/lbl/impl/lbl_AutoControlServer.h>
#include <nn/lbl/impl/lbl_SettingsForVrMode.h>
#include <nn/lbl/impl/lbl_Suspend.h>
#include <nn/lbl/impl/lbl_VrMode.h>

namespace {

const int InitializeCountMax = 16;          // 初期化回数の上限値(カウントのオーバーフロー対策)
int g_InitializeCount;                      // 初期化回数のカウント

nn::lbl::impl::detail::SettingsHolder g_SettingsHolder;                                 // 本体設定類のホルダー
nn::lbl::impl::detail::DeviceAccessorAls g_DeviceAccessorAls(&g_SettingsHolder);        // 照度センサアクセサ
nn::lbl::impl::detail::BrightnessController g_BrightnessController(&g_SettingsHolder);  // 輝度のコントローラ

bool g_IsSuspended = false;

} // namespace

namespace nn {
namespace lbl {
namespace impl {

void Initialize() NN_NOEXCEPT
{
    if (g_InitializeCount < InitializeCountMax)
    {
        ++g_InitializeCount;
        if (1 == g_InitializeCount)
        {
            // ファームウェアデバッグ設定をロード
            g_SettingsHolder.LoadFirmwareDebugSettings();

            // 本体設定をロード
            g_SettingsHolder.LoadSystemSettings();

            g_DeviceAccessorAls.Initialize();
            g_BrightnessController.Initialize();
        }
    }
    // 初期化回数でASSERTには引っ掛けない。
}

void Finalize() NN_NOEXCEPT
{
    if (0 < g_InitializeCount)
    {
        --g_InitializeCount;
        if (0 == g_InitializeCount)
        {
            g_BrightnessController.Finalize();
            g_DeviceAccessorAls.Finalize();
        }
    }
    // 初期化回数でASSERTには引っ掛けない。
}

void SaveCurrentSetting() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    g_SettingsHolder.SaveSystemSettings();
}

void LoadCurrentSetting() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    g_SettingsHolder.LoadSystemSettings();
}

void SetCurrentBrightnessSetting(float brightnessSetting) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    NN_SDK_REQUIRES_LESS_EQUAL(0.0f, brightnessSetting);
    g_SettingsHolder.SetBrightness(brightnessSetting);
    g_BrightnessController.ApplyImmediatelyNext();
}

float GetCurrentBrightnessSetting() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    return g_SettingsHolder.GetBrightness();
}

void SetCurrentBrightnessSettingForVrMode(float brightnessSetting) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    NN_SDK_REQUIRES_LESS_EQUAL(0.0f, brightnessSetting);
    g_SettingsHolder.SetBrightnessForVrMode(brightnessSetting);
}

float GetCurrentBrightnessSettingForVrMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    return g_SettingsHolder.GetBrightnessForVrMode();
}

void ApplyCurrentBrightnessSettingToBacklight() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    g_BrightnessController.UpdateAndApplyBrightness(g_DeviceAccessorAls.GetCurrentIlluminance(nullptr));
}

float GetBrightnessSettingAppliedToBacklight() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    return g_BrightnessController.GetAppliedBrightness();
}

void SwitchBacklightOn(nn::TimeSpan fadeInTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    NN_SDK_REQUIRES_LESS_EQUAL(0LL, fadeInTime.GetNanoSeconds());
    g_BrightnessController.StartFade(true, fadeInTime);
}

void SwitchBacklightOff(nn::TimeSpan fadeOutTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    NN_SDK_REQUIRES_LESS_EQUAL(0LL, fadeOutTime.GetNanoSeconds());
    g_BrightnessController.StartFade(false, fadeOutTime);
}

BacklightSwitchStatus GetBacklightSwitchStatus() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    return g_BrightnessController.GetFadeState();
}

void EnableDimming() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    g_BrightnessController.EnableDimming(true);
}

void DisableDimming() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    g_BrightnessController.EnableDimming(false);
}

bool IsDimmingEnabled() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    return g_BrightnessController.IsDimmingEnabled();
}

void EnableVrMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    g_BrightnessController.EnableVrMode(true);
}

void DisableVrMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    g_BrightnessController.EnableVrMode(false);
    g_BrightnessController.ApplyImmediatelyNext();
}

bool IsVrModeEnabled() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    return g_BrightnessController.IsVrModeEnabled();
}

void EnableAutoBrightnessControl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    g_SettingsHolder.EnableAutoBrightness(true);
    g_BrightnessController.ApplyImmediatelyNext();
}

void DisableAutoBrightnessControl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    g_SettingsHolder.EnableAutoBrightness(false);
}

bool IsAutoBrightnessControlEnabled() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    return g_SettingsHolder.IsAutoBrightnessEnabled();
}

void SetAmbientLightSensorValue(float sensorValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    NN_SDK_REQUIRES_LESS_EQUAL(0.0f, sensorValue);
    g_DeviceAccessorAls.SetCurrentIlluminanceForDebug(sensorValue);
}

float GetAmbientLightSensorValue(bool* pOutOverflowAlert) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    return g_DeviceAccessorAls.GetCurrentIlluminance(pOutOverflowAlert);
}

bool IsAmbientLightSensorAvailable() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);

    return g_DeviceAccessorAls.IsDeviceAccessible();
}

void UpdateBrightness() NN_NOEXCEPT
{
    static const int AlsUpdateRate = 2; // 30rps
    static int s_AlsUpdateCount = AlsUpdateRate;
    if ( g_InitializeCount && !g_IsSuspended )
    {
        if ( g_DeviceAccessorAls.IsDeviceAccessible() && ++s_AlsUpdateCount >= AlsUpdateRate )
        {
            s_AlsUpdateCount = 0;
            g_DeviceAccessorAls.UpdateIlluminance();
        }
        g_BrightnessController.UpdateAndApplyBrightness(g_DeviceAccessorAls.GetCurrentIlluminance(nullptr));
    }
}

void Suspend() NN_NOEXCEPT
{
    if(!g_IsSuspended)
    {
        g_IsSuspended = true;

        g_DeviceAccessorAls.Finalize();
    }
}

void Resume() NN_NOEXCEPT
{
    if(g_IsSuspended)
    {
        g_IsSuspended = false;

        g_DeviceAccessorAls.Initialize();
    }
}

} // impl
} // lbl
} // nn
