﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   認証アプレット起動に関する非公開 API の定義
 */

#include <nn/la/la_Auth.h>
#include <nn/la/la_Api.h>
#include <nn/la/la_AuthAppletArg.h>

#include <nn/applet/applet.h>
#include <nn/result/result_HandlingUtility.h>

//---------------------------------------------------------------------------
// C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace la { namespace auth {

namespace {
static const uint16_t cMajorVersion = 0;
static const uint16_t cMinorVersion = 2;

//---------------------------------------------------------------------------
//
static nn::Result StartLibraryAppletImpl( const AppletArg& in_param ) NN_NOEXCEPT
{
    // LA 共通パラメータを設定する
    nn::la::CommonArgumentsWriter comonArg( cMajorVersion, cMinorVersion );

    ReturnArg out_param;
    size_t    out_size;

    // LA起動自体に失敗してたらそのエラーを返す
    NN_RESULT_DO(
        nn::la::StartLibraryAppletEasy(
            nn::applet::AppletId_LibraryAppletAuth,
            comonArg,
            &in_param, sizeof( in_param ),
            &out_param, sizeof( out_param ), &out_size,
            false
        )
    );

    if( out_size == 0 )
    {
        return nn::la::ResultLibraryAppletCanceled();
    }

    return out_param.result;
}
} // anonymous namespace

//---------------------------------------------------------------------------
//
nn::Result ShowParentalAuthentication( bool isTemporary ) NN_NOEXCEPT
{
    ParentalAuthenticationArg arg;
    arg.SetDefault();
    arg.isTemporary = isTemporary;

    return ShowParentalAuthentication( arg );
}

//---------------------------------------------------------------------------
//
nn::Result ShowParentalAuthentication( const ParentalAuthenticationArg& arg ) NN_NOEXCEPT
{
    AppletArg in_param;
    in_param.userMode        = AppletArg::UserMode::UserMode_Parent;
    in_param.bootMode        = AppletArg::BootMode::BootMode_Authenticate;
    in_param.isTemporary     = arg.isTemporary;
    in_param.backgroundMode  = arg.backgroundMode;
    in_param.skipEnterRestrictedSystemSettings = arg.skipEnterRestrictedSystemSettings;

    return StartLibraryAppletImpl( in_param );
}

//---------------------------------------------------------------------------
//
nn::Result RegisterParentalPasscode() NN_NOEXCEPT
{
    AppletArg in_param;
    in_param.userMode = AppletArg::UserMode::UserMode_Parent;
    in_param.bootMode = AppletArg::BootMode::BootMode_Register;

    return StartLibraryAppletImpl( in_param );
}

//---------------------------------------------------------------------------
//
nn::Result ChangeParentalPasscode() NN_NOEXCEPT
{
    AppletArg in_param;
    in_param.userMode = AppletArg::UserMode::UserMode_Parent;
    in_param.bootMode = AppletArg::BootMode::BootMode_Change;

    return StartLibraryAppletImpl( in_param );
}

//---------------------------------------------------------------------------
//
Result GetResultFromContextStack( ReturnArg* pOut, nn::applet::StorageHandle handle ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOut );
    NN_RESULT_DO( nn::applet::ReadFromStorage( handle, 0, pOut, sizeof( *pOut ) ) );
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::la::auth
