﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "jit_PluginFunctions.h"

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <cstring>
#include <nn/diag/detail/diag_DetailAssert.h>

namespace {

void (*g_Panic)();

void* (*g_Memcpy)(void*, const void*, size_t);
void* (*g_Memmove)(void*, const void*, size_t);
void* (*g_Memset)(void*, int, size_t);
void (*g_AbortImpl)(const char* condition, const char* functionName, const char* fileName, int lineNumber);
void (*g_UnexpectedDefaultImpl)(const char* functionName, const char* fileName, int lineNumber);

}

namespace nn { namespace jit { namespace plugin {

void SetupPluginFunctions(uintptr_t (*lookupSymbol)(const char*)) NN_NOEXCEPT
{
    #define SET_SYMBOL(v, name) v = reinterpret_cast<decltype(v)>(lookupSymbol(name));

    SET_SYMBOL(g_Panic, "PanicForPlugin");

    SET_SYMBOL(g_Memcpy, "memcpy");
    SET_SYMBOL(g_Memmove, "memmove");
    SET_SYMBOL(g_Memset, "memset");

    SET_SYMBOL(g_AbortImpl, "_ZN2nn4diag6detail9AbortImplEPKcS3_S3_i");
    SET_SYMBOL(g_UnexpectedDefaultImpl, "_ZN2nn6detail21UnexpectedDefaultImplEPKcS2_i");

    #undef SET_SYMBOL
}

NN_NORETURN void Panic() NN_NOEXCEPT
{
    g_Panic();
    for (;;)
    {
    }
}

}}}

extern "C" {

void* memcpy(void* dst, const void* src, size_t size)
{
    return g_Memcpy(dst, src, size);
}

void* memmove(void* dst, const void* src, size_t size)
{
    return g_Memmove(dst, src, size);
}

void* memset(void* dst, int value, size_t size)
{
    return g_Memset(dst, value, size);
}

} // extern "C"

namespace nn { namespace detail {

NN_NORETURN void UnexpectedDefaultImpl(
    const char* functionName,
    const char* fileName,
    int lineNumber) NN_NOEXCEPT
{
    if (g_UnexpectedDefaultImpl)
    {
        g_UnexpectedDefaultImpl(functionName, fileName, lineNumber);
    }
    nn::jit::plugin::Panic();
}

}}

namespace nn { namespace diag { namespace detail {

NN_NORETURN void AbortImpl(const char* condition, const char* functionName, const char* fileName, int lineNumber) NN_NOEXCEPT
{
    NN_SDK_LOG("Abort in plugin: %s(%d)\n", fileName, lineNumber);
    g_AbortImpl(condition, functionName, fileName, lineNumber);
    nn::jit::plugin::Panic();
}

NN_NORETURN void AbortImpl(const char* condition, const char* functionName, const char* fileName, int lineNumber, const char* format, ...) NN_NOEXCEPT
{
    NN_UNUSED(format);
    AbortImpl(condition, functionName, fileName, lineNumber);
}

}}}
