﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/jit/plugin/jit_Plugin.h>
#include <nn/jit/plugin/jit_DllInterface.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>

#include "jit_Globals.h"
#include "jit_PluginFunctions.h"
#include "jit_PluginDiagnostics.h"

namespace nn { namespace jit { namespace plugin {

void ConfigurePlugin() NN_NOEXCEPT __attribute__((weak));
void OnPrepared(const nn::jit::plugin::JitPluginEnvironment&) NN_NOEXCEPT __attribute__((weak));
int Control(const nn::jit::plugin::JitPluginEnvironment& info, uint64_t tag, const void* inData, size_t inDataSize, void* outBuffer, size_t outBufferSize) NN_NOEXCEPT __attribute__((weak));

namespace {

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    void SetupDiagnosticsImpl(const DiagnosticsParameters* p) NN_NOEXCEPT
    {
        nn::jit::plugin::SetupPluginDiagnostics(p->lookupSymbol);
    }
#else
    void SetupDiagnosticsImpl(const DiagnosticsParameters* p) NN_NOEXCEPT
    {
        NN_UNUSED(p);
        nn::jit::plugin::SetupPluginDiagnostics(p->lookupSymbol);
    }
#endif

}

nn::jit::plugin::ConfigureParameters* g_pConfigureParameters;

}}}

NN_JIT_PLUGIN_FUNCTION(GetVersion, uint32_t, ())
{
    return nn::jit::plugin::PluginVersion;
}

NN_JIT_PLUGIN_FUNCTION(ResolveBasicSymbols, void, (uintptr_t (*lookupSymbol)(const char* name)))
{
    nn::jit::plugin::SetupPluginFunctions(lookupSymbol);
}

NN_JIT_PLUGIN_FUNCTION(SetupDiagnostics, void, (uint32_t version, const nn::jit::plugin::DiagnosticsParameters* p))
{
    if (version == nn::jit::plugin::DiagnosticsVersion)
    {
        nn::jit::plugin::SetupDiagnosticsImpl(p);
    }
}

NN_JIT_PLUGIN_FUNCTION(Configure, void, (nn::jit::plugin::ConfigureParameters* p))
{
    if (&nn::jit::plugin::ConfigurePlugin)
    {
        nn::jit::plugin::g_pConfigureParameters = p;
        nn::jit::plugin::ConfigurePlugin();
        nn::jit::plugin::g_pConfigureParameters = nullptr;
    }
}

NN_JIT_PLUGIN_FUNCTION(OnPrepared, void, (const nn::jit::plugin::JitPluginEnvironment& info))
{
    if (&nn::jit::plugin::OnPrepared)
    {
        nn::jit::plugin::OnPrepared(info);
    }
}

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    #define NN_JIT_PLUGIN_CHECK_POST_CONDITION(condition) \
        if (!(condition)) \
        { \
            NN_SDK_LOG("[jit error] plugin violates post condition: %s\n", #condition); \
            return 1; \
        }
#else
    #define NN_JIT_PLUGIN_CHECK_POST_CONDITION(condition) static_cast<void>(0)
#endif

NN_JIT_PLUGIN_FUNCTION(GenerateCode, int, (int* pOut, nn::jit::CodeRange* pGeneratedRx, nn::jit::CodeRange* pGeneratedRo, const nn::jit::plugin::JitPluginEnvironment& environment, uint64_t tag, const void* source, size_t sourceSize, const nn::jit::CodeRange& rxBuffer, const nn::jit::CodeRange& roBuffer, const void* inData, size_t inDataSize, void* outBuffer, size_t outBufferSize))
{
    auto ret = nn::jit::plugin::GenerateCode(pGeneratedRx, pGeneratedRo, environment, tag, source, sourceSize, rxBuffer, roBuffer, inData, inDataSize, outBuffer, outBufferSize);
    const auto& environmentInfo = environment.environmentInfo;
    NN_UNUSED(environmentInfo);
    NN_JIT_PLUGIN_CHECK_POST_CONDITION(pGeneratedRx->IsValid(environmentInfo.rxCodeSize));
    NN_JIT_PLUGIN_CHECK_POST_CONDITION(pGeneratedRo->IsValid(environmentInfo.roCodeSize));
    NN_JIT_PLUGIN_CHECK_POST_CONDITION(pGeneratedRx->IsInRange(rxBuffer));
    NN_JIT_PLUGIN_CHECK_POST_CONDITION(pGeneratedRo->IsInRange(roBuffer));
    *pOut = ret;
    return 0;
}

NN_JIT_PLUGIN_FUNCTION(Control, int, (int* pOut, const nn::jit::plugin::JitPluginEnvironment& info, uint64_t tag, const void* inData, size_t inDataSize, void* outBuffer, size_t outBufferSize))
{
    if (!&nn::jit::plugin::Control)
    {
        return 1;
    }
    auto ret = nn::jit::plugin::Control(info, tag, inData, inDataSize, outBuffer, outBufferSize);
    *pOut = ret;
    return 0;
}

NN_JIT_PLUGIN_FUNCTION(Keeper, uintptr_t, ())
{
    return 0;
}

extern "C" {

int __nnmusl_init_dso(unsigned char*, unsigned char*,
                      unsigned char*, unsigned char*,
                      unsigned char*, unsigned char*,
                      unsigned char*, unsigned char*,
                      unsigned char*, unsigned char*,
                      unsigned char*, unsigned char*,
                      unsigned char*, unsigned char*,
                      unsigned char*, unsigned char*,
                      unsigned char*, unsigned char*,
                      unsigned char*, unsigned char*,
                      unsigned char*)
{
    return 2;
}

void __nnmusl_fini_dso(unsigned char*, unsigned char*,
                       unsigned char*, unsigned char*,
                       unsigned char*, unsigned char*)
{
}

void __cxa_guard_acquire()
{
}

void __cxa_guard_release()
{
}

}
