﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/idle/idle_SystemTypes.h>
#include "idle_PolicyParam.h"

namespace nn { namespace idle { namespace server {

    class IHandler
    {
    public:
        IHandler() NN_NOEXCEPT :
            m_IsEnabled(false),
            m_IsTriggered(false),
            m_TimeToTrigger(0)
        {}
        virtual ~IHandler() NN_NOEXCEPT = 0;

        void InitializeBase() NN_NOEXCEPT
        {
            // 起動時に一度だけ呼ばれる
            m_IsEnabled = false;
            m_IsTriggered = false;
            m_TimeToTrigger = 0;
        };
        void Enable() NN_NOEXCEPT { m_IsEnabled = true; }
        void Disable() NN_NOEXCEPT { m_IsEnabled = false; }
        bool IsEnabled() const NN_NOEXCEPT { return m_IsEnabled; }
        bool IsTriggered() const NN_NOEXCEPT { return m_IsTriggered; }

        void Reset() NN_NOEXCEPT
        {
            NotifyActivity();
        }
        void UpdatePolicy(const PolicyParam& policyParam) NN_NOEXCEPT
        {
            m_TimeToTrigger = GetTimeToTrigger(policyParam);
        }
        void NotifyActivity() NN_NOEXCEPT
        {
            if ( m_IsEnabled && m_IsTriggered )
            {
                // 無操作ハンドラ発動状態の解除
                m_IsTriggered = false;
                HandleActiveState();
            }
        }
        void NotifyInactivity(nn::TimeSpan elapsedIdleTime) NN_NOEXCEPT
        {
            if ( m_IsEnabled && !m_IsTriggered && 0 < m_TimeToTrigger && m_TimeToTrigger <= elapsedIdleTime )
            {
                // 無操作ハンドラ発動
                m_IsTriggered = true;
                HandleIdleState();
            }
        }

        // 派生クラスでオーバーライド可能なデフォルト実装
        virtual void Initialize() NN_NOEXCEPT
        {
            // 起動時に一度だけ呼ばれる
        };

        // 派生クラスで実装必須
        virtual void HandleActiveState() NN_NOEXCEPT = 0;
        virtual void HandleIdleState() NN_NOEXCEPT = 0;
        virtual nn::TimeSpan GetTimeToTrigger(const PolicyParam& policyParam) NN_NOEXCEPT = 0;

    private:
        bool                m_IsEnabled;        //!< グローバルな有効・無効状態を示します。マネージャによりコントロールされます。
        bool                m_IsTriggered;      //!< 無操作が発動したかどうかのフラグです。
        nn::TimeSpan        m_TimeToTrigger;    //!< 無操作状態ハンドラ発動までの時間です。設定により一時無効化する場合は 0 を格納します。
    };

}}} // namespace nn::idle::server

