﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_Types.h>                  // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h>

#include <nn/i2c/i2c.h>
#include <nn/i2c/i2c_ServiceType.h>

#include <nn/i2c/i2c_IManager.sfdl.h>
#include <nn/i2c/i2c_ISession.sfdl.h>

#include "i2c_GetManagerByHipc.h"

namespace nn { namespace i2c {

namespace {

inline ISession* GetInterface(const I2cSession& session) NN_NOEXCEPT
{
    return static_cast<nn::i2c::ISession*>(session._handle);
}

}

void Initialize() NN_NOEXCEPT
{
    // 何もなし
}

void Finalize() NN_NOEXCEPT
{
    // 何もなし
}

// SF を外に出さないために外部にはハンドルとして渡すようにラップする。
void OpenSessionForDev(I2cSession* pOutSession, int busIdx, Bit16 slaveAddress, nn::i2c::AddressingMode addressingMode, nn::i2c::SpeedMode speedMode ) NN_NOEXCEPT
{
    nn::sf::SharedPointer<IManager> i2cManager;

    i2cManager = GetManagerByHipc(GetServiceNameForDev(busIdx));

    nn::sf::SharedPointer<nn::i2c::ISession> session;
    auto result = i2cManager->OpenSessionForDev(&session,
                                                  static_cast<std::int32_t>(busIdx),
                                                  slaveAddress,
                                                  static_cast<std::int32_t>(addressingMode),
                                                  static_cast<std::int32_t>(speedMode));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutSession->_handle = session.Detach();
}

// SF を外に出さないために外部にはハンドルとして渡すようにラップする。
void OpenSession(I2cSession* pOutSession, nn::DeviceCode deviceCode) NN_NOEXCEPT
{
    nn::sf::SharedPointer<IManager> i2cManager;

    i2cManager = GetManagerByHipc(GetServiceName(deviceCode));

    nn::sf::SharedPointer<nn::i2c::ISession> session;
    auto result = i2cManager->OpenSession2(&session, deviceCode.GetInternalValue());
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutSession->_handle = session.Detach();
}

// SF を外に出さないために外部にはハンドルとして渡すようにラップする。
void OpenSession(I2cSession* pOutSession, nn::i2c::I2cDevice device) NN_NOEXCEPT
{
    nn::sf::SharedPointer<IManager> i2cManager;

    i2cManager = GetManagerByHipc(GetServiceName(device));

    nn::sf::SharedPointer<nn::i2c::ISession> session;
    auto result = i2cManager->OpenSession(&session, static_cast<std::int32_t>(device));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutSession->_handle = session.Detach();
}

// SessionHandle として外部に出したものを明示的に Close するための API
void CloseSession( const I2cSession& session ) NN_NOEXCEPT
{
    nn::sf::ReleaseSharedObject(static_cast<nn::i2c::ISession*>(session._handle));
}

// Session がもつ各機能のラップ版
Result Send( const I2cSession& session, const void* pInData, size_t dataBytes, nn::i2c::TransactionOption inOption ) NN_NOEXCEPT
{
    nn::sf::InBuffer inBuffer(static_cast<const char*>(pInData), dataBytes);

    auto result = GetInterface(session)->Send(inBuffer, static_cast<std::int32_t>(inOption));
    return result;
}

Result Receive( void* pOutData, const I2cSession& session, size_t dataBytes, nn::i2c::TransactionOption inOption ) NN_NOEXCEPT
{
    nn::sf::OutBuffer outBuffer(static_cast<char*>(pOutData), dataBytes);

    auto result = GetInterface(session)->Receive(outBuffer, static_cast<std::int32_t>(inOption));
    pOutData = outBuffer.GetPointerUnsafe();
    return result;
}

Result ExecuteCommandList( void* pReceiveBuffer, size_t receiveBufferSize,
                           const I2cSession& session,
                           const void* pCommandList, size_t commandListLength ) NN_NOEXCEPT
{

    nn::sf::OutBuffer           outBuffer(static_cast<char*>(pReceiveBuffer), receiveBufferSize);
    nn::sf::InArray<nn::i2c::I2cCommand> i2cCommand(static_cast<const nn::i2c::I2cCommand*>(pCommandList), commandListLength);

    auto result = GetInterface(session)->ExecuteCommandList(outBuffer, i2cCommand);

    pReceiveBuffer = outBuffer.GetPointerUnsafe();
    return result;
}

void SetRetryPolicy(const I2cSession& session, int maxRetryCount, int retryIntervalMicroSeconds) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(maxRetryCount, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(retryIntervalMicroSeconds, 0);

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetInterface(session)->SetRetryPolicy(maxRetryCount, retryIntervalMicroSeconds));
}

}}
