﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_BitPack.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/pcv/pcv.h>

#include <nn/i2c/driver/i2c_BusDev.h>
#include <nn/i2c/i2c_ResultForPrivate.h>

#include "i2c_Command.h"
#include "i2c_TargetSpec.h"
#include "i2c_RegisterI2c-soc.tegra.h"

#ifndef NN_BUILD_CONFIG_SPEC_NX
#include "i2c_RegisterClkRst-soc.tegra.h"
#endif

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
    #include "i2c_RegisterPinmux-hardware.jetson-tk1.h"
#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_SOC_TEGRA_X1)
    #include "i2c_RegisterPinmux-hardware.jetson-tk2.h"
#endif

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

class BusAccessor
{
    NN_DISALLOW_COPY(BusAccessor);
    NN_DISALLOW_MOVE(BusAccessor);

public:
    BusAccessor() NN_NOEXCEPT :
        m_pRegI2c(nullptr),
#ifndef NN_BUILD_CONFIG_SPEC_NX
        m_RegClkRst(),
#endif
        m_RegPinmux(),
        m_ClockSource(DefaultClockSource),                         // クロック周波数の設定
        m_BusFreeWaitMicroSeconds(DefaultBusFreeWaitMicroSeconds), // Stop condition後のwaitはFastModeで最小1.3us (ref. http://www.nxp.com/documents/user_manual/UM10204_JA.pdf P.48)
        m_SpeedMode(SpeedMode::SpeedMode_Fast),                    // Fast mode
        m_TimeoutMicroSeconds(DefaultTimeoutMicroSeconds),         // バスの状態判定向けのタイムアウト時間
        m_CheckRetryTimes(DefaultCheckRetryTimes),                 // タイムアウトするまでに行うバスの状態判定ポーリング回数。細かすぎると負荷が上がるのでほどほどに設定。
        m_UserCount(0),
        m_BusIdx(0),
        m_UserCountMutex(false),
        m_RegisterMutex(false),
        m_IsOpenSuspended(false)
        {}

    void      Open(int busIdx, SpeedMode speedMode) NN_NOEXCEPT;
    void      ExecuteInitialConfig() NN_NOEXCEPT;
    void      Close() NN_NOEXCEPT;

    bool      CheckBusy() const NN_NOEXCEPT;

    void      StartCondition() const NN_NOEXCEPT;
    void      StopCondition() const NN_NOEXCEPT;

    Result    StartTransaction( Bit16 slaveAddress, AddressingMode addressingMode, Command command ) NN_NOEXCEPT;
    Result    Send( const Bit8* pInData,  size_t dataBytes, TransactionOption inOption, Bit16 slaveAddress, AddressingMode addressingMode ) NN_NOEXCEPT;
    Result    Receive(    Bit8 *pOutData, size_t dataBytes, TransactionOption inOption, Bit16 slaveAddress, AddressingMode addressingMode ) NN_NOEXCEPT;
    void      Suspend() NN_NOEXCEPT;
    void      Resume() NN_NOEXCEPT;

    int       GetUserCount() NN_NOEXCEPT;

private:
    /**
     * @brief Transaction の向き
     */
    enum Xfer
    {
        Xfer_Write = 0,
        Xfer_Read  = 1,
    };

private:
    void      ResetController() const NN_NOEXCEPT;                          // [Ref1] 5.7.1 I2C コントローラをリセットする
    void      BusClear() const NN_NOEXCEPT;                                 // [Ref1] 31.1  バスをクリアする
    void      SetPinmux(int busIdx) const NN_NOEXCEPT;                      // Pinmux の設定。Pinmuxライブラリが提供されたら実装変更が必要になる想定。
    void      SetClockRegisters( SpeedMode speedMode ) NN_NOEXCEPT;         // SpeedMode に応じた設定値をクロック関連のレジスタに反映させる
    void      SetPacketModeRegisters() NN_NOEXCEPT;                         // パケットベースモード向けレジスタの初期設定を行う

    bool      IsBusy() const NN_NOEXCEPT;                                   // バスが Busy かチェックする

    Result    GetTransactionResult() const NN_NOEXCEPT;                     // Packet Transfer Status レジスタの状態を Result にマップしたものを取得します
    void      HandleTransactionError( Result result ) NN_NOEXCEPT;          // [31.4.4] Error Handling に従ったエラーハンドリングを行う
    inline Result CheckAndHandleError() NN_NOEXCEPT                         // 上記 2つの処理と割り込みの disable を行う Utility 関数
    {
        auto result = GetTransactionResult();
        HandleTransactionError(result);

        if (result.IsFailure())
        {
            DisableInterruptMask();
            nn::os::ClearInterruptEvent(&m_InterruptEvent);
            return result;
        }
        return ResultSuccess();
    }

    void      DumpRegisters(int line) const NN_NOEXCEPT;                    // デバッグ向けにレジスタをダンプする

    void      SetBaseAddress(int busIdx) NN_NOEXCEPT;                    // 仮想アドレスをセットする。

    void      WriteHeader( Xfer xfer, size_t dataBytes, TransactionOption inOption, Bit16 slaveAddress, AddressingMode addressingMode ) NN_NOEXCEPT;
    inline void DisableInterruptMask() NN_NOEXCEPT
    {
        Write32(&(m_pRegI2c->interruptMaskRegister), 0x00);
        DummyRead(&(m_pRegI2c->interruptMaskRegister));
    }

    Result    FlushFifos() NN_NOEXCEPT;

    void      DisableClock() NN_NOEXCEPT;

private:
    static const uint32_t  DefaultClockSource             = 408 * 1000 * 1000; // クロック周波数 PLLP_OUT0 = 408[MHz]
    static const uint32_t  DefaultTimeoutCycles           = 10;
    static const uint32_t  DefaultBusFreeWaitMicroSeconds = 130;
    static const uint32_t  DefaultTimeoutMicroSeconds     = DefaultTimeoutCycles * 1000 * 1000 / SpeedMode::SpeedMode_Fast; // 10[cycle] / Frequency[1/s]. 10^3 は単位の補正項
    static const uint32_t  DefaultCheckRetryTimes         = 10;

private:
    I2cRegTable*    m_pRegI2c;
#ifndef NN_BUILD_CONFIG_SPEC_NX
    ClkRstRegInfo   m_RegClkRst;
#endif
    PinmuxRegInfo   m_RegPinmux;

    uint32_t        m_ClockSource;                // クロック周波数
    uint32_t        m_BusFreeWaitMicroSeconds;    // バスの設定パラメータ群
    SpeedMode       m_SpeedMode;
    uint32_t        m_TimeoutMicroSeconds;
    uint32_t        m_CheckRetryTimes;

    nn::os::InterruptEventType  m_InterruptEvent;

    int             m_UserCount;                  // バスのユーザー数
    int             m_BusIdx;                     // バスのインデックス
    nn::pcv::Module m_PcvModule;                  // バスの PCV の Module 識別子

    nn::os::Mutex   m_UserCountMutex;             // ロック対象は m_UserCount
    nn::os::Mutex   m_RegisterMutex;              // ロック対象は IO レジスタ

    bool            m_IsOpenSuspended;
};

} // detail
} // driver
} // i2c
} // nn
