﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   I2C ドライバのバスアクセサクラスの実装部(Tegra ファミリー向け)
 * @details Jetson TK1/2 で使用可能な I2Cバスのみを対象とします。
 *          参考文献：
 *            - [Ref1] Tegra K1 Technical Reference Manual, Revision v02p, Jul 15 2014
 */

#include <climits>
#include <mutex>
#include <algorithm>

#include <nn/nn_Common.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os.h>
#include <nn/pcv/pcv.h>

#include <nn/i2c/driver/i2c.h>
#include <nn/i2c/detail/i2c_Log.h>
#include <nn/i2c/i2c_ResultForPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "i2c_BusAccessor-soc.tegra.h"

#include "i2c_Util.h"
#include "i2c_DdUtil.h"

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

namespace {

const nn::TimeSpan Timeout = nn::TimeSpan::FromMilliSeconds(100);

}

/**
 * @brief バスのオープン処理を行う関数
 */
void BusAccessor::Open(int busIdx, SpeedMode speedMode) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lockUserCount( m_UserCountMutex );
    NN_SDK_ASSERT( m_UserCount < INT_MAX ); // 値域チェック
    m_UserCount++;

    NN_DETAIL_I2C_LOG("%s(%d): BusAccessor::Open() is called at busIdx=%d, userCount=%d.\n", __FUNCTION__, __LINE__, busIdx, m_UserCount );

    if( m_UserCount > 1 )
    {
        NN_DETAIL_I2C_LOG("%s(%d): Bus Already Opened.\n", __FUNCTION__, __LINE__ );

        if( m_SpeedMode != speedMode )
        {
            // 初回 Open 時に設定された SpeedMode と異なる場合は ABORT します
            NN_ABORT("SpeedMode should be %d[bps]\n", m_SpeedMode );
        }
        return;
    }

    // 以下、初回 Open 時のみ実行
    std::lock_guard<nn::os::Mutex> lock( m_RegisterMutex );

    // 仮想アドレスをセットする。
    SetBaseAddress(busIdx);

    // Pinmux の設定
    // NX board は一括で初期設定されるので設定しない
#ifndef NN_BUILD_CONFIG_HARDWARE_NX
    SetPinmux(busIdx);
#endif

    m_BusIdx = busIdx;

#if defined(NN_BUILD_CONFIG_SPEC_NX)
    // Pcv の Module 識別子をセット
    switch(busIdx)
    {
    case 0:
        m_PcvModule = nn::pcv::Module_I2c1;
        break;

    case 1:
        m_PcvModule = nn::pcv::Module_I2c2;
        break;

    case 2:
        m_PcvModule = nn::pcv::Module_I2c3;
        break;

    case 3:
        m_PcvModule = nn::pcv::Module_I2c4;
        break;

    case 4:
        m_PcvModule = nn::pcv::Module_I2c5;
        break;

    case 5:
        m_PcvModule = nn::pcv::Module_I2c6;
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
#endif

    // 現在の設定を保持
    m_SpeedMode = speedMode;

    // 割込み関連の設定
    nn::os::InitializeInterruptEvent(&m_InterruptEvent, InterruptNameTable[busIdx], nn::os::EventClearMode_ManualClear);
    nn::os::ClearInterruptEvent(&m_InterruptEvent);

}

void BusAccessor::ExecuteInitialConfig() NN_NOEXCEPT
{
    // 以下、初回 Open 時 / Resume 時のみ実行
    std::lock_guard<nn::os::Mutex> lock( m_RegisterMutex );

#if defined(NN_BUILD_CONFIG_SPEC_NX)
    // I2C5 は PCV 専用なので pcv 系 API は叩かない
    if(m_PcvModule != nn::pcv::Module_I2c5)
    {
        // 参照カウントされるので、ここで Initialize() を叩く
        nn::pcv::Initialize();
    }
#endif

    // I2C コントローラのリセット
    ResetController();

    // クロック関連レジスタのセット
    SetClockRegisters(m_SpeedMode);

    // Packet mode への初期設定
    SetPacketModeRegisters();

    // FIFO の Flush
    FlushFifos();

    // DumpRegisters(__LINE__);
}

/**
 * @brief バスのクローズ処理を行う関数
 */
void BusAccessor::Close() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_UserCountMutex );

    NN_SDK_ASSERT( m_UserCount > INT_MIN ); // 値域チェック
    m_UserCount--;

    // ユーザー数が 0 に達していない場合は何も行わない
    if ( m_UserCount > 0 )
    {
        return;
    }

    // InterruptEvent の Finalize
    nn::os::FinalizeInterruptEvent(&m_InterruptEvent);

#if defined(NN_BUILD_CONFIG_SPEC_NX)
    // 参照カウントされるので、ここで Finalize() を叩く
    // I2C5 は PCV 専用なので pcv 系 API は叩かない
    if(m_PcvModule != nn::pcv::Module_I2c5)
    {
        nn::pcv::Finalize();
    }
#endif

    m_IsOpenSuspended = false;

    NN_DETAIL_I2C_LOG("%s(%d) Bus closed. userCount=%d\n", __FUNCTION__, __LINE__, m_UserCount );
}

/**
 * @brief バスの状態をチェックする関数
 */
bool BusAccessor::CheckBusy() const NN_NOEXCEPT
{
    uint32_t BusWaitIntervalMicroSeconds = m_TimeoutMicroSeconds / m_CheckRetryTimes;
    NN_SDK_ASSERT( BusWaitIntervalMicroSeconds > 0 );

    for ( uint32_t i = 0; i < m_CheckRetryTimes; i++ )
    {
        if ( !IsBusy() )
        {
            return false; // Bus が Free 状態
        }
        if( i == m_CheckRetryTimes - 1 )
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(BusWaitIntervalMicroSeconds));
    }
    return true; // Bus が Busy 状態
}

/**
 * @brief I2Cプロトコル仕様に基づく StartCondition を発行する関数
 */
void BusAccessor::StartCondition() const NN_NOEXCEPT
{
    // Start condition に相当する処理が無いので実装不要。
}

/**
 * @brief I2Cプロトコル仕様に基づく StopCondition を発行する関数
 */
void BusAccessor::StopCondition() const NN_NOEXCEPT
{
    // Stop condition に相当する処理が無いので実装不要
}

/**
 * @brief トランザクション開始時に呼ばれる関数
 */
Result BusAccessor::StartTransaction( Bit16 slaveAddress, AddressingMode addressingMode, Command command ) NN_NOEXCEPT
{
    // トランザクション開始時に必要な処理が無いので実装不要
    return ResultSuccess();
}

void   BusAccessor::WriteHeader( Xfer xfer, size_t dataBytes, TransactionOption inOption, Bit16 slaveAddress, AddressingMode addressingMode ) NN_NOEXCEPT
{
    uint32_t genericHeaderWord0     = 0;
    uint32_t genericHeaderWord1     = 0;
    uint32_t protocolSpecificHeader = 0;
    uint32_t slaveAddr              = ( static_cast<uint32_t>( slaveAddress & 0x7F ) << 1 ) | ( ( xfer == Xfer_Read ) ? 1 : 0 );
    const uint32_t ProtocolI2c      = 1; // [Ref1] 31.2.2 I/O Packet Format

    // [Ref1] Table 133
    Modify32 ( &genericHeaderWord0,  0 << I2cPacket::Generic0_ProtHdrSz::Pos,           I2cPacket::Generic0_ProtHdrSz::Mask );
    Modify32 ( &genericHeaderWord0,  0 << I2cPacket::Generic0_PktId::Pos,               I2cPacket::Generic0_PktId::Mask );
    Modify32 ( &genericHeaderWord0,  0 << I2cPacket::Generic0_ControllerId::Pos,        I2cPacket::Generic0_ControllerId::Mask );
    Modify32 ( &genericHeaderWord0,  ProtocolI2c << I2cPacket::Generic0_Protocol::Pos,  I2cPacket::Generic0_Protocol::Mask );
    Modify32 ( &genericHeaderWord0,  0 << I2cPacket::Generic0_PktType::Pos,             I2cPacket::Generic0_PktType::Mask );

    // [Ref1] Table 134
    Modify32 ( &genericHeaderWord1, static_cast<uint32_t>( dataBytes - 1 ) << I2cPacket::Generic1_PayloadSize::Pos,  I2cPacket::Generic1_PayloadSize::Mask );

    // [Ref1] Table 135
    Modify32 ( &protocolSpecificHeader, 0 << I2cPacket::Transmit_RespPktFreq::Pos,              I2cPacket::Transmit_RespPktFreq::Mask );
    Modify32 ( &protocolSpecificHeader, 0 << I2cPacket::Transmit_RespPktEnable::Pos,            I2cPacket::Transmit_RespPktEnable::Mask );
    Modify32 ( &protocolSpecificHeader,
        ( m_SpeedMode == SpeedMode_HighSpeed ) << I2cPacket::Transmit_HsMode::Pos,
        I2cPacket::Transmit_HsMode::Mask );
    Modify32 ( &protocolSpecificHeader, 0 << I2cPacket::Transmit_ContinueOnNack::Pos,           I2cPacket::Transmit_ContinueOnNack::Mask );
    Modify32 ( &protocolSpecificHeader, 0 << I2cPacket::Transmit_SendStartByte::Pos,            I2cPacket::Transmit_SendStartByte::Mask );
    Modify32 ( &protocolSpecificHeader,
        ( ( xfer == Xfer_Read ) ? 1 : 0 ) << I2cPacket::Transmit_ReadWrite::Pos,
        I2cPacket::Transmit_ReadWrite::Mask );
    Modify32 ( &protocolSpecificHeader,
        !( addressingMode == AddressingMode_BitWidth7 ) << I2cPacket::Transmit_AddressMode::Pos,
        I2cPacket::Transmit_AddressMode::Mask );
    // パケットモードのときはこのビットを立てないと、パケット送信完了割り込みが受けられない
    Modify32 ( &protocolSpecificHeader, 1 << I2cPacket::Transmit_Ie::Pos,                       I2cPacket::Transmit_Ie::Mask );
    Modify32 ( &protocolSpecificHeader,
        !( inOption & TransactionOption_StopCondition ) << I2cPacket::Transmit_RepeatStartStop::Pos,
        I2cPacket::Transmit_RepeatStartStop::Mask );
    Modify32 ( &protocolSpecificHeader, 0 << I2cPacket::Transmit_ContinueXfer::Pos,             I2cPacket::Transmit_ContinueXfer::Mask );
    Modify32 ( &protocolSpecificHeader, 0 << I2cPacket::Transmit_HsMasterAddr::Pos,             I2cPacket::Transmit_HsMasterAddr::Mask );
    Modify32 ( &protocolSpecificHeader, slaveAddr << I2cPacket::Transmit_SlaveAddr::Pos,        I2cPacket::Transmit_SlaveAddr::Mask );

    FlushFifos();

    // TX FIFO に書き込み
    Write32  ( &(m_pRegI2c->txPacketFifo), genericHeaderWord0 );
    Write32  ( &(m_pRegI2c->txPacketFifo), genericHeaderWord1 );
    Write32  ( &(m_pRegI2c->txPacketFifo), protocolSpecificHeader );
}

/**
 * @brief Send 関数
 */
Result   BusAccessor::Send( const Bit8* pInData,  size_t dataBytes, TransactionOption inOption, Bit16 slaveAddress, AddressingMode addressingMode ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_RegisterMutex );

    size_t      remainingBytes = dataBytes;
    const Bit8* pData          = pInData;

    // Send で使用する割り込みを enable にする
    Write32 ( &(m_pRegI2c->interruptMaskRegister),
            I2cRegister::InterruptMask_ArbLostIntEn::Mask |
            I2cRegister::InterruptMask_NoAckIntEn::Mask |
            I2cRegister::InterruptMask_TfifoDataReq::Mask |
            I2cRegister::InterruptMask_PacketXferCompleteIntEn::Mask);

    // Clear interrupt status register
    // 1 を write でクリアされる。
    Write32  ( &(m_pRegI2c->interruptStatusRegister), I2cRegister::InterruptStatus_AllPacketsXferComplete::Mask |
                                                      I2cRegister::InterruptStatus_PacketXferComplete::Mask     |
                                                      I2cRegister::InterruptStatus_NoAck::Mask                  |
                                                      I2cRegister::InterruptStatus_ArbLost::Mask                |
                                                      I2cRegister::InterruptStatus_TfifoOvf::Mask               |
                                                      I2cRegister::InterruptStatus_RfifoOvf::Mask);

    // ヘッダーの書き込み
    WriteHeader(Xfer_Write, dataBytes, inOption, slaveAddress, addressingMode);

    while (NN_STATIC_CONDITION(true))
    {
        // FIFO の Status を取得
        uint32_t fifoStatus = Read32 ( &(m_pRegI2c->fifoStatus) );

        // FIFO Slot Empty のカウントが空いている分、ループを回して FIFO へ書き込む
        for (uint32_t j = 0; remainingBytes > 0 && j < ((fifoStatus & I2cRegister::FifoStatus_TxFifoEmptyCnt::Mask) >> I2cRegister::FifoStatus_TxFifoEmptyCnt::Pos); j++)
        {
            const size_t MaxSingleDataBytes = 4; // 1 word
            size_t nextBytes = (remainingBytes < MaxSingleDataBytes) ? remainingBytes : MaxSingleDataBytes;
            uint32_t data = 0;
            for (size_t i = 0; i < nextBytes; i++ )
            {
                data |= pData[i] << (8 * i);
            }
            Write32  ( &(m_pRegI2c->txPacketFifo), data );

            remainingBytes -= nextBytes;
            pData += nextBytes;
        }

        // すべて FIFO に書き込めたら break
        if (remainingBytes == 0)
        {
            break;
        }

        // まだ送信したいデータがある場合、FIFO がすべて空になるのを待つ
        nn::os::ClearInterruptEvent(&m_InterruptEvent);
        if (!nn::os::TimedWaitInterruptEvent(&m_InterruptEvent, Timeout))
        {
            NN_DETAIL_I2C_ERROR("send time out occur(%d), busIdx = %d, slaveaddr = 0x%x\n", __LINE__, m_BusIdx, slaveAddress);
            //DumpRegisters(__LINE__);
            // BusBusy が起きたのと同じリセット処理を行う
            HandleTransactionError(ResultBusBusy());

            DisableInterruptMask();
            nn::os::ClearInterruptEvent(&m_InterruptEvent);
            return ResultTimeout();
        }

        // エラーのチェック
        NN_RESULT_DO(CheckAndHandleError());
    }

    // Fifo empty を無効化する
    Write32 ( &(m_pRegI2c->interruptMaskRegister),
            I2cRegister::InterruptMask_ArbLostIntEn::Mask |
            I2cRegister::InterruptMask_NoAckIntEn::Mask |
            I2cRegister::InterruptMask_PacketXferCompleteIntEn::Mask);

    // PacketXferComplete を待つ
    while (NN_STATIC_CONDITION(true))
    {
        // 最初にエラーのチェック
        NN_RESULT_DO(CheckAndHandleError());

        // InterruptStatus をチェックしてパケットの転送が完了しているか確認
        uint32_t interruptstatus = Read32( &(m_pRegI2c->interruptStatusRegister));
        if (interruptstatus & I2cRegister::InterruptStatus_PacketXferComplete::Mask)
        {
            NN_RESULT_DO(CheckAndHandleError());
            break;
        }

        // 転送が完了してない場合、完了まで待つ
        nn::os::ClearInterruptEvent(&m_InterruptEvent);
        if (!nn::os::TimedWaitInterruptEvent(&m_InterruptEvent, Timeout))
        {
            NN_DETAIL_I2C_ERROR("packet send time out occur(%d), busIdx = %d, slaveaddr = 0x%x\n", __LINE__, m_BusIdx, slaveAddress);
            //DumpRegisters(__LINE__);
            // BusBusy が起きたのと同じリセット処理を行う
            HandleTransactionError(ResultBusBusy());

            DisableInterruptMask();
            nn::os::ClearInterruptEvent(&m_InterruptEvent);
            return ResultTimeout();
        }
    }

    DisableInterruptMask();
    return ResultSuccess();
}

/**
 * @brief Receive 関数
 */
Result    BusAccessor::Receive( Bit8 *pOutData, size_t dataBytes, TransactionOption inOption, Bit16 slaveAddress, AddressingMode addressingMode ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_RegisterMutex );

    size_t     remainingBytes = dataBytes;
    Bit8*      pData          = pOutData;

    // Receive で使用する割り込みを enable にする
    Write32 ( &(m_pRegI2c->interruptMaskRegister),
            I2cRegister::InterruptMask_ArbLostIntEn::Mask |
            I2cRegister::InterruptMask_NoAckIntEn::Mask |
            I2cRegister::InterruptMask_RfifoDataReq::Mask |
            I2cRegister::InterruptMask_PacketXferCompleteIntEn::Mask);

    // Clear interrupt status register
    // 1 を write でクリアされる。
    Write32  ( &(m_pRegI2c->interruptStatusRegister), I2cRegister::InterruptStatus_AllPacketsXferComplete::Mask |
                                                      I2cRegister::InterruptStatus_PacketXferComplete::Mask     |
                                                      I2cRegister::InterruptStatus_NoAck::Mask                  |
                                                      I2cRegister::InterruptStatus_ArbLost::Mask                |
                                                      I2cRegister::InterruptStatus_TfifoOvf::Mask               |
                                                      I2cRegister::InterruptStatus_RfifoOvf::Mask);

    WriteHeader( Xfer_Read, dataBytes, inOption, slaveAddress, addressingMode );

    while (remainingBytes > 0)
    {
        // Packet の受信完了もしくは FIFO が 7 slot 埋まるのを待つ
        nn::os::ClearInterruptEvent(&m_InterruptEvent);
        if (!nn::os::TimedWaitInterruptEvent(&m_InterruptEvent, Timeout))
        {
            NN_DETAIL_I2C_ERROR("receive time out occur(%d), busIdx = %d, slaveaddr = 0x%x\n", __LINE__, m_BusIdx, slaveAddress);
            //DumpRegisters(__LINE__);
            // BusBusy が起きたのと同じリセット処理を行う
            HandleTransactionError(ResultBusBusy());
            DisableInterruptMask();
            nn::os::ClearInterruptEvent(&m_InterruptEvent);
            return ResultTimeout();
        }

        // エラーのチェック
        NN_RESULT_DO(CheckAndHandleError());

        // FIFO のステータスをチェック
        uint32_t fifoStatus = Read32 ( &(m_pRegI2c->fifoStatus) );

        const uint32_t MaxSingleDataBytes = 4; // 1 word

        // 1度で取り出せるワードの数を確認
        uint32_t numWord = std::min(
            static_cast<uint32_t>((remainingBytes + MaxSingleDataBytes - 1) / MaxSingleDataBytes),
            ((fifoStatus & I2cRegister::FifoStatus_RxFifoFullCnt::Mask) >> I2cRegister::FifoStatus_RxFifoFullCnt::Pos)
        );

        for (uint32_t j = 0; j < numWord; j++)
        {
            uint32_t data  = Read32 ( &(m_pRegI2c->rxFifo) );
            uint32_t nextBytes = std::min(static_cast<uint32_t>(remainingBytes), MaxSingleDataBytes);
            for(uint32_t i = 0; i < nextBytes; i++ )
            {
                pData[i] = (data >> (8 * i)) & 0xff;
            }
            remainingBytes -= nextBytes;
            pData += nextBytes;
        }
    }

    return ResultSuccess();
}

/**
 * @brief 仮想アドレスをセットする関数
 */
void BusAccessor::SetBaseAddress(int busIdx) NN_NOEXCEPT
{
    // 使用する I2C コンローラーに対応するレジスタをセットする
    m_pRegI2c = GetI2cReg(busIdx);
#ifndef NN_BUILD_CONFIG_SPEC_NX
    m_RegClkRst.Set(busIdx);
#endif
    m_RegPinmux.Set(busIdx);
}

/**
 * @brief リセットレジスタを使ってバスをリセットする。
 *        クロックソースは PLLP_OUT0 (408MHz) 一択。
 */
void BusAccessor::ResetController() const NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_SPEC_NX)

    //CL-DVFS 側で処理されるので、I2C5 は触らなくてもよいはず
    if(m_PcvModule != nn::pcv::Module_I2c5)
    {
        // Assert reset signal
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetReset(m_PcvModule, true));

        // [5.7.68] I2C#_CLK_DIVISOR = 0x04
        // Set Clock Rate (408MHz / 5 = 81600000)
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetClockRate(m_PcvModule, 81600000 ));

        // De-assert reset signal
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetReset(m_PcvModule, false));
    }

#else

    // [5.7.1] Precautions
    // 1. Assert reset signal
    SetBitOn32  (m_RegClkRst.pReset, m_RegClkRst.resetBitMask);

    // 2. Enable clock
    // [Ref1] p.2189 : The clock enable must also be given to the I2C controller, before any of registers are written.
    SetBitOn32  (m_RegClkRst.pClockEnable, m_RegClkRst.clockEnableBitMask);

    // 3. Change Clock divisor
    Modify32  ( m_RegClkRst.pClockSource, (0x04 << 0),   (0xFF) );      // [5.7.68] I2C#_CLK_DIVISOR = 0x04
    DummyRead(m_RegClkRst.pClockSource);

    // 4. Wait 1 us
    nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(1) );

    // 5. Change the clock source
    Modify32  ( m_RegClkRst.pClockSource, (0 << 29),     (0x7 << 29) ); // [5.7.68] CLK_M = PLLP_OUT0
    DummyRead(m_RegClkRst.pClockSource);

    // 6. Wait 2 us
    nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(2) );

    // 7. De-assert reset signal
    SetBitOff32 (m_RegClkRst.pReset, m_RegClkRst.resetBitMask);
    DummyRead(m_RegClkRst.pReset);
#endif
}

/**
 * @brief [Ref1] 31.1 バスクリア用関数。Arbitration lost が起きた時に暗黙的に呼ばれる
 */
void BusAccessor::BusClear() const NN_NOEXCEPT
{
    const int BusyLoopWaitMicroSeconds = 1000;
    const int MaxRetryCount = 3;
    bool needRetry = false;
    int tryCount = 0;
    do
    {
        needRetry = false;
        tryCount++;

        // [Ref1] 31.1.1 Bus Clear Master Operation
        // 番号は Bus Clear Programming Procedure の記載内容に準拠しています

        // 1
        ResetController();

        // 2
        uint32_t  clockPulses = 0x9;
        Write32(&(m_pRegI2c->busClearConfig), (clockPulses << I2cRegister::BusClearConfig_BcSclkThreshold::Pos) & I2cRegister::BusClearConfig_BcSclkThreshold::Mask);

        // 3
        SetBitOn32(&(m_pRegI2c->busClearConfig), I2cRegister::BusClearConfig_BcStopCond::Mask);

        // 4
        SetBitOn32(&(m_pRegI2c->busClearConfig), I2cRegister::BusClearConfig_BcTerminate::Mask);

        // 5
        NN_DETAIL_I2C_LOG("BusClear : 5\n");
        SetBitOn32(&(m_pRegI2c->configLoad), I2cRegister::ConfigLoad_MstrConfigLoad::Mask);
        nn::os::Tick startTick5 = nn::os::GetSystemTick();
        while (Read32(&(m_pRegI2c->configLoad)) & (I2cRegister::ConfigLoad_MstrConfigLoad::Mask))
        {
            nn::os::Tick endTick5 = nn::os::GetSystemTick();
            if ((endTick5 - startTick5).ToTimeSpan().GetMicroSeconds() > BusyLoopWaitMicroSeconds)
            {
                needRetry = true;
                break;
            }
            NN_DETAIL_I2C_LOG("BusyLoop\n");
        }

        if (needRetry)
        {
            continue;
        }

        // 6
        NN_DETAIL_I2C_LOG("BusClear : 6\n");
        SetBitOn32(&(m_pRegI2c->busClearConfig), I2cRegister::BusClearConfig_BcEnable::Mask);
        nn::os::Tick startTick6 = nn::os::GetSystemTick();
        while (Read32(&(m_pRegI2c->busClearConfig)) & (I2cRegister::BusClearConfig_BcEnable::Mask))
        {
            nn::os::Tick endTick6 = nn::os::GetSystemTick();
            if ((endTick6 - startTick6).ToTimeSpan().GetMicroSeconds() > BusyLoopWaitMicroSeconds)
            {
                needRetry = true;
                break;
            }
            NN_DETAIL_I2C_LOG("BusyLoop\n");
        }

        if (needRetry)
        {
            continue;
        }

        // 7
        // 8
        // 9
        NN_DETAIL_I2C_LOG("BusClear : 9\n");
        nn::os::Tick startTick9 = nn::os::GetSystemTick();
        while (!(Read32(&(m_pRegI2c->busClearStatus)) & (I2cRegister::BusClearStatus_BcStatus::Mask)))
        {
            nn::os::Tick endTick9 = nn::os::GetSystemTick();
            if ((endTick9 - startTick9).ToTimeSpan().GetMicroSeconds() > BusyLoopWaitMicroSeconds)
            {
                needRetry = true;
                break;
            }
            NN_DETAIL_I2C_LOG("BusyLoop\n");
        }

        if (needRetry)
        {
            continue;
        }
        NN_DETAIL_I2C_LOG("BusClear Done\n");

    } while (tryCount < MaxRetryCount && needRetry);
}

/**
 * @brief Pinmux の設定。Pinmuxライブラリが提供されたら実装変更が必要になる想定。
 */
void BusAccessor::SetPinmux(int busIdx) const NN_NOEXCEPT
{
    // PINMUX の設定。所定のバス番号に対応する SDA/SCL をアクティベートします。
    // Mask が 0 の場合は操作不要と見做して設定処理を飛ばします。
    if( m_RegPinmux.sclBitMask != 0 )
    {
        NN_DETAIL_I2C_LOG("%s(%d): Set PinmuxScl @ busIdx=%d.\n", __FUNCTION__, __LINE__, busIdx );
        Modify32  ( m_RegPinmux.pScl, m_RegPinmux.sclBitValue, m_RegPinmux.sclBitMask );
    }

    if( m_RegPinmux.sdaBitMask != 0 )
    {
        NN_DETAIL_I2C_LOG("%s(%d): Set PinmuxSda @ busIdx=%d.\n", __FUNCTION__, __LINE__, busIdx );
        Modify32  ( m_RegPinmux.pSda, m_RegPinmux.sdaBitValue, m_RegPinmux.sdaBitMask );
    }
    DummyRead(m_RegPinmux.pScl);
}

/**
 * @brief Clock関連のレジスタ設定を行う関数
 */
void BusAccessor::SetClockRegisters( SpeedMode speedMode ) NN_NOEXCEPT
{
    uint32_t tHigh;
    uint32_t tLow;
    uint32_t clkDivisor;
    uint32_t debounce;
    uint32_t sourceDiv;
    bool     isHighSpeed = false;

    // [Ref1] Table 137 @ p.2189
    switch(speedMode)
    {
    case SpeedMode_Standard  :
        tHigh      = 0x02;
        tLow       = 0x04;
        clkDivisor = 0x19;
        debounce   = 2;
        sourceDiv  = 0x13;
        break;
    case SpeedMode_Fast      :
        tHigh      = 0x02;
        tLow       = 0x04;
        clkDivisor = 0x19;
        debounce   = 2;
        sourceDiv  = 0x04;
        break;
    case SpeedMode_FastPlus  :
        tHigh      = 0x02;
        tLow       = 0x04;
        clkDivisor = 0x10;
        debounce   = 0;
        sourceDiv  = 0x02;
        break;
    case SpeedMode_HighSpeed :
        tHigh      = 0x03;
        tLow       = 0x08;
        clkDivisor = 0x02;
        debounce   = 0;
        sourceDiv  = 0x02;
        isHighSpeed = true;
        break;
    default: NN_UNEXPECTED_DEFAULT;

    }

    // Frequency Divisor の設定
    if( !isHighSpeed )
    {
        // SM/FM/FM+ は処理が同一
        Write32( &(m_pRegI2c->interfaceTiming_0),  ( tHigh << I2cRegister::InterfaceTiming_0_Thigh::Pos |
                                                     tLow  << I2cRegister::InterfaceTiming_0_Tlow::Pos )
                                                     & I2cRegister::InterfaceTiming_0::Mask );
        Write32( &(m_pRegI2c->clkDivisorRegister), ( clkDivisor << I2cRegister::ClkDivisorRegister_StdFastMode::Pos )
                                                     & I2cRegister::ClkDivisorRegister_StdFastMode::Mask);
    }
    else
    {
        // HS は書き込み先レジスタが上記3モードとは異なる
        Write32( &(m_pRegI2c->hsInterfaceTiming_0),  ( tHigh << I2cRegister::HsInterfaceTiming_0_Thigh::Pos |
                                                       tLow  << I2cRegister::HsInterfaceTiming_0_Tlow::Pos )
                                                       & I2cRegister::HsInterfaceTiming_0::Mask);
        Write32( &(m_pRegI2c->clkDivisorRegister),   ( clkDivisor << I2cRegister::ClkDivisorRegister_HsMode::Pos )
                                                       & I2cRegister::ClkDivisorRegister_HsMode::Mask);
    }

    Write32( &(m_pRegI2c->cnfg), ( debounce << I2cRegister::Cnfg_Debounce::Pos ) & I2cRegister::Cnfg_Debounce::Mask);                  // Debounce  = 2
    DummyRead(&(m_pRegI2c->cnfg));

    // [5.7.68] I2C#_CLK_DIVISOR の変更
#if defined(NN_BUILD_CONFIG_SPEC_NX)

    const uint32_t PllpOut0Hz = 408000000;

    // I2C5 は ClockSource は触らない
    if(m_PcvModule != nn::pcv::Module_I2c5)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetReset(m_PcvModule, true));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetClockRate(m_PcvModule, PllpOut0Hz / (sourceDiv + 1)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetReset(m_PcvModule, false));
    }
#else
    Modify32( m_RegClkRst.pClockSource, ( sourceDiv << 0), (0xFFFF) );
    DummyRead(m_RegClkRst.pClockSource);
#endif
}

/**
 * @brief パケットベースモード向けレジスタの初期設定を行う
 */
void BusAccessor::SetPacketModeRegisters() NN_NOEXCEPT
{
    // Tx : 7 word 空きがあったら Trigger
    // Rx : 7 word 貯まっていたら Trigger
    const uint32_t TxFifoTrigDefault = 7;
    const uint32_t RxFifoTrigDefault = 7;

    SetBitOn32  ( &(m_pRegI2c->cnfg),       I2cRegister::Cnfg_PacketModeEn::Mask);
    SetBitOn32  ( &(m_pRegI2c->configLoad), I2cRegister::ConfigLoad_MstrConfigLoad::Mask);

    // FIFO の Trigger level を設定
    Bit32 fifoControl = (TxFifoTrigDefault << I2cRegister::FifoControl_TxFifoTrig::Pos) | (RxFifoTrigDefault << I2cRegister::FifoControl_RxFifoTrig::Pos);
    Write32  ( &(m_pRegI2c->fifoControl), fifoControl );
}


/**
 * @brief Bus の状態をチェックする関数
 */
bool BusAccessor::IsBusy() const NN_NOEXCEPT
{
    // Packet based な実装のときは busy を示すステータスレジスタが無いため false としておく
    return false;
}

/**
 * @brief Packet Transfer Status レジスタと I2C Interrupt Status の状態を Result にマップしたものを取得します
 */
Result BusAccessor::GetTransactionResult() const NN_NOEXCEPT
{
    uint32_t packetstatus    = Read32( &(m_pRegI2c->packetTransferStatus));
    uint32_t interruptstatus = Read32( &(m_pRegI2c->interruptStatusRegister));

    if( packetstatus & I2cRegister::PacketTransferStatus_NoAckForAddr::Mask ||
        packetstatus & I2cRegister::PacketTransferStatus_NoAckForData::Mask ||
        interruptstatus & I2cRegister::InterruptStatus_NoAck::Mask)
    {
        return ResultNoAck();
    }
    if( packetstatus & I2cRegister::PacketTransferStatus_ArbLost::Mask ||
        interruptstatus & I2cRegister::InterruptStatus_ArbLost::Mask)
    {
        //  [Ref1] 31.1 Arbitration lost 時の対処
        BusClear();
        // Arbitrationlost をユーザーに通知したところでハンドリングできないから Busy と見做す。
        return ResultBusBusy();
    }

    return  ResultSuccess();
}

/**
 * @brief [31.4.4] Error Handling に従ったエラーハンドリングを行う
 */
void BusAccessor::HandleTransactionError( Result result ) NN_NOEXCEPT
{
    if (result.IsFailure())
    {
        if ( ResultNoAck::Includes(result) ||
             ResultBusBusy::Includes(result) )
        {
            // I2C コントローラのリセット
            ResetController();

            // クロック関連レジスタのセット
            SetClockRegisters(m_SpeedMode);

            // Packet mode への初期設定
            SetPacketModeRegisters();

            // FIFO の Flush
            FlushFifos();
        }
        else
        {
            NN_ABORT("UNEXPECTED ERROR\n");
        }
    }
    else
    {
        // 成功時は何もしない
    }
}

/**
 * @brief [31.5.20] FIFO を Flush する関数
 */
Result   BusAccessor::FlushFifos() NN_NOEXCEPT
{
    const int NumberOfMaxTries = 5;
    int       currentTries     = 0;

    // Tx : 7 word 空きがあったら Trigger
    // Rx : 7 word 貯まっていたら Trigger
    const uint32_t TxFifoTrigDefault = 7;
    const uint32_t RxFifoTrigDefault = 7;

    // FIFO の Trigger level を設定
    Bit32 fifoControl =
        (TxFifoTrigDefault << I2cRegister::FifoControl_TxFifoTrig::Pos) |
        (RxFifoTrigDefault << I2cRegister::FifoControl_RxFifoTrig::Pos) |
        I2cRegister::FifoControl_RxFifoFlush::Mask |
        I2cRegister::FifoControl_TxFifoFlush::Mask;

    Write32  ( &(m_pRegI2c->fifoControl), fifoControl );

    while( Read32( &(m_pRegI2c->fifoControl)) & ( I2cRegister::FifoControl_RxFifoFlush::Mask | I2cRegister::FifoControl_TxFifoFlush::Mask ) )
    {
        if( ++currentTries >= NumberOfMaxTries )
        {
            NN_DETAIL_I2C_ERROR("Failed to flush FIFO.\n");
            return ResultBusBusy();
        }
        NN_DETAIL_I2C_LOG("Flushing FIFO...\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    }
    return ResultSuccess();
}

int BusAccessor::GetUserCount() NN_NOEXCEPT
{
    return m_UserCount;
}

void BusAccessor::DisableClock() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetClockEnabled(m_PcvModule, false));
}

void BusAccessor::Suspend() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lockUserCount(m_UserCountMutex);
    std::lock_guard<nn::os::Mutex> lock(m_RegisterMutex);

    if (!m_IsOpenSuspended)
    {
        m_IsOpenSuspended = true;

#if defined(NN_BUILD_CONFIG_SPEC_NX)

        if (m_PcvModule != nn::pcv::Module_I2c5)
        {
            DisableClock();
        }
#endif
    }
}

void BusAccessor::Resume() NN_NOEXCEPT
{
    // ExecuteInitialiConfig() でロックが取られるのでここでは取らない。

    if (m_IsOpenSuspended)
    {
        ExecuteInitialConfig();
    }
    m_IsOpenSuspended = false;
}

/**
 * @brief デバッグ用関数。関連レジスタの情報をダンプする。
 */
void BusAccessor::DumpRegisters(int line) const NN_NOEXCEPT
{
    NN_DETAIL_I2C_INFO("\t%s(%d)\n", __FUNCTION__, line );
    NN_DETAIL_I2C_INFO("\n\t*** I2C Register info ***\n\n");

    NN_DETAIL_I2C_INFO("\t[Register locations]---------------------\n");
#ifndef NN_BUILD_CONFIG_SPEC_NX
    NN_DETAIL_I2C_INFO("\t  clockSource                  = 0x%08x @ 0x%p\n", *(m_RegClkRst.pClockSource), m_RegClkRst.pClockSource);
    NN_DETAIL_I2C_INFO("\t  clockEnable                  = 0x%08x @ 0x%p\n", *(m_RegClkRst.pClockEnable), m_RegClkRst.pClockEnable);
    NN_DETAIL_I2C_INFO("\t  clockEnableBitMask           = 0x%08x\n",        m_RegClkRst.clockEnableBitMask);

    NN_DETAIL_I2C_INFO("\t  reset                        = 0x%08x @ 0x%p\n", *(m_RegClkRst.pReset), m_RegClkRst.pReset);
    NN_DETAIL_I2C_INFO("\t  resetBitMask                 = 0x%08x\n",        m_RegClkRst.resetBitMask);
#endif

    NN_DETAIL_I2C_INFO("\t  pinmuxScl                    = 0x%08x @ 0x%p\n", *(m_RegPinmux.pScl), m_RegPinmux.pScl );
    NN_DETAIL_I2C_INFO("\t  pinmuxSclBitValue            = 0x%08x\n",        m_RegPinmux.sclBitValue );
    NN_DETAIL_I2C_INFO("\t  pinmuxSclBitMask             = 0x%08x\n",        m_RegPinmux.sclBitMask );
    NN_DETAIL_I2C_INFO("\t  pinmuxSda                    = 0x%08x @ 0x%p\n", *(m_RegPinmux.pSda), m_RegPinmux.pSda );
    NN_DETAIL_I2C_INFO("\t  pinmuxSdaBitValue            = 0x%08x\n",        m_RegPinmux.sdaBitValue );
    NN_DETAIL_I2C_INFO("\t  pinmuxSdaBitMask             = 0x%08x\n",        m_RegPinmux.sdaBitMask );

    NN_DETAIL_I2C_INFO("\t[Register values]---------------------\n");
    NN_DETAIL_I2C_INFO("\t  i2c->cnfg                    = 0x%08x\n", m_pRegI2c->cnfg );
    NN_DETAIL_I2C_INFO("\t  i2c->cmdAddr0                = 0x%08x\n", m_pRegI2c->cmdAddr0 );
    NN_DETAIL_I2C_INFO("\t  i2c->cmdAddr1                = 0x%08x\n", m_pRegI2c->cmdAddr1 );
    NN_DETAIL_I2C_INFO("\t  i2c->cmdData1                = 0x%08x\n", m_pRegI2c->cmdData1 );
    NN_DETAIL_I2C_INFO("\t  i2c->cmdData2                = 0x%08x\n", m_pRegI2c->cmdData2 );
    // NN_DETAIL_I2C_INFO("\t i2c->NN_PADDING4             = 0x%08x\n", m_pRegI2c->NN_PADDING4 );
    // NN_DETAIL_I2C_INFO("\t i2c->NN_PADDING4             = 0x%08x\n", m_pRegI2c->NN_PADDING4 );
    NN_DETAIL_I2C_INFO("\t  i2c->status                  = 0x%08x\n", m_pRegI2c->status );
    NN_DETAIL_I2C_INFO("\t  i2c->slCnfg                  = 0x%08x\n", m_pRegI2c->slCnfg );
    NN_DETAIL_I2C_INFO("\t  i2c->slRcvd                  = 0x%08x\n", m_pRegI2c->slRcvd );
    NN_DETAIL_I2C_INFO("\t  i2c->slStatus                = 0x%08x\n", m_pRegI2c->slStatus );
    NN_DETAIL_I2C_INFO("\t  i2c->slAddr1                 = 0x%08x\n", m_pRegI2c->slAddr1 );
    NN_DETAIL_I2C_INFO("\t  i2c->slAddr2                 = 0x%08x\n", m_pRegI2c->slAddr2 );
    NN_DETAIL_I2C_INFO("\t  i2c->tlowSext                = 0x%08x\n", m_pRegI2c->tlowSext );
    // NN_DETAIL_I2C_INFO("\t i2c->NN_PADDING4             = 0x%08x\n", m_pRegI2c->NN_PADDING4 );
    NN_DETAIL_I2C_INFO("\t  i2c->slDelayCount            = 0x%08x\n", m_pRegI2c->slDelayCount );
    NN_DETAIL_I2C_INFO("\t  i2c->slIntMask               = 0x%08x\n", m_pRegI2c->slIntMask );
    NN_DETAIL_I2C_INFO("\t  i2c->slIntSource             = 0x%08x\n", m_pRegI2c->slIntSource );
    NN_DETAIL_I2C_INFO("\t  i2c->slIntSet                = 0x%08x\n", m_pRegI2c->slIntSet );
    // NN_DETAIL_I2C_INFO("\t i2c->NN_PADDING4             = 0x%08x\n", m_pRegI2c->NN_PADDING4 );
    NN_DETAIL_I2C_INFO("\t  i2c->txPacketFifo            = 0x%08x\n", m_pRegI2c->txPacketFifo );
    NN_DETAIL_I2C_INFO("\t  i2c->rxFifo                  = 0x%08x\n", m_pRegI2c->rxFifo );
    NN_DETAIL_I2C_INFO("\t  i2c->packetTransferStatus    = 0x%08x\n", m_pRegI2c->packetTransferStatus );
    NN_DETAIL_I2C_INFO("\t  i2c->fifoControl             = 0x%08x\n", m_pRegI2c->fifoControl );
    NN_DETAIL_I2C_INFO("\t  i2c->fifoStatus              = 0x%08x\n", m_pRegI2c->fifoStatus );
    NN_DETAIL_I2C_INFO("\t  i2c->interruptMaskRegister   = 0x%08x\n", m_pRegI2c->interruptMaskRegister );
    NN_DETAIL_I2C_INFO("\t  i2c->interruptStatusRegister = 0x%08x\n", m_pRegI2c->interruptStatusRegister );
    NN_DETAIL_I2C_INFO("\t  i2c->clkDivisorRegister      = 0x%08x\n", m_pRegI2c->clkDivisorRegister );
    NN_DETAIL_I2C_INFO("\t  i2c->interruptSourceRegister = 0x%08x\n", m_pRegI2c->interruptSourceRegister );
    NN_DETAIL_I2C_INFO("\t  i2c->interruptSetRegister    = 0x%08x\n", m_pRegI2c->interruptSetRegister );
    NN_DETAIL_I2C_INFO("\t  i2c->slvTxPacketFifo         = 0x%08x\n", m_pRegI2c->slvTxPacketFifo );
    NN_DETAIL_I2C_INFO("\t  i2c->slvRxFifo               = 0x%08x\n", m_pRegI2c->slvRxFifo );
    NN_DETAIL_I2C_INFO("\t  i2c->slvPacketStatus         = 0x%08x\n", m_pRegI2c->slvPacketStatus );
    NN_DETAIL_I2C_INFO("\t  i2c->busClearConfig          = 0x%08x\n", m_pRegI2c->busClearConfig );
    NN_DETAIL_I2C_INFO("\t  i2c->busClearStatus          = 0x%08x\n", m_pRegI2c->busClearStatus );
    NN_DETAIL_I2C_INFO("\t  i2c->configLoad              = 0x%08x\n", m_pRegI2c->configLoad );
    // NN_DETAIL_I2C_INFO("\t i2c->NN_PADDING4             = 0x%08x\n", m_pRegI2c->NN_PADDING4 );
    NN_DETAIL_I2C_INFO("\t  i2c->interfaceTiming_0       = 0x%08x\n", m_pRegI2c->interfaceTiming_0 );
    NN_DETAIL_I2C_INFO("\t  i2c->interfaceTiming_1       = 0x%08x\n", m_pRegI2c->interfaceTiming_1 );
    NN_DETAIL_I2C_INFO("\t  i2c->hsInterfaceTiming_0     = 0x%08x\n", m_pRegI2c->hsInterfaceTiming_0 );
    NN_DETAIL_I2C_INFO("\t  i2c->hsInterfaceTiming_1     = 0x%08x\n", m_pRegI2c->hsInterfaceTiming_1 );
}

} // detail
} // driver
} // i2c
} // nn
