﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_InterruptEvent.h>
#include <nn/i2c/driver/i2c_BusDev.h>

#include "i2c_Command.h"

#include "i2c_TargetSpec-hardware.bdsl-imx6.h"

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

class BusAccessor
{
    NN_DISALLOW_COPY(BusAccessor);
    NN_DISALLOW_MOVE(BusAccessor);

public:
    BusAccessor() NN_NOEXCEPT :
        m_BusFreeWaitMicroSeconds(DefaultBusFreeWaitMicroSeconds) , // Stop condition後のwaitはFastModeで最小1.3us (ref. http://www.nxp.com/documents/user_manual/UM10204_JA.pdf P.48)
        m_SpeedMode(SpeedMode_Fast)                               , // Fast mode [Hz]
        m_TimeoutMicroSeconds(DefaultTimeoutMicroSeconds)         , // Micro seconds (ref. 35.5.7 NOTE)
        m_CheckRetryTimes(DefaultCheckRetryTimes)                 , // トランザクションがタイムアウトするまでのポーリング回数。細かすぎると負荷が上がるのでほどほどに設定。
        m_UserCount(0)
        {}

    void Open(int busIdx, SpeedMode speedMode ) NN_NOEXCEPT;
    void ExecuteInitialConfig() NN_NOEXCEPT;
    void Close() NN_NOEXCEPT;

    bool CheckBusy() const NN_NOEXCEPT;

    void StartCondition() const NN_NOEXCEPT;
    void StopCondition() const NN_NOEXCEPT;

    Result StartTransaction( Bit16 slaveAddress, AddressingMode addressingMode, Command command ) NN_NOEXCEPT;
    Result Send( const Bit8* pInData,  size_t dataBytes, TransactionOption inOption, Bit16 slaveAddress, AddressingMode addressingMode ) NN_NOEXCEPT;
    Result Receive(    Bit8 *pOutData, size_t dataBytes, TransactionOption inOption, Bit16 slaveAddress, AddressingMode addressingMode ) NN_NOEXCEPT;
    void   Suspend() NN_NOEXCEPT;
    void   Resume() NN_NOEXCEPT;

    int    GetUserCount() NN_NOEXCEPT
    {
        return m_UserCount;
    }

    void   OpenForDev(int busIdx, SpeedMode speedMode) NN_NOEXCEPT;

private:
    void SetVirtualAddress() NN_NOEXCEPT;

private:

    int                         m_BusIdx;
    nn::os::InterruptEventType  m_InterruptEvent;          // 割り込み用 (TODO: 割り込み対応は仮実装状態なのでまともな実装にする)

    uint32_t   m_BusFreeWaitMicroSeconds;  // バスの設定パラメータ群
    SpeedMode  m_SpeedMode;
    uint32_t   m_TimeoutMicroSeconds;
    uint32_t   m_CheckRetryTimes;

    int        m_UserCount;                  // バスのユーザー数

    uintptr_t  m_I2cBaseAddress;

    static const uint32_t DefaultTimeoutCycles           = 10;
    static const uint32_t DefaultBusFreeWaitMicroSeconds = 130;
    static const uint32_t DefaultSpeedGradeHz            = 400 * 1000;
    static const uint32_t DefaultTimeoutMicroSeconds     = DefaultTimeoutCycles * 1000 * 1000 / SpeedMode_Fast; // 10[cycle] / Frequency[1/s]. 10^3 は単位の補正項
    static const uint32_t DefaultCheckRetryTimes         = 10;

};

} // detail
} // driver
} // i2c
} // nn
