﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Common.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>

#include <nn/os/os_InterruptEvent.h>

#include <nn/os.h>

#include <nn/i2c/driver/i2c.h>

// ターゲットボートごとに分岐したヘッダ
#include "i2c_BusAccessor-hardware.bdsl-imx6.h"
#include "i2c_Register-hardware.bdsl-imx6.h"

#include "i2c_Util.h"
#include "i2c_DdUtil.h"

namespace {

uint8_t ComputeDivider(nn::i2c::SpeedMode speedMode) NN_NOEXCEPT
{
    // 桁あふれが起きないよう十分長いbit長にしておく。型はいずれもuint64_t
    uint64_t div = nn::i2c::driver::detail::BaseClock / speedMode;

    for (int i = 0; i < nn::i2c::driver::detail::DivFactorsLength; i++ )
    {
        if ( nn::i2c::driver::detail::DivFactors[i].div >= div )
        {
            return nn::i2c::driver::detail::DivFactors[i].ifdr;
        }
    }
    return nn::i2c::driver::detail::DivFactors[nn::i2c::driver::detail::DivFactorsLength - 1].ifdr;
}

} // namespace

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

void BusAccessor::Open(int busIdx, SpeedMode speedMode) NN_NOEXCEPT
{
    m_BusIdx    = busIdx;
    m_SpeedMode = speedMode;

    NN_SDK_ASSERT( m_UserCount < INT_MAX ); // 値域チェック
    // TORIAEZU : カウントだけはしておく
    m_UserCount++;

    // 仮想アドレスをセットする。
    SetVirtualAddress();

    NN_DETAIL_I2C_IFDR( m_BusIdx, m_I2cBaseAddress ) = NN_DETAIL_I2C_IFDR_IC(ComputeDivider(m_SpeedMode));
    NN_DETAIL_I2C_I2SR( m_BusIdx, m_I2cBaseAddress ) = 0;
    NN_DETAIL_I2C_I2CR( m_BusIdx, m_I2cBaseAddress ) = I2cControlRegister_Ien; // ref. [ 35.7.3 I2C Control Register @ IEN ] This bit must be set before any other I2C_I2CR bits have an effect.

    nn::os::InitializeInterruptEvent( &m_InterruptEvent, NN_DETAIL_I2C_IRQ(m_BusIdx),  nn::os::EventClearMode_ManualClear);
    nn::os::ClearInterruptEvent(&m_InterruptEvent);
}

void BusAccessor::ExecuteInitialConfig() NN_NOEXCEPT
{
    // 該当処理なし
}

void BusAccessor::Close() NN_NOEXCEPT
{
    NN_DETAIL_I2C_I2CR( m_BusIdx, m_I2cBaseAddress ) &= ~I2cControlRegister_Ien;
    m_UserCount--;
}

bool BusAccessor::CheckBusy() const NN_NOEXCEPT
{
    uint32_t BusWaitIntervalMicroSeconds = m_TimeoutMicroSeconds / m_CheckRetryTimes;
    NN_SDK_ASSERT( BusWaitIntervalMicroSeconds > 0 );

    for ( int i = 0; i < m_CheckRetryTimes; i++ )
    {
        if( !( NN_DETAIL_I2C_I2SR(m_BusIdx, m_I2cBaseAddress) & I2cStatusRegister_Ibb ) )
        {
            return false;
        }
        NN_DETAIL_I2C_LOG("%s(%d): Bus Busy. retries=%d/%d\n", __FUNCTION__, __LINE__, i + 1, m_CheckRetryTimes );

        if( i == m_CheckRetryTimes - 1 )
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(BusWaitIntervalMicroSeconds));
    }
    return true; // Bus が Busy 状態
}

void BusAccessor::StartCondition() const NN_NOEXCEPT
{
    NN_DETAIL_I2C_I2CR(m_BusIdx, m_I2cBaseAddress) = I2cControlRegister_Ien | I2cControlRegister_Mtx | I2cControlRegister_Txak | I2cControlRegister_Iien;
    NN_DETAIL_I2C_I2CR(m_BusIdx, m_I2cBaseAddress) |= I2cControlRegister_Msta;
}

void BusAccessor::StopCondition() const NN_NOEXCEPT
{
    NN_DETAIL_I2C_I2CR(m_BusIdx, m_I2cBaseAddress) &= ~I2cControlRegister_Msta;
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(m_BusFreeWaitMicroSeconds));
}

Result BusAccessor::StartTransaction( Bit16 slaveAddress, AddressingMode addressingMode, Command command ) NN_NOEXCEPT
{
    NN_UNUSED(addressingMode);

    // 7ビットアドレスモードのみのサポートを想定。10bitアドレスモードは現状非サポート
    if( addressingMode != AddressingMode_BitWidth7 )
    {
        NN_ABORT("The addressing mode is not supported.\n");
    }

    Bit8 isReceive = ( command == Command_Receive ) ? 0x01 : 0x00; // slave address の末尾に付ける R/W bit
    Bit8 startByte = slaveAddress << 1 | isReceive;

    // TORIAEZU : TransactionOption, Slave address, AddressingMode はインタフェースを揃えるためのダミー引数
    return Send(&startByte, 1, static_cast<nn::i2c::TransactionOption>(0), 0, AddressingMode::AddressingMode_BitWidth7 );
}

Result BusAccessor::Send( const Bit8* pInData,  size_t dataBytes, TransactionOption inOption, Bit16 slaveAddress, AddressingMode addressingMode ) NN_NOEXCEPT
{
    NN_DETAIL_I2C_I2SR(m_BusIdx, m_I2cBaseAddress) &= ~I2cStatusRegister_Iif;

    nn::os::ClearInterruptEvent(&m_InterruptEvent);

    for (int i = 0; i < dataBytes; i++)
    {
        NN_DETAIL_I2C_I2DR(m_BusIdx, m_I2cBaseAddress) = NN_DETAIL_I2C_I2DR_DATA(pInData[i]);
        nn::os::WaitInterruptEvent(&m_InterruptEvent);
        uint16_t sts = NN_DETAIL_I2C_I2SR(m_BusIdx, m_I2cBaseAddress);
        NN_DETAIL_I2C_I2SR(m_BusIdx, m_I2cBaseAddress) &= ~I2cStatusRegister_Iif;
        nn::os::ClearInterruptEvent(&m_InterruptEvent);

        if (sts & I2cStatusRegister_Ial)
        {
            // Arbitration Lost 発生時は IAL をクリア
            // ref. p1877 "35.5.7 Arbitration Lost"
            NN_DETAIL_I2C_I2SR(m_BusIdx, m_I2cBaseAddress) &= ~I2cStatusRegister_Ial;
            NN_DETAIL_I2C_LOG("%s(%d): Arbitration Lost\n", __FUNCTION__, __LINE__ );
            return nn::i2c::ResultBusBusy(); // ハンドリング方法は Bus busy と同等であるためエラーコードは ResultBusBusy() とする。
        }
        else if (sts & I2cStatusRegister_Rxak)
        {
            NN_DETAIL_I2C_LOG("%s(%d): No Ack\n", __FUNCTION__, __LINE__ );
            return nn::i2c::ResultNoAck();
        }
        else if (sts & I2cStatusRegister_Icf)
        {
        }
        else
        {
            NN_ABORT("Unknown Error");
        }
    }

    return ResultSuccess();
}

Result BusAccessor::Receive( Bit8 *pOutData, size_t dataBytes, TransactionOption inOption, Bit16 slaveAddress, AddressingMode addressingMode ) NN_NOEXCEPT
{
    // Set TXAK=1 when 2nd Last Byte is to be read (ref. p.1905 Figure35-5 Flowchart of typical I2C interrrupt routine)
    NN_DETAIL_I2C_I2CR(m_BusIdx, m_I2cBaseAddress) &= ~(I2cControlRegister_Mtx | (dataBytes == 1 ? 0 : I2cControlRegister_Txak));
    NN_DETAIL_I2C_I2SR(m_BusIdx, m_I2cBaseAddress) &= ~I2cStatusRegister_Iif;

    nn::os::ClearInterruptEvent(&m_InterruptEvent);


    pOutData[0] = NN_DETAIL_I2C_I2DR(m_BusIdx, m_I2cBaseAddress);

    for (int i = 0; i < dataBytes; i++)
    {
        nn::os::WaitInterruptEvent(&m_InterruptEvent);
        uint16_t sts = NN_DETAIL_I2C_I2SR(m_BusIdx, m_I2cBaseAddress);
        NN_DETAIL_I2C_I2SR(m_BusIdx, m_I2cBaseAddress) &= ~I2cStatusRegister_Iif;

        nn::os::ClearInterruptEvent(&m_InterruptEvent);

        // TODO: まともなエラーハンドリング
        if (sts & I2cStatusRegister_Ial)
        {
            // Arbitration Lost 発生時は IAL をクリア
            // ref. p1877 "35.5.7 Arbitration Lost"
            NN_DETAIL_I2C_I2SR(m_BusIdx, m_I2cBaseAddress) &= ~I2cStatusRegister_Ial;
            NN_DETAIL_I2C_LOG("%s(%d): Arbitration Loss\n", __FUNCTION__, __LINE__ );
            return nn::i2c::ResultBusBusy();
        }
        else if (sts & I2cStatusRegister_Icf)
        {
        }
        else
        {
            NN_ABORT("Unknown Error");
        }

        if (i == dataBytes - 1)
        {
            NN_DETAIL_I2C_I2CR(m_BusIdx, m_I2cBaseAddress) &= ~I2cControlRegister_Msta;
        }
        else if (i == dataBytes - 2)
        {
            NN_DETAIL_I2C_I2CR(m_BusIdx, m_I2cBaseAddress) |=  I2cControlRegister_Txak;
        }

        pOutData[i] = NN_DETAIL_I2C_I2DR_DATA(NN_DETAIL_I2C_I2DR(m_BusIdx, m_I2cBaseAddress));

    }

    return ResultSuccess();
}

void BusAccessor::SetVirtualAddress() NN_NOEXCEPT
{
    // IO マッピング関連
    // I2C
    const nn::dd::PhysicalAddress I2cPhysicalAddress  = 0x0021A0000ull;
    const size_t                  I2cAddressSize      = 0x9000;

    m_I2cBaseAddress = GetVirtualAddress(I2cPhysicalAddress, I2cAddressSize);
}

void BusAccessor::Suspend() NN_NOEXCEPT
{

}

void BusAccessor::Resume() NN_NOEXCEPT
{

}


} // detail
} // driver
} // i2c
} // nn
