﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/http.h>
#include <nn/http/http_JsonResponse.h>

#include <nn/nn_SdkLog.h>
#include <nn/util/util_StringUtil.h>
#include <curl/curl.h>

#include <cctype>
#include <cstdlib>
#include <mutex>

#include "http_Utility.h"

using nn::msgpack::JsonStreamParser;
using nn::msgpack::JsonStreamParserSettings;

namespace nn {
namespace http {

class InputStreamAdaptor : public msgpack::InputStream{
public:
    InputStreamAdaptor(Response& res, void* pBuffer, size_t size) NN_NOEXCEPT
        : m_Response(res)
    {
        InputStream::ResetBuffer(pBuffer, size);
    }

private:
    virtual size_t FillBuffer_(void* p, size_t nbytes) NN_NOEXCEPT
    {
        size_t sizeFilled;
        Result result = m_Response.ReadBody(&sizeFilled, p, nbytes);
        if (result.IsFailure())
        {
            SetError(EIO);
            return 0;
        }
        return sizeFilled;
    }
    virtual bool Close_() NN_NOEXCEPT
    {
        return true;
    }

    Response& m_Response;
};

JsonResponse::JsonResponse(void* pWorkBuffer, size_t sizeWorkBuffer)
    : m_pWorkBuffer(pWorkBuffer)
    , m_sizeWorkBuffer(sizeWorkBuffer)
{
}

JsonResponse::~JsonResponse()
{
}

Result JsonResponse::ReadAndParse(TokenHandler tokenHandler)
{
    Result result;
    InputStreamAdaptor istream(*this, m_pWorkBuffer, m_sizeWorkBuffer);
    {
        JsonStreamParser parser;
        JsonStreamParserSettings setting;
        setting.format = JsonStreamParserSettings::kFormatJson;
        setting.max_array_size    = 128;
        setting.max_map_size      = 64;
        setting.max_depth         = 4;
        setting.token_buffer_size = 1024;

        msgpack::ErrnoT err;
        err = parser.Init(setting);
        if (msgpack::IsError(err))
        {
            return ResultOutOfMemory();
        }

        err = parser.Open(&istream);
        if (msgpack::IsError(err))
        {
            return ResultOutOfMemory();
        }

        JsonStreamParser::Event ev;
        if ((ev = parser.Next()) != JsonStreamParser::kEventStartMap)
        {
            NN_HTTP_ERROR("JSON response is not started with '{'.\n");
            return ResultJsonInvalidSyntax();
        }

        while ((ev = parser.Next()) != JsonStreamParser::kEventEndDocument)
        {
            if (msgpack::IsError(parser.GetError()))
            {
                NN_HTTP_ERROR("JSON response contains invalid field.\n");
                return ResultJsonInvalidField();
            }
            if (!tokenHandler(parser, ev))
            {
                NN_HTTP_ERROR("JSON response contains invalid field.\n");
                return ResultJsonInvalidSyntax();
            }
            if (IsCancelRequested())
            {
                return ResultCancelled();
            }
        }

        WaitCompleted();

        // エラーが発生していないか確認
        if (m_CurlCode != CURLE_OK)
        {
            NN_HTTP_ERROR("Parsing response is aborted due to %s.\n", curl_easy_strerror(m_CurlCode));
            return ConvertCurlCodeToResult(m_CurlCode);
        }
    }
    return ResultSuccess();
}

Result JsonResponse::ReadAndParseWithRule(const ParseRule pRulesArray[])
{
    return ReadAndParse([&pRulesArray](msgpack::JsonStreamParser& parser, msgpack::JsonStreamParser::Event ev) -> bool {
        if (ev != JsonStreamParser::kEventKeyName)
        {
            return true;
        }

        char buffer[24];
        const char* pKeyName = parser.GetToken().buf;

        for (int i = 0; pRulesArray[i].pKeyName != nullptr; ++i)
        {
            const ParseRule& rule = pRulesArray[i];
            NN_SDK_ASSERT(rule.pParentKeyName == nullptr, "Parent key is not implemented.");
            if (std::strcmp(pKeyName, rule.pKeyName) == 0)
            {
                ev = parser.Next();
                const msgpack::JsonStreamParser::Token& token = parser.GetToken();
                switch (rule.expectedValueType)
                {
                case ValueType_Int64Force:
                    if (ev == JsonStreamParser::kEventString)
                    {
                        util::Strlcpy(buffer, token.buf, sizeof(buffer));
                        *static_cast<int64_t*>(rule.pStorage) = std::strtoll(buffer, nullptr, 10);
                        //TODO: check errno
                        break;
                    }
                    // no break;

                case ValueType_Int64:
                    if (!msgpack::IsSuccess(JsonStreamParser::ToInt64(token, static_cast<int64_t*>(rule.pStorage))))
                    {
                        return false;
                    }
                    break;

                case ValueType_Uint64:
                    if (!msgpack::IsSuccess(JsonStreamParser::ToUint64(token, static_cast<uint64_t*>(rule.pStorage))))
                    {
                        return false;
                    }
                    break;

                case ValueType_String:
                    {
                        if (ev != JsonStreamParser::kEventString)
                        {
                            return false;
                        }

                        size_t written = util::Strlcpy(static_cast<char*>(rule.pStorage), token.buf, static_cast<int>(rule.sizeStorage));
                        if (written >= rule.sizeStorage)
                        {
                            return false;
                        }
                    }
                    break;

                default:
                    NN_SDK_ASSERT(false, "Found invalid value type in rules");
                }
            }
        }
        return true;
    });
}


}
} // ~namespace nn::http
