﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include "htclow_PacketFactory.h"

namespace nn { namespace htclow { namespace server {

void SendPacketFactory::InitializeHeaderCommon(PacketHeader* pOutHeader) NN_NOEXCEPT
{
    pOutHeader->protocol = ProtocolId;
    pOutHeader->version = GetVersion();
    pOutHeader->reserved = 0;
}

std::unique_ptr<SendPacket> SendPacketFactory::MakeSynPacket(detail::ChannelInternalType channel) NN_NOEXCEPT
{
    auto p = std::make_unique<SendPacket>();

    if (!p || !p->IsAllocationSucceeded())
    {
        return nullptr;
    }

    InitializeHeaderCommon(p->GetHeader());
    p->GetHeader()->channel = channel;
    p->GetHeader()->packetType = PacketType::Syn;
    p->GetHeader()->bodySize = 0;
    p->GetHeader()->sequenceId = 0L;

    NN_SDK_ASSERT_EQUAL(p->GetBodySize(), p->GetHeader()->bodySize);
    return p;
}

std::unique_ptr<SendPacket> SendPacketFactory::MakeDataPacket(detail::ChannelInternalType channel, const void* body, int bodySize) NN_NOEXCEPT
{
    auto p = std::make_unique<SendPacket>(static_cast<int>(sizeof(PacketHeader)) + bodySize);

    if (!p || !p->IsAllocationSucceeded())
    {
        return nullptr;
    }

    InitializeHeaderCommon(p->GetHeader());
    p->GetHeader()->channel = channel;
    p->GetHeader()->packetType = PacketType::Data;
    p->GetHeader()->bodySize = bodySize;
    p->GetHeader()->sequenceId = 0L;

    p->CopyBody(body, bodySize);

    NN_SDK_ASSERT_EQUAL(p->GetBodySize(), p->GetHeader()->bodySize);
    return p;
}

std::unique_ptr<SendPacket> SendPacketFactory::MakeDataAckPacket(detail::ChannelInternalType channel, int64_t sequenceId) NN_NOEXCEPT
{
    auto p = std::make_unique<SendPacket>();

    if (!p || !p->IsAllocationSucceeded())
    {
        return nullptr;
    }

    InitializeHeaderCommon(p->GetHeader());
    p->GetHeader()->channel = channel;
    p->GetHeader()->packetType = PacketType::DataAck;
    p->GetHeader()->bodySize = 0;
    p->GetHeader()->sequenceId = sequenceId;

    NN_SDK_ASSERT_EQUAL(p->GetBodySize(), p->GetHeader()->bodySize);
    return p;
}

std::unique_ptr<SendPacket> SendPacketFactory::MakeFinPacket(detail::ChannelInternalType channel) NN_NOEXCEPT
{
    auto p = std::make_unique<SendPacket>();

    if (!p || !p->IsAllocationSucceeded())
    {
        return nullptr;
    }

    InitializeHeaderCommon(p->GetHeader());
    p->GetHeader()->channel = channel;
    p->GetHeader()->packetType = PacketType::Fin;
    p->GetHeader()->bodySize = 0;
    p->GetHeader()->sequenceId = 0L;

    NN_SDK_ASSERT_EQUAL(p->GetBodySize(), p->GetHeader()->bodySize);
    return p;
}

std::unique_ptr<SendPacket> SendPacketFactory::MakeFinAckPacket(detail::ChannelInternalType channel, int64_t sequenceId) NN_NOEXCEPT
{
    auto p = std::make_unique<SendPacket>();

    if (!p || !p->IsAllocationSucceeded())
    {
        return nullptr;
    }

    InitializeHeaderCommon(p->GetHeader());
    p->GetHeader()->channel = channel;
    p->GetHeader()->packetType = PacketType::FinAck;
    p->GetHeader()->bodySize = 0;
    p->GetHeader()->sequenceId = sequenceId;

    NN_SDK_ASSERT_EQUAL(p->GetBodySize(), p->GetHeader()->bodySize);
    return p;
}

std::unique_ptr<ReceivePacket> ReceivePacketFactory::MakePacket(const PacketHeader* pHeader) NN_NOEXCEPT
{
    auto p = std::make_unique<ReceivePacket>(static_cast<int>(sizeof(PacketHeader)) + pHeader->bodySize);

    if (!p || !p->IsAllocationSucceeded())
    {
        return nullptr;
    }

    p->CopyHeader(pHeader);

    return p;
}

}}}
