﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>
#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/htclow/htclow_ResultPrivate.h>

#include "htclow_DebugDriver.h"

namespace nn { namespace htclow { namespace server { namespace driver {

Result DebugDriver::Open() NN_NOEXCEPT
{
    m_ReceiveBufferHead = 0;
    m_ReceiveBufferTail = 0;
    m_SendBufferHead = 0;
    m_SendBufferTail = 0;

    m_ReceiveBufferUpdateEvent.Clear();
    m_SendBufferUpdateEvent.Clear();
    m_CancelEvent.Clear();

    // DebugDriver は Windows 環境でしか使わないので、malloc でメモリ確保して問題ない
    m_pReceiveBuffer = reinterpret_cast<uint8_t*>(std::malloc(BufferSize));
    m_pSendBuffer = reinterpret_cast<uint8_t*>(std::malloc(BufferSize));

    if (m_pReceiveBuffer == nullptr || m_pSendBuffer == nullptr)
    {
        return ResultDebugDriverError();
    }

    NN_RESULT_SUCCESS;
}

void DebugDriver::Close() NN_NOEXCEPT
{
    std::free(m_pReceiveBuffer);
    std::free(m_pSendBuffer);
    Cancel();
}

Result DebugDriver::Connect() NN_NOEXCEPT
{
    if (m_CancelEvent.TryWait())
    {
        return ResultCancelled();
    }
    NN_RESULT_SUCCESS;
}

void DebugDriver::Shutdown() NN_NOEXCEPT
{
}

Result DebugDriver::Send(const void* pBuffer, int bufferSize) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_SendBufferMutex)> lock(m_SendBufferMutex);

    if (m_SendBufferTail + bufferSize > BufferSize)
    {
        return ResultDebugDriverError();
    }

    std::memcpy(m_pSendBuffer + m_SendBufferTail, pBuffer, bufferSize);
    m_SendBufferTail += bufferSize;

    m_SendBufferUpdateEvent.Signal();
    NN_RESULT_SUCCESS;
}

Result DebugDriver::Receive(void* pOutBuffer, int bufferSize) NN_NOEXCEPT
{
    for (;;)
    {
        m_ReceiveBufferUpdateEvent.Clear();

        {
            std::lock_guard<decltype(m_ReceiveBufferMutex)> lock(m_ReceiveBufferMutex);

            if (m_ReceiveBufferTail - m_ReceiveBufferHead >= bufferSize)
            {
                std::memcpy(pOutBuffer, m_pReceiveBuffer + m_ReceiveBufferHead, bufferSize);
                m_ReceiveBufferHead += bufferSize;

                NN_RESULT_SUCCESS;
            }
        }

        const auto trigger = nn::os::WaitAny(m_ReceiveBufferUpdateEvent.GetBase(), m_CancelEvent.GetBase());
        if (trigger == 1)
        {
            return ResultCancelled();
        }
    }
}

void DebugDriver::Cancel() NN_NOEXCEPT
{
    m_CancelEvent.Signal();
}

Result DebugDriver::ReadFromSendBuffer(void* pOutBuffer, int readSize) NN_NOEXCEPT
{
    for (;;)
    {
        m_SendBufferUpdateEvent.Clear();

        {
            std::lock_guard<decltype(m_SendBufferMutex)> lock(m_SendBufferMutex);

            if (m_SendBufferTail - m_SendBufferHead >= readSize)
            {
                std::memcpy(pOutBuffer, m_pSendBuffer + m_SendBufferHead, readSize);
                m_SendBufferHead += readSize;

                NN_RESULT_SUCCESS;
            }
        }

        const auto trigger = nn::os::WaitAny(m_SendBufferUpdateEvent.GetBase(), m_CancelEvent.GetBase());
        if (trigger == 1)
        {
            return ResultCancelled();
        }
    }
}

Result DebugDriver::WriteToReceiveBuffer(const void* pBuffer, int writeSize) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_ReceiveBufferMutex)> lock(m_ReceiveBufferMutex);

    if (m_ReceiveBufferTail + writeSize > BufferSize)
    {
        return ResultDebugDriverError();
    }

    std::memcpy(m_pReceiveBuffer + m_ReceiveBufferTail, pBuffer, writeSize);
    m_ReceiveBufferTail += writeSize;

    m_ReceiveBufferUpdateEvent.Signal();

    NN_RESULT_SUCCESS;
}

}}}}
