﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/htcfs/htcfs_Result.h>
#include <nn/htcfs/htcfs_ResultPrivate.h>

#include "htcfs_DirectoryServiceObject.h"
#include "htcfs_FileServiceObject.h"
#include "htcfs_FileSystemServiceObject.h"
#include "htcfs_Impl.h"

namespace nn { namespace htcfs { namespace server {

namespace {

// TORIAEZU: tma::file_io と同じサイズ
const int ServiceObjectHeapSize = 4096 * 8;

struct FileServiceObjectAllocatorTag;
typedef nn::sf::ExpHeapStaticAllocator<ServiceObjectHeapSize, FileServiceObjectAllocatorTag> FileServiceObjectAllocator;

class FileServiceObjectAllocatorInitializer
{
public:
    FileServiceObjectAllocatorInitializer() NN_NOEXCEPT
    {
        FileServiceObjectAllocator::Initialize(nn::lmem::CreationOption_ThreadSafe);
    }
} g_FileServiceObjectAllocatorInitializer;

struct DirectoryServiceObjectAllocatorTag;
typedef nn::sf::ExpHeapStaticAllocator<ServiceObjectHeapSize, DirectoryServiceObjectAllocatorTag> DirectoryServiceObjectAllocator;

class DirectoryServiceObjectAllocatorInitializer
{
public:
    DirectoryServiceObjectAllocatorInitializer() NN_NOEXCEPT
    {
        DirectoryServiceObjectAllocator::Initialize(nn::lmem::CreationOption_ThreadSafe);
    }
} g_DirectoryServiceObjectAllocatorInitializer;

// path が1文字以上のヌル終端文字列であることを検査する
bool IsValidPath(const nn::htcfs::Path& path)
{
    const auto len = strnlen(path.str, sizeof(path.str));
    return 0 < len && len < sizeof(path.str);
}

}

nn::Result FileSystemServiceObject::GetEntryType(nn::sf::Out<std::int32_t> outType, const nn::htcfs::Path& path) NN_NOEXCEPT
{
    if (!IsValidPath(path))
    {
        return ResultInvalidArgument();
    }

    nn::fs::DirectoryEntryType type;
    NN_RESULT_DO( HtcfsImpl::GetInstance()->GetEntryType(&type, path.str));
    *outType = type;
    NN_RESULT_SUCCESS;
}

nn::Result FileSystemServiceObject::OpenFile(nn::sf::Out<nn::sf::SharedPointer<nn::htcfs::IFile>> outHandle, const nn::htcfs::Path& path, std::uint32_t mode) NN_NOEXCEPT
{
    if (!IsValidPath(path))
    {
        return ResultInvalidArgument();
    }

    int32_t fileHandle;
    NN_RESULT_DO(HtcfsImpl::GetInstance()->OpenFile(&fileHandle, path.str, static_cast<nn::fs::OpenMode>(mode)));

    auto p = nn::sf::ObjectFactory<FileServiceObjectAllocator::Policy>::CreateSharedEmplaced<
        IFile,
        FileServiceObject
    >(fileHandle);

    outHandle.Set(std::move(p));
    NN_RESULT_SUCCESS;
}

nn::Result FileSystemServiceObject::CreateFile(const nn::htcfs::Path& path, std::int64_t size) NN_NOEXCEPT
{
    if (!IsValidPath(path) || size < 0)
    {
        return ResultInvalidArgument();
    }

    return HtcfsImpl::GetInstance()->CreateFile(path.str, size);
}

nn::Result FileSystemServiceObject::DeleteFile(const nn::htcfs::Path& path) NN_NOEXCEPT
{
    if (!IsValidPath(path))
    {
        return ResultInvalidArgument();
    }

    return HtcfsImpl::GetInstance()->DeleteFile(path.str);
}

nn::Result FileSystemServiceObject::RenameFile(const nn::htcfs::Path& fromName, const nn::htcfs::Path& toName) NN_NOEXCEPT
{
    if (!IsValidPath(fromName) || !IsValidPath(toName))
    {
        return ResultInvalidArgument();
    }

    return HtcfsImpl::GetInstance()->RenameFile(fromName.str, toName.str);
}

nn::Result FileSystemServiceObject::FileExists(nn::sf::Out<bool> outExists, const nn::htcfs::Path& path) NN_NOEXCEPT
{
    if (!IsValidPath(path))
    {
        return ResultInvalidArgument();
    }

    bool exists;
    NN_RESULT_DO(HtcfsImpl::GetInstance()->FileExists(&exists, path.str));
    *outExists = exists;
    NN_RESULT_SUCCESS;
}

nn::Result FileSystemServiceObject::GetFileTimeStamp(nn::sf::Out<std::uint64_t> outCreateTime, nn::sf::Out<std::uint64_t> outAccessTime, nn::sf::Out<std::uint64_t> outModifyTime, const nn::htcfs::Path& path) NN_NOEXCEPT
{
    if (!IsValidPath(path))
    {
        return ResultInvalidArgument();
    }

    uint64_t createTime;
    uint64_t accessTime;
    uint64_t modifyTime;

    auto result = HtcfsImpl::GetInstance()->GetFileTimeStamp(&createTime, &accessTime, &modifyTime, path.str);

    *outCreateTime = createTime;
    *outAccessTime = accessTime;
    *outModifyTime = modifyTime;

    return result;
}

nn::Result FileSystemServiceObject::OpenDirectory(nn::sf::Out<nn::sf::SharedPointer<nn::htcfs::IDirectory>> outHandle, const nn::htcfs::Path& path, std::int32_t mode) NN_NOEXCEPT
{
    if (!IsValidPath(path))
    {
        return ResultInvalidArgument();
    }

    int32_t directoryHandle;
    NN_RESULT_DO(HtcfsImpl::GetInstance()->OpenDirectory(&directoryHandle, path.str, static_cast<nn::fs::OpenDirectoryMode>(mode)));

    auto p = nn::sf::ObjectFactory<DirectoryServiceObjectAllocator::Policy>::CreateSharedEmplaced<
        IDirectory,
        DirectoryServiceObject
    >(directoryHandle);

    outHandle.Set(std::move(p));
    NN_RESULT_SUCCESS;
}

nn::Result FileSystemServiceObject::CreateDirectory(const nn::htcfs::Path& path) NN_NOEXCEPT
{
    if (!IsValidPath(path))
    {
        return ResultInvalidArgument();
    }

    return HtcfsImpl::GetInstance()->CreateDirectory(path.str);
}

nn::Result FileSystemServiceObject::DeleteDirectory(const nn::htcfs::Path& path, bool recursively) NN_NOEXCEPT
{
    if (!IsValidPath(path))
    {
        return ResultInvalidArgument();
    }

    return HtcfsImpl::GetInstance()->DeleteDirectory(path.str, recursively);
}

nn::Result FileSystemServiceObject::DirectoryExists(nn::sf::Out<bool> outExists, const nn::htcfs::Path& path) NN_NOEXCEPT
{
    if (!IsValidPath(path))
    {
        return ResultInvalidArgument();
    }

    bool exists;
    NN_RESULT_DO(HtcfsImpl::GetInstance()->DirectoryExists(&exists, path.str));
    *outExists = exists;
    NN_RESULT_SUCCESS;
}

nn::Result FileSystemServiceObject::RenameDirectory(const nn::htcfs::Path& fromName, const nn::htcfs::Path& toName) NN_NOEXCEPT
{
    if (!IsValidPath(fromName) || !IsValidPath(toName))
    {
        return ResultInvalidArgument();
    }

    return HtcfsImpl::GetInstance()->RenameDirectory(fromName.str, toName.str);
}

}}}
