﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/os/os_EventApi.h>
#include <nn/os/os_SystemEventApi.h>
#include <nn/hdcp/detail/hdcp_Log.h>
#include "hdcp_HdcpCore.h"
#include "hdcp_NvHdcp.h"
#include "hdcp_NvHdcpStub.h"

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

namespace nn { namespace hdcp { namespace impl { namespace detail {

namespace {
const int64_t DefaultTimeoutMsec = 12 * 1000;
}

HdcpCore::HdcpCore() NN_NOEXCEPT
:
m_CurrentNvState(HDCP_STATE_OFF),
m_TimeoutMsec(DefaultTimeoutMsec),
m_StateTransitionEvent(nn::os::EventClearMode_AutoClear, true),
m_pNvHdcp(&NvHdcp::GetInstance())
{
    // デバッグ設定。設定がなければデフォルト値が使われる。
    {
        // スタブでエミュレーションするか。
        bool isStubUsed = false;
        auto typeSize = sizeof(isStubUsed);
        auto keySize = nn::settings::fwdbg::GetSettingsItemValue(&isStubUsed, typeSize, "hdcp", "stub_emulation");
        isStubUsed &= keySize == typeSize;
        if (isStubUsed)
        {
            m_pNvHdcp = &NvHdcpStub::GetInstance();
        }
    }
    {
        // タイムアウト時間の変更。
        int32_t timeoutMsec = DefaultTimeoutMsec;
        auto typeSize = sizeof(timeoutMsec);
        auto keySize = nn::settings::fwdbg::GetSettingsItemValue(&timeoutMsec, typeSize, "hdcp", "authentication_timeout_ms");
        if (keySize == typeSize)
        {
            m_TimeoutMsec = static_cast<int64_t>(timeoutMsec);
        }
    }
}

Result HdcpCore::Open() NN_NOEXCEPT
{
    auto error = m_pNvHdcp->Open();
    if (HDCP_RET_NO_MEMORY == error)
    {
        NN_DETAIL_HDCP_ERROR("Error code(%d) returned from Open().\n", error);
        return nn::hdcp::ResultOutOfMemory();
    }
    else if (HDCP_RET_SUCCESS != error)
    {
        NN_DETAIL_HDCP_ERROR("Error code(%d) returned from Open().\n", error);
        return nn::hdcp::ResultOperationFailed();
    }

    nn::os::InitializeEvent(&m_TimeoutEvent, false, nn::os::EventClearMode_AutoClear);
    auto f = [](void* context)
    {
        auto hdcpCore = reinterpret_cast<HdcpCore*>(context);
        while (hdcpCore->m_IsTimeoutThreadRunning)
        {
            if (HDCP_STATE_UNAUTHENTICATED == hdcpCore->m_CurrentNvState && !hdcpCore->m_IsAuthenticationTimeout)
            {
                if (!nn::os::TimedWaitEvent(&hdcpCore->m_TimeoutEvent, nn::TimeSpan::FromMilliSeconds(hdcpCore->m_TimeoutMsec)))
                {
                    // 認証タイムアウトをシグナルする。
                    hdcpCore->m_IsAuthenticationTimeout = true;
                    hdcpCore->m_StateTransitionEvent.Signal();
                }
            }
            else
            {
                nn::os::WaitEvent(&hdcpCore->m_TimeoutEvent);
            }
        }
    };
    m_IsTimeoutThreadRunning = true;
    m_IsAuthenticationTimeout = false;
    static NN_ALIGNAS(4096) char s_Stack[8 * 1024];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_TimeoutThread, f, this, s_Stack, sizeof(s_Stack), NN_SYSTEM_THREAD_PRIORITY(hdcp, TimeoutHandler)));
    nn::os::SetThreadNamePointer(&m_TimeoutThread, NN_SYSTEM_THREAD_NAME(hdcp, TimeoutHandler));
    nn::os::StartThread(&m_TimeoutThread);
    return nn::ResultSuccess();
}

void HdcpCore::Close() NN_NOEXCEPT
{
    m_IsTimeoutThreadRunning = false;
    nn::os::SignalEvent(&m_TimeoutEvent);
    nn::os::DestroyThread(&m_TimeoutThread);
    nn::os::ClearEvent(&m_TimeoutEvent);
    nn::os::FinalizeEvent(&m_TimeoutEvent);
    m_IsAuthenticationTimeout = false;

    // nvhdcp のコードを見る限り、ここでエラーが返ることはまずない。
    auto error = m_pNvHdcp->Close();
    if (HDCP_RET_SUCCESS != error)
    {
        NN_DETAIL_HDCP_ERROR("Error code(%d) returned from Close().\n", error);
    }
    NN_SDK_ASSERT(HDCP_RET_SUCCESS == error);
}

Result HdcpCore::GetCurrentState(HdcpAuthenticationState* state) NN_NOEXCEPT
{
    auto error = m_pNvHdcp->GetStatus();
    switch (error)
    {
    case HDCP_RET_SUCCESS:
        // 認証成功。
        *state = HdcpAuthenticationState_Authenticated;
        break;
    case HDCP_RET_NOT_PLUGGED:
        // 抜かれている。
    case HDCP_RET_READS_FAILED:
        // HDCP_RET_READS_FAILED は認証タイムアウト時には返らなくなり、解像度変更直後にのみ返りえるエラーとなった。
        // この status が返る状態になった直後、HDCP_STATE_UNAUTHENTICATED にすぐに遷移して認証が再開される。
        *state = HdcpAuthenticationState_NotPlugged;
        break;
    case HDCP_RET_DISABLED:
        // 設定が無効時。
        *state = HdcpAuthenticationState_Unauthenticated;
        break;
    case HDCP_RET_AUTH_IN_PROCESS:
        *state = m_IsAuthenticationTimeout ?
            HdcpAuthenticationState_Timeout :       // 認証タイムアウト。
            HdcpAuthenticationState_Processing;     // 認証中。
        break;
    //case HDCP_RET_NOT_SUPPORTED:
    //    // HDCP_CAPABLE bit は今は扱っていないので返らない。
    //    *state = HdcpAuthenticationState_UnsupportedDisplay;
    //    break;
    case HDCP_RET_READM_FAILED:
        // 上記は認証タイムアウトとしての役割がなくなったため、内部エラーとして扱う。
    case HDCP_RET_AUTH_FAULT:
        // 復帰できない内部エラー。
    default:
        NN_DETAIL_HDCP_ERROR("Unexpected error code(%d) returned from GetStatus().\n", error);
        return nn::hdcp::ResultOperationFailed();
    }
    return nn::ResultSuccess();
}

Result HdcpCore::SetHdcpConfig(HdcpMode mode) NN_NOEXCEPT
{
    auto error = m_pNvHdcp->Enable(HdcpMode::HdcpMode_Enabled == mode);
    if (HDCP_RET_SUCCESS != error)
    {
        NN_DETAIL_HDCP_ERROR("Error code(%d) returned from SetHdcpConfig().\n", error);
        return nn::hdcp::ResultOperationFailed();
    }
    return nn::ResultSuccess();
}

Result HdcpCore::GetHdcpStateTransitionEvent(nn::os::NativeHandle* pOutEventHandle) NN_NOEXCEPT
{
    auto f = [](HDCP_CLIENT_HANDLE handle, NvU32 state, void* context)
    {
        //NN_DETAIL_HDCP_TRACE("handle:[%p] state:[%d] context:[%p]\n", handle, state, context);
        NN_UNUSED(handle);
        auto hdcpCore = reinterpret_cast<HdcpCore*>(context);
        hdcpCore->m_CurrentNvState = state;
        hdcpCore->m_IsAuthenticationTimeout = false;
        // 認証タイムアウトのためにシグナル。
        nn::os::SignalEvent(&hdcpCore->m_TimeoutEvent);
        hdcpCore->m_StateTransitionEvent.Signal();
    };
    auto error = m_pNvHdcp->SetTransitHandler(f, this);
    if (HDCP_RET_SUCCESS != error)
    {
        NN_DETAIL_HDCP_ERROR("Error code(%d) returned from SetTransitHandler().\n", error);
        return nn::hdcp::ResultOperationFailed();
    }
    auto initError = m_pNvHdcp->InitializeAsyncEvent();
    if (HDCP_RET_NO_MEMORY == initError)
    {
        NN_DETAIL_HDCP_ERROR("Error code(%d) returned from InitializeAsyncEvent().\n", error);
        return nn::hdcp::ResultOutOfMemory();
    }
    else if (HDCP_RET_SUCCESS != initError)
    {
        NN_DETAIL_HDCP_ERROR("Error code(%d) returned from InitializeAsyncEvent().\n", error);
        return nn::hdcp::ResultOperationFailed();
    }
    *pOutEventHandle = m_StateTransitionEvent.GetReadableHandle();
    return nn::ResultSuccess();
}

}}}} // namespace nn::hdcp::impl::detail
