﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/



#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/gpio/driver/gpio.h>
#include <nn/gpio/driver/gpio_PadAccessorDev.h> // OpenSessionForDev

#include <nn/sf/impl/sf_ExpHeapAllocator.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/gpio/gpio_IManager.sfdl.h>
#include <nn/gpio/gpio_IPadSession.sfdl.h>
#include <nn/gpio/server/gpio_ManagerImpl.h>

namespace nn { namespace gpio { namespace server {

// IGpioPadSession 実装クラス
class PadSessionImpl
{
public:
    // コンストラクタで親の SharedPointer を設定し、値によって、OpenSession / ForDev を呼び分ける
    // [Gen1] TODO: Deprecate
    PadSessionImpl(ManagerImpl* pParent, int number, bool isForDev) NN_NOEXCEPT
        : m_Parent(pParent, true)
    {
        if(isForDev)
        {
            nn::gpio::driver::OpenSessionForDev(&m_InternalPadSession, number);
        }
        else
        {
            nn::gpio::driver::OpenSession(&m_InternalPadSession, static_cast<nn::gpio::GpioPadName>(number));
        }
    }

    // [Gen2]
    explicit PadSessionImpl(ManagerImpl* pParent) NN_NOEXCEPT
        : m_Parent(pParent, true)
    {
        // Gen2 OpenSession は Result を返すのでメソッドを別に実装し、ここでは呼ばない
    }

    // Session のデストラクタで Session のクローズを行う
    ~PadSessionImpl()
    {
        nn::gpio::driver::CloseSession(&m_InternalPadSession);
    }

    // GPIO ドライバライブラリの API のうち SystemEvent を使わないもの
    nn::Result OpenSession(nn::DeviceCode deviceCode) NN_NOEXCEPT;
    nn::Result SetDirection(std::int32_t direction) NN_NOEXCEPT;
    nn::Result SetValue(std::int32_t value) NN_NOEXCEPT;
    nn::Result SetValueForSleepState(std::int32_t value) NN_NOEXCEPT;
    nn::Result GetValue(nn::sf::Out<std::int32_t> outValue) NN_NOEXCEPT;
    nn::Result GetDirection(nn::sf::Out<std::int32_t> direction) NN_NOEXCEPT;
    nn::Result SetInterruptMode(std::int32_t mode) NN_NOEXCEPT;
    nn::Result GetInterruptMode(nn::sf::Out<std::int32_t> mode) NN_NOEXCEPT;
    nn::Result SetInterruptEnable(bool enable) NN_NOEXCEPT;
    nn::Result GetInterruptEnable(nn::sf::Out<bool> enable) NN_NOEXCEPT;
    nn::Result GetInterruptStatus(nn::sf::Out<std::int32_t> status) NN_NOEXCEPT;
    nn::Result ClearInterruptStatus() NN_NOEXCEPT;
    nn::Result BindInterrupt(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT;
    nn::Result UnbindInterrupt() NN_NOEXCEPT;
    nn::Result SetDebounceEnabled(bool isEnable) NN_NOEXCEPT;
    nn::Result GetDebounceEnabled(nn::sf::Out<bool> isEnable) NN_NOEXCEPT;
    nn::Result SetDebounceTime(int msecTime) NN_NOEXCEPT;
    nn::Result GetDebounceTime(nn::sf::Out<int> msecTime) NN_NOEXCEPT;

private:

    // 親への SharedPointer
    nn::sf::SharedPointer<ManagerImpl> m_Parent;

    // GPIO ライブラリが持つ GpioPadSession 構造体のメンバ変数
    nn::gpio::driver::GpioPadSession m_InternalPadSession;

    // パッドごとに紐付けられる SystemEvent
    nn::os::SystemEvent m_SystemEvent;
};

// ここから各 GPIO API の内部実装
nn::Result PadSessionImpl::OpenSession(nn::DeviceCode deviceCode) NN_NOEXCEPT
{
    return nn::gpio::driver::OpenSession(&m_InternalPadSession, deviceCode);
}

nn::Result PadSessionImpl::SetDirection(std::int32_t direction) NN_NOEXCEPT
{
    // SFDL で enum が現状定義できないため、ここで GPIO ライブラリに渡すためにキャスト
    nn::gpio::Direction gpioDirection = static_cast<nn::gpio::Direction>(direction);

    nn::gpio::driver::SetDirection(&m_InternalPadSession, gpioDirection);

    return nn::ResultSuccess();
}


nn::Result PadSessionImpl::SetValue(std::int32_t value) NN_NOEXCEPT
{
    // SFDL で enum が現状定義できないため、ここで GPIO ライブラリに渡すためにキャスト
    nn::gpio::GpioValue gpioValue = static_cast<nn::gpio::GpioValue>(value);

    nn::gpio::driver::SetValue(&m_InternalPadSession, gpioValue);

    return nn::ResultSuccess();
}

nn::Result PadSessionImpl::SetValueForSleepState(std::int32_t value) NN_NOEXCEPT
{
    // SFDL で enum が現状定義できないため、ここで GPIO ライブラリに渡すためにキャスト
    nn::gpio::GpioValue gpioValue = static_cast<nn::gpio::GpioValue>(value);

    nn::gpio::driver::SetValueForSleepState(&m_InternalPadSession, gpioValue);

    return nn::ResultSuccess();
}

nn::Result PadSessionImpl::GetValue(nn::sf::Out<std::int32_t> outValue) NN_NOEXCEPT
{
    // outValue に値を入れる
    *outValue = static_cast<int32_t>(nn::gpio::driver::GetValue(&m_InternalPadSession));

    return nn::ResultSuccess();
}

nn::Result PadSessionImpl::GetDirection(nn::sf::Out<std::int32_t> direction) NN_NOEXCEPT
{
    *direction = static_cast<int32_t>(nn::gpio::driver::GetDirection(&m_InternalPadSession));

    return nn::ResultSuccess();
}

nn::Result PadSessionImpl::SetInterruptMode(std::int32_t mode) NN_NOEXCEPT
{
    nn::gpio::driver::SetInterruptMode(&m_InternalPadSession, static_cast<nn::gpio::InterruptMode>(mode));

    return nn::ResultSuccess();
}

nn::Result PadSessionImpl::GetInterruptMode(nn::sf::Out<std::int32_t> mode) NN_NOEXCEPT
{
    *mode = static_cast<int32_t>(nn::gpio::driver::GetInterruptMode(&m_InternalPadSession));

    return nn::ResultSuccess();
}

nn::Result PadSessionImpl::SetInterruptEnable(bool enable) NN_NOEXCEPT
{
    nn::gpio::driver::SetInterruptEnable(&m_InternalPadSession, enable);

    return nn::ResultSuccess();
}

nn::Result PadSessionImpl::GetInterruptEnable(nn::sf::Out<bool> enable) NN_NOEXCEPT
{
    *enable = nn::gpio::driver::GetInterruptEnable(&m_InternalPadSession);
    return nn::ResultSuccess();
}

nn::Result PadSessionImpl::GetInterruptStatus(nn::sf::Out<std::int32_t> status) NN_NOEXCEPT
{
    *status = nn::gpio::driver::GetInterruptStatus(&m_InternalPadSession);

    return nn::ResultSuccess();
}

nn::Result PadSessionImpl::ClearInterruptStatus() NN_NOEXCEPT
{
    nn::gpio::driver::ClearInterruptStatus(&m_InternalPadSession);

    return nn::ResultSuccess();
}

nn::Result PadSessionImpl::BindInterrupt(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    // session のメンバ変数である m_SystemEvent を BindInterrupt で GPIO ライブラリ内部のピン定義に紐付け、
    // その SystemEvent を外に渡す
    nn::Result result = nn::gpio::driver::BindInterrupt(m_SystemEvent.GetBase(), &m_InternalPadSession);
    nn::os::NativeHandle internalHandle = nn::os::GetReadableHandleOfSystemEvent(m_SystemEvent.GetBase());

    // 内部の IsManaged を false にしているので、寿命管理はされない。
    *handle = nn::sf::NativeHandle( internalHandle, false );

    return result;
}

nn::Result PadSessionImpl::UnbindInterrupt() NN_NOEXCEPT
{
    // SF 側で Session のメンバ変数に m_SystemEvent があるので、これを Unbind すればよい
    nn::gpio::driver::UnbindInterrupt(&m_InternalPadSession);

    return nn::ResultSuccess();
}

nn::Result PadSessionImpl::SetDebounceEnabled(bool isEnable) NN_NOEXCEPT
{
    nn::gpio::driver::SetDebounceEnabled(&m_InternalPadSession, isEnable);

    return nn::ResultSuccess();
}

nn::Result PadSessionImpl::GetDebounceEnabled(nn::sf::Out<bool> isEnable) NN_NOEXCEPT
{
    *isEnable = nn::gpio::driver::GetDebounceEnabled(&m_InternalPadSession);

    return nn::ResultSuccess();
}

nn::Result PadSessionImpl::SetDebounceTime(int msecTime) NN_NOEXCEPT
{
    nn::gpio::driver::SetDebounceTime(&m_InternalPadSession, msecTime);

    return nn::ResultSuccess();
}

nn::Result PadSessionImpl::GetDebounceTime(nn::sf::Out<int> msecTime) NN_NOEXCEPT
{
    *msecTime = nn::gpio::driver::GetDebounceTime(&m_InternalPadSession);

    return nn::ResultSuccess();
}

// マネージャーの実装
ManagerImpl::ManagerImpl() NN_NOEXCEPT
{
    // コンストラクタ内で拡張ヒープを初期化
    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_NoOption);
    // アロケータにヒープハンドルをアタッチ
    m_Allocator.Attach(m_HeapHandle);
}

ManagerImpl::~ManagerImpl() NN_NOEXCEPT
{
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

// [Gen2]
nn::Result ManagerImpl::OpenSession2(nn::sf::Out<nn::sf::SharedPointer<nn::gpio::IPadSession>> outSession, nn::detail::DeviceCodeType deviceCode) NN_NOEXCEPT
{
    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    // GetImpl したいので、Factory の戻りを auto ではなく EmplacedRef で受けておく
    typedef nn::sf::ObjectFactory<MyAllocator::Policy> Factory;
    nn::sf::EmplacedRef<IPadSession, PadSessionImpl> p = Factory::CreateSharedEmplaced<IPadSession, PadSessionImpl>(&m_Allocator, this);
    NN_ABORT_UNLESS(p != nullptr, "[Gpio Server] Create session failed. Gpio server can not create session any more\n");

    // 失敗時は p をさらに move しないまま破棄して関数が返るため、自動的にリソースは解放される
    NN_RESULT_DO(p.GetImpl().OpenSession(deviceCode));

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    return nn::ResultSuccess();
}

// [Gen1] TODO: Deprecate
nn::Result ManagerImpl::OpenSessionForDev(nn::sf::Out<nn::sf::SharedPointer<nn::gpio::IPadSession>> outSession, std::int32_t pad) NN_NOEXCEPT
{
    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy>  Factory;
    auto p = Factory::CreateSharedEmplaced<IPadSession, PadSessionImpl>(&m_Allocator, this, static_cast<int>(pad), true);
    NN_ABORT_UNLESS(p != nullptr, "[Gpio Server] Create session failed. Gpio server can not create session any more\n");

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    return nn::ResultSuccess();
}

// [Gen1] TODO: Deprecate
nn::Result ManagerImpl::OpenSession(nn::sf::Out<nn::sf::SharedPointer<nn::gpio::IPadSession>> outSession, std::int32_t pad) NN_NOEXCEPT
{
    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy>  Factory;
    auto p = Factory::CreateSharedEmplaced<IPadSession, PadSessionImpl>(&m_Allocator, this, static_cast<int>(pad), false);
    NN_ABORT_UNLESS(p != nullptr, "[Gpio Server] Create session failed. Gpio server can not create session any more\n");

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    return nn::ResultSuccess();
}

// [Gen1] TODO: Deprecate
nn::Result ManagerImpl::OpenSessionForTest(nn::sf::Out<nn::sf::SharedPointer<nn::gpio::IPadSession>> outSession, std::int32_t pad) NN_NOEXCEPT
{
    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy>  Factory;
    auto p = Factory::CreateSharedEmplaced<IPadSession, PadSessionImpl>(&m_Allocator, this, static_cast<int>(pad), true);

    // この関数は nullptr のチェックをしない

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    return nn::ResultSuccess();
}

// [Gen2]
nn::Result ManagerImpl::IsWakeEventActive2(nn::sf::Out<bool> isActive, nn::detail::DeviceCodeType deviceCode) NN_NOEXCEPT
{
    bool tempIsActive;
    NN_RESULT_DO(nn::gpio::driver::IsWakeEventActive(&tempIsActive, deviceCode));
    *isActive = tempIsActive;
    return nn::ResultSuccess();
}

// [Gen1] TODO: Deprecate
nn::Result ManagerImpl::IsWakeEventActive(nn::sf::Out<bool> isActive, std::int32_t name) NN_NOEXCEPT
{
    *isActive = nn::gpio::driver::IsWakeEventActive(static_cast<nn::gpio::GpioPadName>(name));
    return nn::ResultSuccess();
}

// [Gen2] NOT SUPPORTED
// [Gen1] TODO: Deprecate
nn::Result ManagerImpl::GetWakeEventActiveFlagSet(nn::sf::Out<nn::gpio::WakeBitFlag> flag) NN_NOEXCEPT
{
    *flag = nn::gpio::driver::GetWakeEventActiveFlagSet();
    return nn::ResultSuccess();
}

// [Gen2]
nn::Result ManagerImpl::SetWakeEventActiveFlagSetForDebug2(nn::detail::DeviceCodeType deviceCode, bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(nn::gpio::driver::SetWakeEventActiveFlagSetForDebug(deviceCode, isEnabled));
    return nn::ResultSuccess();
}

// [Gen1] TODO: Deprecate
nn::Result ManagerImpl::SetWakeEventActiveFlagSetForDebug(std::int32_t name, bool isEnabled) NN_NOEXCEPT
{
    nn::gpio::driver::SetWakeEventActiveFlagSetForDebug(static_cast<nn::gpio::GpioPadName>(name), isEnabled);
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::SetWakePinDebugMode(std::int32_t mode) NN_NOEXCEPT
{
    nn::gpio::driver::SetWakePinDebugMode(static_cast<nn::gpio::driver::WakePinDebugMode>(mode));
    return nn::ResultSuccess();
}

}}}
