﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   GPIO ドライバの API インタフェース部分。
 * @details  割り込みスレッド用のスタックリソースの定義を含む。
 */

#include <mutex>
#include <nn/nn_DeviceCode.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/gpio/detail/gpio_Log.h>
#include <nn/gpio/driver/gpio_Lib.h>
#include <nn/gpio/driver/gpio_PadAccessor.h>
#include <nn/gpio/driver/gpio_PadAccessorDev.h>
#include <nn/gpio/driver/gpio_IGpioDriver.h>
#include <nn/gpio/driver/gpio_Pad.h>
#include <nn/gpio/driver/detail/gpio_PadSessionImpl.h>
#include <nn/gpio/gpio_Result.h>

#include "detail/gpio_Core.h"
#include "gpio_Gen1PadNameConvertTable.h"

// #define NN_DETAL_GPIO_TRACE_ALL_CLIENT_API_CALL
#ifdef NN_DETAL_GPIO_TRACE_ALL_CLIENT_API_CALL
#define NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN() NN_DETAIL_GPIO_TRACE("%s IN\n", __FUNCTION__)
#define NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT() NN_DETAIL_GPIO_TRACE("%s OUT\n", __FUNCTION__)
#else
#define NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN()
#define NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT()
#endif

namespace nn {
namespace gpio {
namespace driver {

void Initialize() NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();
    detail::InitializeDrivers();
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

void Finalize() NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();
    detail::FinalizeDrivers();
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

nn::Result OpenSessionBody(GpioPadSession* pOutSession, Pad* pPad) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    NN_SDK_ASSERT_NOT_NULL(pPad);

    bool needsRollback = true;

    // セッションオブジェクトの構築
    auto pSession = new (&detail::GetPadSessionImpl(*pOutSession)) detail::PadSessionImpl();
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            pSession->~PadSessionImpl();
        }
    };

    // TORIAEZU: 現在の API 体系ではアクセスモードは RW のみ
    NN_RESULT_DO(pSession->Open(pPad, nn::ddsf::AccessMode_ReadAndWrite));

    needsRollback = false; // 成功

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
    NN_RESULT_SUCCESS;
}

nn::Result OpenSession(GpioPadSession* pOutSession, nn::DeviceCode deviceCode) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    // デバイスコードエントリリストをルックアップ
    Pad* pPad = nullptr;
    NN_RESULT_DO(detail::FindPad(&pPad, deviceCode));
    NN_SDK_ASSERT_NOT_NULL(pPad);

    NN_RESULT_DO(OpenSessionBody(pOutSession, pPad));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
    NN_RESULT_SUCCESS;
}

// [Gen1] TODO: Deprecate
void OpenSession(GpioPadSession* pOutSession, GpioPadName name) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    // GpioPadName -> nn::DeviceCode 変換
    nn::DeviceCode deviceCode = nn::DeviceCode::GetInvalidCode();
    NN_ABORT_UNLESS(FindCompatDeviceCodeEntryForGen1(&deviceCode, name));

    NN_ABORT_UNLESS_RESULT_SUCCESS(OpenSession(pOutSession, deviceCode));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

// [Gen1] TODO: Deprecate
void OpenSessionForDev(GpioPadSession* pOutSession, int padNumber) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    // pad number をキーにデバイスコードエントリリストをルックアップ
    Pad* pPad = nullptr;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::FindPadByNumber(&pPad, padNumber));
    NN_SDK_ASSERT_NOT_NULL(pPad);

    NN_ABORT_UNLESS_RESULT_SUCCESS(OpenSessionBody(pOutSession, pPad));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

void CloseSession(GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    // 内部オブジェクトが構築済であることは呼び出し側の責任で保証
    // （原理的に実装側がチェックすることができない）
    auto& sessionImpl = detail::GetPadSessionImpl(*pSession);

    // デストラクタ内ですべての必要な破棄がされなければいけない
    sessionImpl.~PadSessionImpl();

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

void SetDirection(GpioPadSession* pSession, Direction direction) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    auto& pad = detail::GetPadSessionImplWithOpenCheck(*pSession).GetDevice().SafeCastTo<Pad>();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pad.GetDriver().SafeCastTo<IGpioDriver>().SetDirection(&pad, direction));
}

Direction GetDirection(GpioPadSession* pSession ) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    Direction direction;
    auto& pad = detail::GetPadSessionImplWithOpenCheck(*pSession).GetDevice().SafeCastTo<Pad>();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pad.GetDriver().SafeCastTo<IGpioDriver>().GetDirection(&direction, &pad));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
    return direction;
}

void SetInterruptMode(GpioPadSession* pSession, InterruptMode mode ) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    auto& pad = detail::GetPadSessionImplWithOpenCheck(*pSession).GetDevice().SafeCastTo<Pad>();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pad.GetDriver().SafeCastTo<IGpioDriver>().SetInterruptMode(&pad, mode));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

InterruptMode GetInterruptMode(GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    InterruptMode mode;
    auto& pad = detail::GetPadSessionImplWithOpenCheck(*pSession).GetDevice().SafeCastTo<Pad>();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pad.GetDriver().SafeCastTo<IGpioDriver>().GetInterruptMode(&mode, &pad));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
    return mode;
}

void SetInterruptEnable(GpioPadSession* pSession, bool enable ) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    auto& session = detail::GetPadSessionImplWithOpenCheck(*pSession);
    NN_ABORT_UNLESS_RESULT_SUCCESS(session.SetInterruptEnabled(enable));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

bool GetInterruptEnable(GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    bool isEnabled;
    auto& session = detail::GetPadSessionImplWithOpenCheck(*pSession);
    NN_ABORT_UNLESS_RESULT_SUCCESS(session.GetInterruptEnabled(&isEnabled));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
    return isEnabled;
}

InterruptStatus GetInterruptStatus(GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    InterruptStatus status;
    auto& pad = detail::GetPadSessionImplWithOpenCheck(*pSession).GetDevice().SafeCastTo<Pad>();
    auto& driver = pad.GetDriver().SafeCastTo<IGpioDriver>();
    auto& interruptControlMutex = driver.GetInterruptControlMutex(pad);
    std::lock_guard<decltype(interruptControlMutex)> lock(interruptControlMutex); // 割り込みステータス系 API は保護対象
    NN_ABORT_UNLESS_RESULT_SUCCESS(driver.GetInterruptStatus(&status, &pad));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
    return status;
}

void ClearInterruptStatus(GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    auto& pad = detail::GetPadSessionImplWithOpenCheck(*pSession).GetDevice().SafeCastTo<Pad>();
    auto& driver = pad.GetDriver().SafeCastTo<IGpioDriver>();
    auto& interruptControlMutex = driver.GetInterruptControlMutex(pad);
    std::lock_guard<decltype(interruptControlMutex)> lock(interruptControlMutex); // 割り込みステータス系 API は保護対象
    NN_ABORT_UNLESS_RESULT_SUCCESS(driver.ClearInterruptStatus(&pad));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

GpioValue GetValue(GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    GpioValue value;
    auto& pad = detail::GetPadSessionImplWithOpenCheck(*pSession).GetDevice().SafeCastTo<Pad>();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pad.GetDriver().SafeCastTo<IGpioDriver>().GetValue(&value, &pad));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
    return value;
}

void SetValue(GpioPadSession* pSession, GpioValue value ) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    auto& pad = detail::GetPadSessionImplWithOpenCheck(*pSession).GetDevice().SafeCastTo<Pad>();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pad.GetDriver().SafeCastTo<IGpioDriver>().SetValue(&pad, value));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

void SetValueForSleepState(GpioPadSession* pSession, GpioValue value ) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    auto& pad = detail::GetPadSessionImplWithOpenCheck(*pSession).GetDevice().SafeCastTo<Pad>();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pad.GetDriver().SafeCastTo<IGpioDriver>().SetValueForSleepState(&pad, value));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

void SetDebounceEnabled(GpioPadSession* pSession, bool enable) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    auto& pad = detail::GetPadSessionImplWithOpenCheck(*pSession).GetDevice().SafeCastTo<Pad>();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pad.GetDriver().SafeCastTo<IGpioDriver>().SetDebounceEnabled(&pad, enable));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

bool GetDebounceEnabled(GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    bool isEnabled;
    auto& pad = detail::GetPadSessionImplWithOpenCheck(*pSession).GetDevice().SafeCastTo<Pad>();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pad.GetDriver().SafeCastTo<IGpioDriver>().GetDebounceEnabled(&isEnabled, &pad));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
    return isEnabled;
}

void SetDebounceTime(GpioPadSession* pSession, int msecTime) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    auto& pad = detail::GetPadSessionImplWithOpenCheck(*pSession).GetDevice().SafeCastTo<Pad>();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pad.GetDriver().SafeCastTo<IGpioDriver>().SetDebounceTime(&pad, msecTime));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

int GetDebounceTime(GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    int msecTime;
    auto& pad = detail::GetPadSessionImplWithOpenCheck(*pSession).GetDevice().SafeCastTo<Pad>();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pad.GetDriver().SafeCastTo<IGpioDriver>().GetDebounceTime(&msecTime, &pad));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
    return msecTime;
}

nn::Result BindInterrupt(nn::os::SystemEventType* pEvent, GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    auto& session = detail::GetPadSessionImplWithOpenCheck(*pSession);
    NN_RESULT_DO(session.BindInterrupt(pEvent));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
    NN_RESULT_SUCCESS;
}

void UnbindInterrupt(GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    auto& session = detail::GetPadSessionImplWithOpenCheck(*pSession);
    session.UnbindInterrupt();

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

nn::Result IsWakeEventActive(bool* pOutIsActive, nn::DeviceCode deviceCode) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    // デバイスコードエントリリストをルックアップ
    Pad* pPad = nullptr;
    NN_RESULT_DO(detail::FindPad(&pPad, deviceCode));
    NN_SDK_ASSERT_NOT_NULL(pPad);

    bool isActive;
    NN_RESULT_DO(pPad->GetDriver().SafeCastTo<IGpioDriver>().IsWakeEventActive(&isActive, pPad));
    if ( pOutIsActive )
    {
        *pOutIsActive = isActive;
    }

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
    NN_RESULT_SUCCESS;
}

// [Gen1] TODO: Deprecate
bool IsWakeEventActive(GpioPadName name) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    // GpioPadName -> nn::DeviceCode 変換
    nn::DeviceCode deviceCode = nn::DeviceCode::GetInvalidCode();
    if ( !FindCompatDeviceCodeEntryForGen1(&deviceCode, name) )
    {
        return false;
    }

    bool isActive;
    NN_ABORT_UNLESS_RESULT_SUCCESS(IsWakeEventActive(&isActive, deviceCode));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
    return isActive;
}

nn::Result SetWakeEventActiveFlagSetForDebug(nn::DeviceCode deviceCode, bool enable) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    // デバイスコードエントリリストをルックアップ
    Pad* pPad = nullptr;
    NN_RESULT_DO(detail::FindPad(&pPad, deviceCode));
    NN_SDK_ASSERT_NOT_NULL(pPad);

    NN_RESULT_DO(pPad->GetDriver().SafeCastTo<IGpioDriver>().SetWakeEventActiveFlagSetForDebug(pPad, enable));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
    NN_RESULT_SUCCESS;
}

// [Gen1] TODO: Deprecate
void SetWakeEventActiveFlagSetForDebug(GpioPadName name, bool enable) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    // GpioPadName -> nn::DeviceCode 変換
    nn::DeviceCode deviceCode = nn::DeviceCode::GetInvalidCode();
    NN_ABORT_UNLESS(FindCompatDeviceCodeEntryForGen1(&deviceCode, name));

    NN_ABORT_UNLESS_RESULT_SUCCESS(SetWakeEventActiveFlagSetForDebug(deviceCode, enable));

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

void SetWakePinDebugMode(WakePinDebugMode mode) NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    for ( auto& driver : detail::GetDriverList() )
    {
        driver.SafeCastTo<IGpioDriver>().SetWakePinDebugMode(mode);
    }

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

void Suspend() NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    for ( auto& driver : detail::GetDriverList() )
    {
        driver.SafeCastTo<IGpioDriver>().Suspend();
    }

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

void SuspendLow() NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    for ( auto& driver : detail::GetDriverList() )
    {
        driver.SafeCastTo<IGpioDriver>().SuspendLow();
    }

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

void Resume() NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    for ( auto& driver : detail::GetDriverList() )
    {
        driver.SafeCastTo<IGpioDriver>().Resume();
    }

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

void ResumeLow() NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    for ( auto& driver : detail::GetDriverList() )
    {
        driver.SafeCastTo<IGpioDriver>().ResumeLow();
    }

    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_OUT();
}

// [Gen1] TODO: 削除
#if 0
void SetInitialGpioConfig() NN_NOEXCEPT
{
    NN_DETAIL_GPIO_FATAL("Driver generation mismatch: Gen2 driver no longer supports SetInitialGpioConfig.\n");
    NN_ABORT();
}

void SetInitialWakePinConfig() NN_NOEXCEPT
{
    NN_DETAIL_GPIO_FATAL("Driver generation mismatch: Gen2 driver no longer supports SetInitialWakePinConfig.\n");
    NN_ABORT();
}
#endif

// [Gen1] TODO: 削除
WakeBitFlag GetWakeEventActiveFlagSet() NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE_CLIENT_API_CALL_IN();

    NN_DETAIL_GPIO_FATAL("Driver generation mismatch: Gen2 driver no longer supports GetWakeEventActiveFlagSet.\n");
    NN_ABORT();
    return WakeBitFlag();
}

} // driver
} // gpio
} // nn
